package com.bcxin.tenant.bcx.rest.apis.controllers;

import com.alibaba.excel.EasyExcel;
import com.alibaba.excel.write.builder.ExcelWriterBuilder;
import com.alibaba.excel.write.handler.SheetWriteHandler;
import com.alibaba.excel.write.metadata.WriteSheet;
import com.alibaba.excel.write.metadata.holder.WriteSheetHolder;
import com.alibaba.excel.write.metadata.holder.WriteWorkbookHolder;
import com.bcxin.tenant.bcx.infrastructures.EntityCollection;
import com.bcxin.tenant.bcx.infrastructures.exceptions.BadTenantException;
import com.bcxin.tenant.bcx.jdks.PageDataRpcProvider;
import com.bcxin.tenant.bcx.jdks.PageMetaRpcProvider;
import com.bcxin.tenant.bcx.jdks.requests.pages.BatchPageDataRequest;
import com.bcxin.tenant.bcx.jdks.requests.pages.MappingPageDataSearchRequest;
import com.bcxin.tenant.bcx.jdks.responses.pages.PageDataResponse;
import com.bcxin.tenant.bcx.rest.apis.components.ExtractDataComponent;
import com.bcxin.tenant.bcx.rest.apis.controllers.requests.DynamicDataImportRequest;
import io.swagger.v3.oas.annotations.Operation;
import io.swagger.v3.oas.annotations.Parameter;
import io.swagger.v3.oas.annotations.enums.ParameterIn;
import io.swagger.v3.oas.annotations.responses.ApiResponse;
import io.swagger.v3.oas.annotations.tags.Tag;
import jakarta.servlet.http.HttpServletResponse;
import org.apache.poi.ss.usermodel.Sheet;
import org.springframework.http.ResponseEntity;
import org.springframework.util.CollectionUtils;
import org.springframework.util.StringUtils;
import org.springframework.web.bind.annotation.*;
import org.springframework.web.multipart.MultipartFile;

import java.io.IOException;
import java.net.URLEncoder;
import java.text.SimpleDateFormat;
import java.util.*;
import java.util.stream.Collectors;

@Tag(name = "DynamicMappingPageDataController", description = "动态映射表单数据")
@RestController
@RequestMapping("/dynamic-mapping-pages")
public class DynamicMappingPageDataController extends ControllerAbstract {
    private final PageDataRpcProvider rpcProvider;
    private final PageMetaRpcProvider pageMetaRpcProvider;
    private final ExtractDataComponent extractDataComponent;

    public DynamicMappingPageDataController(PageDataRpcProvider rpcProvider, PageMetaRpcProvider pageMetaRpcProvider, ExtractDataComponent extractDataComponent) {
        this.rpcProvider = rpcProvider;
        this.pageMetaRpcProvider = pageMetaRpcProvider;
        this.extractDataComponent = extractDataComponent;
    }

    @Operation(
            summary = "搜索通用表单数据",
            responses = {
                    @ApiResponse(responseCode = "200", description = "返回系统的 dispatchToken."),
                    @ApiResponse(responseCode = "401", description = "无效用户返回401."),
                    @ApiResponse(responseCode = "403", description = "禁止使用的企业, 该接口返回403."),
            }
    )
    @PostMapping("/search")
    public ResponseEntity doSearch(
            @RequestBody MappingPageDataSearchRequest request) {
        if(!StringUtils.hasLength(request.getMappingId())) {
            throw new BadTenantException("系统异常; 映射表单信息不为空");
        }

        EntityCollection<PageDataResponse> rps = this.rpcProvider.search(request);

        return this.ok(rps);
    }

    @Operation(summary = "导出-通用表单数据", description = "导出-通用表单数据 ",
            requestBody =
            @io.swagger.v3.oas.annotations.parameters.RequestBody(
                    required = true),
            responses = {
                    @ApiResponse(responseCode = "200", description = "返回签到信息列表及驻勤点信息列表")
            },
            parameters = {
                    @Parameter(in = ParameterIn.HEADER, required = true, name = "dispatchToken",
                            description = "来自认证接口产生的调度系统的/identity/auto-login产生的dispatchToken"),
                    @Parameter(in = ParameterIn.QUERY, required = true, name = "permission",
                            description = "固定传:advance 表示根据用户当前的权限而非依赖于调度台来实现权限管理; 不支持监管方")
            }
    )
    @PostMapping("/export")
    public void export(
            @RequestBody MappingPageDataSearchRequest request,
                       HttpServletResponse response) throws IOException {
        String rqName = "通用数据";
        if (StringUtils.hasLength(request.getExportFileName())) {
            rqName = request.getExportFileName();
        }

        String name = String.format("%s-%s", rqName, new SimpleDateFormat("yyyy-MM-dd").format(new Date()));
        request.setForExport(true);
        EntityCollection<PageDataResponse> entityCollection = this.rpcProvider.search(request);
        Collection<PageDataResponse> dataList = entityCollection.getData();

        /**
         * 保证顺序
         */
        Set<String> headerKeys = new HashSet<>();
        Map<String, String> headerSet = new HashMap<>();
        request.getColumnLabels().entrySet().forEach(ix -> {
            headerSet.put(ix.getKey(), ix.getValue());
            headerKeys.add(ix.getKey());
        });
        List<List<String>> head = new ArrayList<>();
        for (String key : headerKeys) {
            head.add(Collections.singletonList(headerSet.get(key)));
        }

        List<List<Object>> dataToWrite = new ArrayList<>();
        for (PageDataResponse data : dataList) {
            List<Object> rowData = new ArrayList<>();
            Map<String, Object> items = data.getItems();
            for (String key : headerKeys) {
                rowData.add(items.getOrDefault(key, ""));
            }

            dataToWrite.add(rowData);
        }

        response.setContentType("application/vnd.openxmlformats-officedocument.spreadsheetml.sheet");
        response.setCharacterEncoding("utf-8");
        // 这里URLEncoder.encode可以防止中文乱码 当然和easyexcel没有关系
        String fileName = URLEncoder.encode(name, "UTF-8").replaceAll("\\+", "%20");
        response.setHeader("Content-disposition", "attachment;filename*=utf-8''" + URLEncoder.encode(name, "UTF-8") + ".xlsx");


        WriteSheet writeSheet = EasyExcel.writerSheet(fileName).build();
        ExcelWriterBuilder builder = EasyExcel.write(response.getOutputStream());
        builder.head(head)
                .sheet(writeSheet.getSheetName())
                .registerWriteHandler(new SheetWriteHandler() {
                    @Override
                    public void afterSheetCreate(WriteWorkbookHolder writeWorkbookHolder,
                                                 WriteSheetHolder writeSheetHolder) {
                        Sheet sheet = writeSheetHolder.getSheet();
                        // 为每一列设置宽度
                        for (int i = 0; i < head.size(); i++) {
                            if (i >= head.size() - 2) {
                                sheet.setColumnWidth(i, 20 * 256);  // 第一列20个字符宽
                            } else {
                                sheet.setColumnWidth(i, 30 * 256);  // 其他列25个字符宽
                            }
                        }
                    }
                })
                .doWrite(dataToWrite);
    }
}
