package com.bcxin.flink.core.properties.impls;

import com.bcxin.flink.core.properties.CheckpointConfigProperty;
import com.bcxin.tenant.open.infrastructures.exceptions.BadTenantException;
import lombok.AccessLevel;
import lombok.Getter;
import lombok.Setter;
import org.apache.commons.io.FileUtils;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.File;
import java.io.IOException;
import java.util.Arrays;
import java.util.Collection;
import java.util.Optional;
import java.util.stream.Collectors;

@Getter(AccessLevel.PRIVATE)
@Setter(AccessLevel.PRIVATE)
public class DiskCheckpointConfigPropertyImpl implements CheckpointConfigProperty {
    private static final Logger logger = LoggerFactory.getLogger(DiskCheckpointConfigPropertyImpl.class);

    @Override
    public String getSavepointPath() {
        if (StringUtils.isEmpty(this.getLastCheckpointPath())) {
            return null;
        }

        return String.format("file:///%s", this.getLastCheckpointPath());
    }

    @Override
    public String getCheckpointPath() {
        if (StringUtils.isEmpty(this.getCheckPointLocation())) {
            throw new BadTenantException("Disk的checkpoint路径不能为空");
        }

        return String.format("file:///%s", getCheckPointLocation());
    }

    private String checkPointLocation;

    private String lastCheckpointPath;

    private DiskCheckpointConfigPropertyImpl(){}

    public static DiskCheckpointConfigPropertyImpl create(String checkPointLocation) {
        DiskCheckpointConfigPropertyImpl property = new DiskCheckpointConfigPropertyImpl();
        property.setCheckPointLocation(checkPointLocation);

        File checkPointDir = FileUtils.getFile(checkPointLocation);
        if (checkPointDir == null) {
            return property;
        }

        if (!checkPointDir.exists()) {
            try {
                FileUtils.forceMkdir(checkPointDir);
            } catch (IOException e) {
                logger.error(String.format("强制创建目录发生异常:%s", checkPointDir), e);
                e.printStackTrace();
            }
        }

        File[] subDirs = checkPointDir.listFiles(ii -> ii.isDirectory());
        if (subDirs != null) {
            for (File sd : subDirs) {
                logger.info("当前checkpoint的子目录为={};", sd.getPath());
                logger.info("当前checkpoint的子目录的子节点为={};", Arrays.stream(sd.listFiles()).map(ix -> ix.getPath()).collect(Collectors.joining(";")));
            }

            Collection<File> allCheckpointFiles = Arrays
                    .stream(subDirs)
                    .filter(ii -> Arrays.stream(ii.listFiles()).anyMatch(ix -> ix.getName().startsWith(DYNAMIC_CHK_PREFIX) &&
                            Arrays.stream(ix.listFiles()).anyMatch(ic -> ic.getName().equalsIgnoreCase(CHECK_POINT_META))))
                    .collect(Collectors.toList());

            Optional<File> lastCheckpointDirOptional = allCheckpointFiles.stream()
                    .sorted((i1, i2) -> i1.lastModified() > i2.lastModified() ? -1 : 1)
                    .findFirst();

            logger.info("上次checkpoint的目录位置={};", lastCheckpointDirOptional.isPresent() ? lastCheckpointDirOptional.get().getPath() : "无");
            if (lastCheckpointDirOptional.isPresent()) {
                File lastCheckpointDir = lastCheckpointDirOptional.get();

                Optional<File> checkDirOptional = Arrays.stream(
                                lastCheckpointDir.listFiles(ii -> ii.getName().startsWith(DYNAMIC_CHK_PREFIX) &&
                                        Arrays.stream(ii.listFiles(ix -> ix.getName().equalsIgnoreCase(CHECK_POINT_META))).count() > 0))
                        .findFirst();
                logger.info("上次checkpoint的文件位置={}", checkDirOptional.isPresent() ? checkDirOptional.get().getPath() : "无");
                if (!checkDirOptional.isPresent()) {
                    throw new IllegalArgumentException(String.format("非预期异常; 不应该找不到前一个SavePoint点的有效路径(%s)", lastCheckpointDir.getPath()));
                }

                property.setLastCheckpointPath(checkDirOptional.get().getPath());
            }
        } else {
            logger.info("当前checkpoint({})的子目录为=空;", checkPointDir.getPath());
        }

        return property;
    }
}
