package com.bcxin.message.service.messagecenter;

import cn.wildfirechat.common.ErrorCode;
import cn.wildfirechat.pojos.InputOutputUserInfo;
import cn.wildfirechat.pojos.OutputCreateUser;
import cn.wildfirechat.sdk.UserAdmin;
import cn.wildfirechat.sdk.model.IMResult;
import com.alibaba.fastjson.JSONObject;
import com.baomidou.mybatisplus.extension.conditions.query.LambdaQueryChainWrapper;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.bcxin.message.common.emus.MsgSendRequestStatus;
import com.bcxin.message.common.emus.MsgSendSource;
import com.bcxin.message.common.utils.IdGeneratorSnowflake;
import com.bcxin.message.dao.mapper.messagecenter.MsgSendLogMapper;
import com.bcxin.message.dtos.request.SendRobotMessageRequest;
import com.bcxin.message.entity.messagecenter.MsgSendLogEntity;
import com.bcxin.message.entity.tenant.TenantUsersEntity;
import com.bcxin.message.scheduled.message.SendRetryScheduler;
import com.bcxin.message.service.tenant.TenantUsersService;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.util.Date;

/**
 * description：消息中心发送日志表 service
 * author：linchunpeng
 * date：2023/11/17
 */
@Slf4j
@Service
public class MsgSendLogService extends ServiceImpl<MsgSendLogMapper, MsgSendLogEntity> {

    @Autowired
    private IdGeneratorSnowflake snowflake;

    @Autowired
    private SendRetryScheduler sendRetryScheduler;

    @Autowired
    private MsgSendRequestService msgSendRequestService;

    @Autowired
    private TenantUsersService tenantUsersService;

    //失败重试次数
    @Value("${message.retry.count}")
    private Integer retryCount;

    /**
     * description：记录发送日志
     * author：linchunpeng
     * date：2023/11/21
     */
    @Transactional
    public void record(SendRobotMessageRequest request, String sendStrategy, Integer sendSuccess, String sendResult) {
        MsgSendLogEntity msgSendLogEntity;
        int sendRetryCount = 0;
        MsgSendLogEntity exist = this.findByRequestId(request.getRequestId());
        if (exist != null) {
            msgSendLogEntity = exist;
            sendRetryCount = exist.getSendRetryCount() + 1;
        } else {
            msgSendLogEntity = new MsgSendLogEntity();
            msgSendLogEntity.setId(snowflake.snowflakeId());
            msgSendLogEntity.setRequestId(request.getRequestId());
            msgSendLogEntity.setAppid(request.getAppid());
            msgSendLogEntity.setSender(request.getSender());
            msgSendLogEntity.setSenderType(request.getSenderType());
            msgSendLogEntity.setSendStrategy(sendStrategy);
            msgSendLogEntity.setSendParam(JSONObject.toJSONString(request));
            msgSendLogEntity.setCreateTime(new Date());
        }
        msgSendLogEntity.setSendSuccess(sendSuccess);
        msgSendLogEntity.setSendResult(sendResult);
        msgSendLogEntity.setSendRetryCount(sendRetryCount);

        String resultMsg = sendSuccess == 1 ? "成功" : "失败";
        if (sendRetryCount == 0) {
            msgSendLogEntity.setRemark(String.format("发送消息%s", resultMsg));
        } else {
            msgSendLogEntity.setRemark(String.format("发送重试，第%s次发送消息%s", sendRetryCount, resultMsg));
        }
        msgSendLogEntity.setUpdateTime(new Date());
        if (exist != null) {
            this.lambdaUpdate()
                    .eq(MsgSendLogEntity::getId, exist.getId())
                    .set(MsgSendLogEntity::getSendSuccess, msgSendLogEntity.getSendSuccess())
                    .set(MsgSendLogEntity::getSendResult, msgSendLogEntity.getSendResult())
                    .set(MsgSendLogEntity::getSendRetryCount, msgSendLogEntity.getSendRetryCount())
                    .set(MsgSendLogEntity::getRemark, msgSendLogEntity.getRemark())
                    .set(MsgSendLogEntity::getUpdateTime, msgSendLogEntity.getUpdateTime())
                    .update();
        } else {
            this.save(msgSendLogEntity);
        }

        //如果是消息来源是发送请求表，需要同步发送结果
        if (request.getSource() != null && request.getSource() == MsgSendSource.SEND_REQUEST_TABLE.getCode()) {
            msgSendRequestService.updateSendStatus(request.getRequestId(),
                    sendSuccess == 1 ? MsgSendRequestStatus.SUCCESS : MsgSendRequestStatus.FAIL);
        }

        //如果是失败，并且是第一次发送，需要判断用户是否都有创建，没有需要创建用户
        if (sendSuccess == 0 && sendRetryCount == 0) {
            SendRobotMessageRequest sendRobotMessageRequest = JSONObject.parseObject(msgSendLogEntity.getSendParam(), SendRobotMessageRequest.class);
            //判断发送者
            existUserCreate(sendRobotMessageRequest.getSender());
            if ("user".equals(sendRobotMessageRequest.getTargetType())) {
                //判断接受者
                for (String userId : sendRobotMessageRequest.getTargetIds()) {
                    existUserCreate(userId);
                }
            }
        }
        //如果是失败，需要判断重试次数 < 规定次数，如果小于，则调度重试机制
        if (sendSuccess == 0 && sendRetryCount <= retryCount) {
            if (sendRetryCount == 0) {
                //第一次，新增调度
                sendRetryScheduler.scheduleSendRetry(msgSendLogEntity);
            } else {
                //不是第一次，重启调度
                sendRetryScheduler.reScheduleSendRetry(msgSendLogEntity);
            }
        }
    }

    /**
     * description：通过requestId查询记录
     * author：linchunpeng
     * date：2023/11/22
     */
    public MsgSendLogEntity findByRequestId(String requestId) {
        LambdaQueryChainWrapper<MsgSendLogEntity> lqw = this.lambdaQuery();
        lqw.eq(MsgSendLogEntity::getRequestId, requestId);
        return lqw.one();
    }


    /**
     * description：判断是否存在用户，不存在直接创建
     * author：linchunpeng
     * date：2025/8/26
     */
    private void existUserCreate(String userId) {
        try {
            IMResult<InputOutputUserInfo> userResult = UserAdmin.getUserByUserId(userId);
            InputOutputUserInfo userInfo = userResult.getResult();
            if (userInfo == null) {
                log.info("用户id：{}，不存在，创建一个", userId);
                TenantUsersEntity tenantUser = tenantUsersService.getById(userId);
                if (tenantUser != null) {
                    InputOutputUserInfo user = new InputOutputUserInfo();
                    user.setUserId(userId);
                    user.setName(userId);
                    user.setDisplayName(tenantUser.getName());
                    user.setMobile(tenantUser.getTelephone());
                    IMResult<OutputCreateUser> createUserResult = UserAdmin.createUser(user);
                    if (createUserResult.getErrorCode() == ErrorCode.ERROR_CODE_SUCCESS) {
                        log.info("用户：{}，创建成功", userId);
                    } else {
                        log.info("用户：{}，创建失败，msg：{}", userId, createUserResult.getErrorCode().getMsg());
                    }
                } else {
                    log.info("用户：{}，在智能人事不存在", userId);
                }
            }
        } catch (Exception e) {
            log.error("创建im用户异常，{}", e.getMessage(), e);
        }
    }
}
