package com.bcxin.event.job.core.domain.impls;

import com.bcxin.event.core.exceptions.BadEventException;
import com.bcxin.event.core.exceptions.NoSupportEventException;
import com.bcxin.event.job.core.domain.*;
import com.bcxin.event.job.core.domain.documents.enums.DocumentType;
import com.bcxin.event.job.core.domain.dtos.BinlogMapDTO;
import com.bcxin.event.job.core.domain.dtos.DocumentExpiredDTO;
import com.bcxin.event.job.core.domain.dtos.RedisConfig;
import com.github.benmanes.caffeine.cache.Cache;
import com.github.benmanes.caffeine.cache.Caffeine;
import freemarker.template.utility.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import redis.clients.jedis.Jedis;
import redis.clients.jedis.Pipeline;

import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.text.SimpleDateFormat;
import java.time.Duration;
import java.time.temporal.ChronoUnit;
import java.util.*;
import java.util.concurrent.ConcurrentHashMap;
import java.util.function.Function;
import java.util.function.Supplier;
import java.util.stream.Collectors;

public class CacheProviderImpl implements CacheProvider {
    private static Logger logger = LoggerFactory.getLogger(CacheProviderImpl.class);
    private static ThreadLocal<Pipeline> _currentThreadLocalPipeline = new InheritableThreadLocal<>();
    private static ThreadLocal<Jedis> _currentThreadLocalJedisWriter = new InheritableThreadLocal<>();
    private final RedisConfig redisConfig;

    public CacheProviderImpl(RedisConfig redisConfig) {
        this.redisConfig = redisConfig;
    }

    private Pipeline beginMulti(Jedis selectedJedis) {
        Pipeline pipeline = _currentThreadLocalPipeline.get();

        if (pipeline == null) {
            pipeline = selectedJedis.pipelined();
            _currentThreadLocalPipeline.set(pipeline);
        } else {
            throw new BadEventException("此时的Pipline的值应该不为空才是");
        }

        return pipeline;
    }

    private Pipeline getPipeline() {
        Pipeline pipeline = _currentThreadLocalPipeline.get();
        if (pipeline == null) {
            throw new NoSupportEventException();
        }

        return pipeline;
    }

    private Jedis getSelectedJedisWriter() {
        return _currentThreadLocalJedisWriter.get();
    }

    @Override
    public void upset(BinlogMapDTO data) {
        if (data == null || data.getDocumentType() == null) {
            return;
        }

        try (Jedis selectedJedis = JedisPoolFactory.getNewJedisResource(redisConfig)) {
            try (Jedis selectedJedisForReadonly = JedisPoolFactory.getNewJedisResource(redisConfig)) {
                _currentThreadLocalJedisWriter.set(selectedJedis);
                Collection<String> includeColumnNames = new ArrayList<>();
                includeColumnNames.add("id");
                includeColumnNames.add("name");
                includeColumnNames.add("status");
                try (Pipeline pipeline = this.beginMulti(selectedJedis)) {
                    switch (data.getDocumentType()) {
                        case Organization:
                            CacheDataProcessor.processOrganizationSuperviseDepartmentLevelInfo(
                                    this, data.getBefore(), data.getContent(),
                                    data.getDocumentType(), data.getId()
                            );
                            includeColumnNames.add("approved_information_status");
                            includeColumnNames.add("supervise_region_code");
                            includeColumnNames.add("supervise_depart_id");
                            includeColumnNames.add("supervise_depart_name");
                            includeColumnNames.add("place_of_business");
                            includeColumnNames.add("place_of_register");
                            includeColumnNames.add("type");
                            break;
                        case User_Credentials:
                            CacheDataProcessor.processUserCredentials(
                                    this,
                                    data.getBefore(), data.getContent(),
                                    data.getDocumentType(), data.getId()
                            );
                            includeColumnNames.add("number");
                            includeColumnNames.add("head_photo");
                            includeColumnNames.add("front_photo");
                            includeColumnNames.add("reverse_photo");
                            includeColumnNames.add("valid_date_from");
                            includeColumnNames.add("valid_date_to");
                            includeColumnNames.add("tenant_user_id");
                            includeColumnNames.add("credential_type");
                            includeColumnNames.add("selected");
                            break;
                        case Department:
                            break;
                        case Employee:
                            CacheDataProcessor.processEmployeeData(
                                    this,
                                    data.getBefore(),
                                    data.getContent(),
                                    data.getDocumentType(),
                                    data.getId());
                            includeColumnNames.add("hired_date");
                            includeColumnNames.add("leave_");
                            includeColumnNames.add("is_domain_admin");
                            includeColumnNames.add("hired_");
                            break;
                        case SecurityStation:
                            CacheDataProcessor.processSecurityStation(
                                    this, data.getBefore(), data.getContent(),
                                    data.getDocumentType(), data.getId(), data.getDomainId());
                            includeColumnNames.add("ITEM_attendanceSiteName");
                            includeColumnNames.add("ITEM_attendanceSiteType");
                            includeColumnNames.add("ITEM_attendanceSiteState");
                            includeColumnNames.add("ITEM_contractStartDate");
                            includeColumnNames.add("ITEM_contractEndDate");
                            break;
                        case SecurityStationPerson:
                            CacheDataProcessor.processSecurityStationPerson(
                                    this, data.getBefore(), data.getContent(), data.getDocumentType(), data.getId(), data.getDomainId()
                            );
                            includeColumnNames.add("ITEM_attendanceSiteId");
                            break;
                    }

                    if (data.getDocumentType() != null) {
                        String id = data.getId();
                        Map<String, String> value = data.getContent();
                        Map<String, String> before = data.getBefore();

                        /**
                         * 这边只能采用一个个key替换的方式; 覆盖的话; 会早从其他关联数据丢失
                         */
                        this.update(data.getDocumentType(), id, value);
                        if (this.getPipeline() != null) {
                            this.getPipeline().set(CacheTableConstant.getSyncKey(data.getDocumentType()),
                                    String.valueOf(data.getSyncVersion()));
                        } else {
                            selectedJedis.set(CacheTableConstant.getSyncKey(data.getDocumentType()),
                                    String.valueOf(data.getSyncVersion()));
                        }
                    }

                    List<Object> result = pipeline.syncAndReturnAll();
                } catch (Exception ex) {
                    ex.printStackTrace();
                    throw ex;
                }
            } finally {
                /**
                 * 一定要释放掉对象
                 */
                _currentThreadLocalPipeline.set(null);
                /**
                 * 一定要释放掉对象
                 */
                _currentThreadLocalJedisWriter.set(null);
            }
        }
    }

    @Override
    public void update(DocumentType documentType, String id, Map<String, String> data) {
        if (data == null) {
            return;
        }

        for (String key : data.keySet()) {
            String value = data.get(key);
            if (value == null) {
                if (this.getPipeline() != null) {
                    this.getPipeline().hdel(CacheTableConstant.getRedisHashKey(documentType, id), key);
                } else {
                    this.getSelectedJedisWriter().hdel(CacheTableConstant.getRedisHashKey(documentType, id), key);
                }
            } else {
                if (this.getPipeline() != null) {
                    this.getPipeline().hset(CacheTableConstant.getRedisHashKey(documentType, id), key, value);
                } else {
                    this.getSelectedJedisWriter().hset(CacheTableConstant.getRedisHashKey(documentType, id), key, value);
                }
            }
        }
    }

    @Override
    public void addRelative(DocumentType documentType, String id, Collection<String> relativeIds) {
        if (relativeIds == null || relativeIds.isEmpty()) {
            return;
        }

        String[] data = relativeIds.toArray(new String[relativeIds.size()]);

        if (this.getPipeline() != null) {
            this.getPipeline().sadd(CacheTableConstant.getRedisHashKey(documentType, id), data);
        } else {
            this.getSelectedJedisWriter().sadd(CacheTableConstant.getRedisHashKey(documentType, id), data);
        }
    }

    @Override
    public void removeRelative(DocumentType documentType, String id, Collection<String> relativeIds) {
        if (relativeIds == null || relativeIds.isEmpty()) {
            return;
        }

        String[] data = relativeIds.toArray(new String[relativeIds.size()]);

        if (this.getPipeline() != null) {
            this.getPipeline().srem(CacheTableConstant.getRedisHashKey(documentType, id), data);
        } else {
            this.getSelectedJedisWriter().srem(CacheTableConstant.getRedisHashKey(documentType, id), data);
        }
    }

    @Override
    public void close() {

    }

    private void executeMulti() {
        if (this.getPipeline() != null) {
            List<Object> rt = this.getPipeline().syncAndReturnAll();
            System.err.println(rt);
        }
    }


    @Override
    public Map<String, String> getDocument(DocumentType documentType, String id) {
        try (Jedis jedis = JedisPoolFactory.getNewJedisResource(redisConfig)) {
            return jedis.hgetAll(CacheTableConstant.getRedisHashKey(documentType, id));
        }
    }

    @Override
    public Map<String, String> getDocumentFromCache(DocumentType documentType, String id, DockMapDbExtractor dbExtractor) {
        return getDocumentFromCache(documentType, id, dbExtractor, 5 * 24 * 60);
    }

    @Override
    public void expireCache(DocumentType documentType, String id) {
        String documentKey = getCacheKey(documentType, id);
        try (Jedis jedis = JedisPoolFactory.getNewJedisResource(redisConfig)) {
            if (documentType == DocumentType.Station_Employee_Set) {
                Set<String> stationEmployeeIds = null;
                try (Jedis jedisReader = JedisPoolFactory.getNewJedisResource(redisConfig)) {
                    stationEmployeeIds = jedisReader.smembers(getCacheKey(DocumentType.Station_Employee_Set, id));
                }

                if (stationEmployeeIds != null && !stationEmployeeIds.isEmpty()) {
                    String[] stationEmployeeIdKeys =
                            stationEmployeeIds.stream().map(ix ->
                                            getCacheKey(DocumentType.Employee_Selected_Security_Station, ix))
                                    .toArray(size -> new String[size]);

                    try (Pipeline pipeline = jedis.pipelined()) {
                        pipeline.del(stationEmployeeIdKeys);

                        pipeline.syncAndReturnAll();
                    }
                }
            } else {
                jedis.del(documentKey);
            }
        } finally {
            logger.info("清除 redis:{}={}:{}", redisConfig.getHost(), redisConfig.getPort(), documentKey);
        }
    }

    @Override
    public Map<String, String> getDocumentFromCache(DocumentType documentType, String id, DockMapDbExtractor dbExtractor, int cacheInMinutes) {
        if (StringUtils.isEmpty(id)) {
            return Collections.EMPTY_MAP;
        }

        String documentKey = getCacheKey(documentType, id);

        Map<String, String> value = null;
        try {
            try (Jedis jedis = JedisPoolFactory.getNewJedisResource(redisConfig)) {
                value = jedis.hgetAll(documentKey);
                if (value == null || value.isEmpty()) {
                    synchronized (CacheProviderImpl.class) {
                        if (value == null || value.isEmpty()) {
                            value = dbExtractor.extract(documentType, id);//getDocMapFromDb(documentType, connection, id);\
                            if (value == null || value.isEmpty()) {
                                value.put("empty", "empty");
                            }

                            try (Pipeline pipeline = jedis.pipelined()) {
                                pipeline.hset(documentKey, value);
                                /**
                                 * 10分钟后过期
                                 */
                                if (cacheInMinutes < 0) {
                                    cacheInMinutes = 60;
                                }

                                if (documentType == DocumentType.Employee_Selected_Security_Station) {
                                    String stationId = value.get("ATTENDANCESITEID");
                                    if (stationId != null && !stationId.isEmpty()) {
                                        String stationEmployeeIdsKey = getCacheKey(DocumentType.Station_Employee_Set, stationId);
                                        Set<String> stationEmployeeIds = null;
                                        try (Jedis jedisReader = JedisPoolFactory.getNewJedisResource(redisConfig)) {
                                            stationEmployeeIds = jedisReader.smembers(stationEmployeeIdsKey);
                                            if (stationEmployeeIds == null || stationEmployeeIds.isEmpty()) {
                                                stationEmployeeIds = new HashSet<>();
                                            }
                                            stationEmployeeIds.add(id);
                                        }

                                        String[] stationEmployeeIdArray = stationEmployeeIds.stream().toArray(size -> new String[size]);
                                        pipeline.sadd(stationEmployeeIdsKey, stationEmployeeIdArray);
                                        pipeline.expire(stationEmployeeIdsKey, cacheInMinutes * 60);
                                    }
                                }

                                pipeline.expire(documentKey, cacheInMinutes * 60);
                                pipeline.sync();
                            }

                            logger.info("添加 redis:{}={}:{}", redisConfig.getHost(), redisConfig.getPort(), documentKey);
                        }
                    }
                }
            }
        } catch (Exception ex) {
            /**
             * 针对Redis无效的情况
             */
            value = dbExtractor.extract(documentType, id);//getDocMapFromDb(documentType, connection, id);
            logger.error("v2.redis信息获取异常(从db进行获取):sql={};id={};ex={}", documentType.getFetchSql(), id, ex);
        }

        return value;
    }

    @Override
    public void expireCache(Collection<DocumentExpiredDTO> documents) {
        if (documents == null || documents.isEmpty()) {
            return;
        }

        Collection<String> documentKeys =
                documents.stream()
                        .map(ii -> getCacheKey(ii.getDocumentType(), ii.getId()))
                        .collect(Collectors.toList());

        try (Jedis jedis = JedisPoolFactory.getNewJedisResource(redisConfig)) {
            jedis.del(documentKeys.toArray(new String[documentKeys.size()]));
        }
    }

    private static String getCacheKey(DocumentType documentType, String id) {
        String documentKey = String.format("dt:%s:%s", documentType, id);

        return documentKey;
    }

    @Override
    public long getSetSize(DocumentType documentType, String id) {
        try (Jedis jedis = JedisPoolFactory.getNewJedisResource(redisConfig)) {
            return jedis.scard(CacheTableConstant.getRedisSetKey(documentType, id));
        }
    }

    @Override
    public Set<String> getSet(DocumentType documentType, String id) {
        try (Jedis jedis = JedisPoolFactory.getNewJedisResource(redisConfig)) {
            return jedis.smembers(CacheTableConstant.getRedisSetKey(documentType, id));
        }
    }
}
