package com.bcxin.event.job.core.domain;

import com.bcxin.event.core.FlinkConstants;
import com.bcxin.event.core.exceptions.BadEventException;
import com.bcxin.event.core.utils.DebeziumUtil;
import com.bcxin.event.job.core.domain.documents.enums.DocumentType;
import freemarker.template.utility.CollectionUtils;
import org.apache.commons.lang3.StringUtils;

import java.text.SimpleDateFormat;
import java.util.*;

public class CacheDataProcessor {
    private static final String NUM_OF_SECURITY_MEN = "numofsecuritymen";
    private static final String NUM_OF_SECURITY_STATIONS = "numofsecuritystation";

    private static final String DEPARTMENT_NAME = "departname";

    private static final String ID_CARD_NUMBER = "id_card.number";
    private static final String ID_CARD_TYPE = "id_card.type";
    private static final String ID_CARD_FRONT_PHOTO = "id_card.front_photo";
    private static final String ID_CARD_HEAD_PHOTO = "id_card.head_photo";
    private static final String ID_CARD_REVERSE_PHOTO = "id_card.reverse_photo";
    private static final String ID_CARD_VALID_DATE_FROM = "id_card.valid_date_from";
    private static final String ID_CARD_VALID_DATE_TO = "id_card.valid_date_to";
    private static final String ID_CARD_ADDRESS = "id_card.address";


    /**
     * 针对保安人员的在离职信息发生变更的时候,
     * 1, 将职员信息加到Set集合中; 方便后续进行保安人员数量的统计
     * 2, 更新最新的驻勤信息到职员表
     * @param cacheProvider
     * @param before
     * @param content
     * @param documentType
     * @param idValue
     */
    public static void processEmployeeData(
            CacheProvider cacheProvider, Map<String, String> before, Map<String, String> content,
            DocumentType documentType, String idValue) {
        if (documentType != DocumentType.Employee) {
            return;
        }
        String beforeStatus = DebeziumUtil.getMapValue(before,"status");
        String afterStatus = DebeziumUtil.getMapValue(content,"status");
        String afterOrganizationId = DebeziumUtil.getMapValue(content,"organization_id");

        if ("1".equalsIgnoreCase(afterStatus) && ((beforeStatus != null && "0".equalsIgnoreCase(beforeStatus)) || StringUtils.isEmpty(beforeStatus))) {
            refreshEmployee_Station_OrganizationSetAndNumber(cacheProvider, afterOrganizationId, idValue, true);
        } else if ("0".equalsIgnoreCase(afterStatus) && (beforeStatus == null || "1".equalsIgnoreCase(beforeStatus))) {
            refreshEmployee_Station_OrganizationSetAndNumber(cacheProvider, afterOrganizationId, idValue, false);
        }

        String beforeDepartmentId = DebeziumUtil.getMapValue(before,"department_id");
        String departmentId = DebeziumUtil.getMapValue(content,"department_id");
        if (!StringUtils.equals(beforeDepartmentId, departmentId)) {
            Map<String, String> departmentMap = cacheProvider.getDocument(DocumentType.Department, departmentId);
            if (departmentMap != null) {
                cacheProvider.update(DocumentType.Employee, idValue, Collections.singletonMap(DEPARTMENT_NAME, DebeziumUtil.getMapValue(departmentMap,"name")));
            }
        }
    }

    /**
     * 针对驻勤点变更的时候,
     * 1, 比如: 生效变为无效的时候, 我们把驻勤点信息删除/加入到set集合中
     * 2, 并更新保安人员的最新驻勤点信息
     * 3, 更新公司的最新驻勤点数量
     *
     * @param domainIdValue
     * @param idValue
     */
    public static void processSecurityStation(
            CacheProvider cacheProvider, Map<String, String> before, Map<String, String> content,
            DocumentType documentType, String idValue, String domainIdValue) {
        if (documentType==DocumentType.SecurityStation) {

            boolean hasChanged = false;

            /**
             * 从原来的生效变为失效的情况
             */
            if (!checkIfValidSecurityStation(content) && checkIfValidSecurityStation(before)) {
                cacheProvider.removeRelative(DocumentType.Organization_Station_Set, domainIdValue, Collections.singleton(idValue));

                /**
                 * 如果当前的驻勤点状态发生变化; 那么需要试试这些人员的最新驻勤点
                 * 如果人员的驻勤点是当前的驻勤点的话; 那么将直接进行清除
                 */
                Set<String> stationEmployeeIds = cacheProvider.getSet(DocumentType.Station_Employee_Set, idValue);
                if (stationEmployeeIds != null) {
                    stationEmployeeIds.forEach(emId -> {
                        /**
                         *  根据当前职员的情况更新有效的驻勤点信息
                         */
                        refreshEmployeeStation(cacheProvider, emId);
                    });
                }

                hasChanged = true;
            } else if (checkIfValidSecurityStation(content)) {
                String beforeAttendanceSiteName = DebeziumUtil.getMapValue(before,"ITEM_attendanceSiteName" );
                String afterAttendanceSiteName = DebeziumUtil.getMapValue(content,"ITEM_attendanceSiteName" );

                if (!checkIfValidSecurityStation(before) || !StringUtils.equals(beforeAttendanceSiteName,afterAttendanceSiteName)) {
                    cacheProvider.addRelative(DocumentType.Organization_Station_Set, domainIdValue, Collections.singleton(idValue));
                    /**
                     * todo: 再次确认下
                     * 如果当前的驻勤点状态发生变化; 那么需要重新更改当前的驻勤点.
                     * 这时候应该找不到才是
                     */
                    Set<String> stationEmployeeIds = cacheProvider.getSet(DocumentType.Station_Employee_Set, idValue);
                    try {
                        if (stationEmployeeIds != null) {
                            stationEmployeeIds.forEach(eid -> {
                                cacheProvider.update(DocumentType.Employee, eid,
                                        getUpdatedFieldsForStationMap(content)
                                );
                            });
                        }
                    } catch (Exception ex) {
                        ex.printStackTrace();
                        throw new BadEventException(ex);
                    }
                }

                hasChanged = true;
            }


            /**
             * 刷新企业的最新驻勤点数量
             */
            if (hasChanged) {
                long totalStationInOrgan = cacheProvider.getSetSize(DocumentType.Organization_Station_Set, domainIdValue);
                cacheProvider.update(DocumentType.Organization, domainIdValue, Collections.singletonMap(NUM_OF_SECURITY_STATIONS, String.valueOf(totalStationInOrgan)));
            }
        }
    }


    /**
     * 当驻勤点保安人员发生变更的时候;
     * 1, 更新到驻勤点为key的在职保安员人员集合
     * 2, 当驻勤点人员信息发生变更的时候; 如果刚好所操作的驻勤点是有效的,
     * 那么我们需要刷新保安人员的最新驻勤信息.
     * @param cacheProvider
     * @param before
     * @param content
     * @param documentType
     * @param idValue
     * @param domainIdValue
     */
    public static void processSecurityStationPerson(CacheProvider cacheProvider, Map<String, String> before, Map<String, String> content,
                                             DocumentType documentType, String idValue, String domainIdValue) {
        if (documentType != DocumentType.SecurityStationPerson) {
            return;
        }

        String attendanceSiteId = DebeziumUtil.getMapValue(before, "item_attendancesiteid");

        String beforeSecurityId = DebeziumUtil.getMapValue(before, "item_securityid");
        String securityId = beforeSecurityId;

        String afterSecurityId = DebeziumUtil.getMapValue(content, "item_securityid");
        String afterId = DebeziumUtil.getMapValue(content, "id");
        String beforeId = DebeziumUtil.getMapValue(before, "id");

        if (StringUtils.isEmpty(attendanceSiteId)) {
            attendanceSiteId = DebeziumUtil.getMapValue(content, "item_attendancesiteid");
            securityId = afterSecurityId;
        }

        boolean changedFlag = false;

        /**
         * 删除驻勤点的该驻勤人员和删除该人员拥有的驻勤点列表
         */
        if (!StringUtils.isEmpty(beforeId) && StringUtils.isEmpty(afterId)) {
            cacheProvider.removeRelative(DocumentType.Station_Employee_Set,
                    attendanceSiteId, Collections.singleton(securityId)
            );

            cacheProvider.removeRelative(DocumentType.Employee_Station_Set, securityId, Collections.singleton(attendanceSiteId));
            /**
             * 获取当前用户的基本信息; 如果用户当前的驻勤点为本次操作的对象的话;
             * 那么, 将驻勤点设置为其他
             */
            Map<String, String> employeeMap = cacheProvider.getDocument(DocumentType.Employee, securityId);
            /**
             * 如果当前驻勤点有效且
             */
            if (employeeMap != null &&
                    attendanceSiteId.equalsIgnoreCase(DebeziumUtil.getMapValue(employeeMap, FlinkConstants.EMPLOYEE_STATION_ID))) {

                /**
                 *  根据当前职员的情况更新有效的驻勤点信息
                 */
                refreshEmployeeStation(cacheProvider,securityId);
            }

            changedFlag = true;
        } else if (StringUtils.isEmpty(beforeId) && !StringUtils.isEmpty(afterId)) {
            /**
             * 对于新增驻勤人员信息的话; 如果当前的驻勤点有效, 那么我们将本次驻勤点设置为当前用户的驻勤点信息
             */
            Map<String, String> station = cacheProvider.getDocument(DocumentType.SecurityStation, attendanceSiteId);
            if (station != null && checkIfValidSecurityStation(station)) {
                cacheProvider.update(DocumentType.Employee, securityId,
                        getUpdatedFieldsForStationMap(station)
                );

                /**
                 * 有效驻勤点的时候才算入驻勤人数中
                 */
                cacheProvider.addRelative(DocumentType.Station_Employee_Set,
                        attendanceSiteId, Collections.singleton(securityId)
                );
            }

            changedFlag = true;
        }


        if (changedFlag) {
            /**
             * 更改该驻勤点的总人数
             */
            long value = cacheProvider.getSetSize(DocumentType.Station_Employee_Set, attendanceSiteId);
            cacheProvider.update(
                    DocumentType.SecurityStation, attendanceSiteId,
                    Collections.singletonMap(NUM_OF_SECURITY_MEN,
                            String.valueOf(value)));
        }
    }

    /**
     * 新增或者修改企业监管归属的时候; 设置归属附加信息
     * @param cacheProvider
     * @param before
     * @param content
     * @param documentType
     * @param idValue
     */
    public static void processOrganizationSuperviseDepartmentLevelInfo(
            CacheProvider cacheProvider,
            Map<String, String> before, Map<String, String> content,
            DocumentType documentType, String idValue
    ) {
        /**
         * 不用考虑删除的情况; 因为我们不会执行
         */
        if (documentType == DocumentType.Organization) {
            String supervise_depart_id = content.get("supervise_depart_id");
            String beforeSupervise_depart_id = null;
            if(before!=null) {
                beforeSupervise_depart_id = before.get("supervise_depart_id");
            }

            if (!StringUtils.equals(supervise_depart_id, beforeSupervise_depart_id)) {
                if (StringUtils.isEmpty(supervise_depart_id) || "#".equalsIgnoreCase(supervise_depart_id)) {
                    //todo 无效数据
                    Map<String, String> superViseDepartMap = new HashMap<>();
                    superViseDepartMap.put(FlinkConstants.SUPERVISE_DEPART_ID, "#");

                    superViseDepartMap.put(FlinkConstants.SUPERVISE_DEPART_NAME, "#");
                    superViseDepartMap.put(FlinkConstants.SUPERVISE_REGIONId, "#");
                    cacheProvider.update(DocumentType.Organization, idValue, superViseDepartMap);

                    cacheProvider.removeRelative(DocumentType.SuperviseDepartmentLevel_ORGANIZATION_SET,
                            supervise_depart_id, Collections.singleton(idValue));
                } else {
                    Map<String, String> departLevelMap = cacheProvider.getDocument(DocumentType.SuperviseDepartmentLevel, supervise_depart_id);
                    if (departLevelMap != null) {
                        Map<String, String> superViseDepartMap = new HashMap<>();
                        superViseDepartMap.put(FlinkConstants.SUPERVISE_DEPART_ID, supervise_depart_id);

                        superViseDepartMap.put(FlinkConstants.SUPERVISE_DEPART_NAME,
                                DebeziumUtil.getMapValue(departLevelMap, "after.item_name"));
                        superViseDepartMap.put(FlinkConstants.SUPERVISE_REGIONId,
                                DebeziumUtil.getMapValue(departLevelMap, "after.item_regionid"));

                        cacheProvider.update(DocumentType.Organization, idValue, superViseDepartMap);
                    }

                    /**
                     * 将数据加入到监管归属的集合; 方便后续当监管归属信息归集的时候; 进行更改操作
                     */
                    cacheProvider.addRelative(DocumentType.SuperviseDepartmentLevel_ORGANIZATION_SET, supervise_depart_id, Collections.singleton(idValue));
                }
            }
        }
    }


    /**
     * 根据当前用户的基本情况; 刷新最新驻勤点信息
     * @param cacheProvider
     * @param securityId
     */
    private static void refreshEmployeeStation(CacheProvider cacheProvider,String securityId)
    {
        /**
         * 获取当前用户有效的所有驻勤点id
         *
         */
        Set<String> employeeStationIds = cacheProvider.getSet(DocumentType.Employee_Station_Set, securityId);
        if (employeeStationIds != null && !employeeStationIds.isEmpty()) {
            String finalSecurityId = securityId;
            employeeStationIds.forEach(stationId -> {
                /**
                 * 设置刷新当前用户的驻勤点信息
                 */
                Map<String, String> checkingStationMap = cacheProvider.getDocument(DocumentType.SecurityStation, stationId);
                if (checkingStationMap != null && checkIfValidSecurityStation(checkingStationMap)) {
                    cacheProvider.update(DocumentType.Employee, finalSecurityId,
                            getUpdatedFieldsForStationMap(checkingStationMap)
                    );
                    return;
                }
            });
        }else {
            /**
             * 如果当前用户找不到有效的驻勤点; 那么系统刷新驻勤点为空
             */
            Map<String, String> emptyEmployeeStationMap = new HashMap<>();
            emptyEmployeeStationMap.put(FlinkConstants.EMPLOYEE_STATION_ID, "#");
            emptyEmployeeStationMap.put(FlinkConstants.EMPLOYEE_STATION_NAME, "#");
            emptyEmployeeStationMap.put(FlinkConstants.EMPLOYEE_STATION_SUPERVISE_DEPART_ID, "#");
            emptyEmployeeStationMap.put(FlinkConstants.EMPLOYEE_STATION_SUPERVISE_DEPART_NAME, "#");
        }
    }

    public static void processUserCredentials(CacheProvider cacheProvider, Map<String, String> before, Map<String, String> content,
                                                          DocumentType documentType, String idValue) {
        if (documentType != DocumentType.User_Credentials) {
            return;
        }

        String tenant_user_id = DebeziumUtil.getMapValue(content, "tenant_user_id");
        String number = DebeziumUtil.getMapValue(content, "number");
        String type = DebeziumUtil.getMapValue(content, "credential_type");
        String front_photo = DebeziumUtil.getMapValue(content, "front_photo");
        String head_photo = DebeziumUtil.getMapValue(content, "head_photo");
        String reverse_photo = DebeziumUtil.getMapValue(content, "reverse_photo");
        String valid_date_from = DebeziumUtil.getMapValue(content, "valid_date_from");
        String valid_date_to = DebeziumUtil.getMapValue(content, "valid_date_to");
        String address = DebeziumUtil.getMapValue(content, "address");


        String selected = DebeziumUtil.getMapValue(content, "selected");
        Map<String, String> data = new HashMap<>();
        if ("1".equalsIgnoreCase(selected)) {
            SimpleDateFormat dateFormat = new SimpleDateFormat("yyyy-MM-dd");
            data.put(ID_CARD_NUMBER, number);
            data.put(ID_CARD_TYPE, type);
            data.put(ID_CARD_FRONT_PHOTO, front_photo);
            data.put(ID_CARD_ADDRESS, address);
            data.put(ID_CARD_REVERSE_PHOTO, reverse_photo);
            data.put(ID_CARD_VALID_DATE_FROM, valid_date_from);
            data.put(ID_CARD_VALID_DATE_TO, valid_date_to);
            data.put(ID_CARD_HEAD_PHOTO, head_photo);

            cacheProvider.update(DocumentType.User, tenant_user_id, data);
        }
    }

    /**
     * 根据保安人员的在离职人员信息更新到驻勤点/保安公司
     */
    private static void refreshEmployee_Station_OrganizationSetAndNumber(CacheProvider cacheProvider,
                                                                         String domainId, String employeeId,
                                                                         boolean isLeave) {
        /**
         * 获取当前人员所在的全部驻勤点Id
         */
        Set<String> data = cacheProvider.getSet(DocumentType.Employee_Station_Set, employeeId);
        if (data != null) {
            data.forEach(stationId -> {
                /**
                 * 根据是否离职; 将人员加入驻勤点
                 */
                if (isLeave) {
                    cacheProvider.removeRelative(DocumentType.Station_Employee_Set, stationId, Collections.singleton(employeeId));
                } else {
                    cacheProvider.addRelative(DocumentType.Station_Employee_Set, stationId, Collections.singleton(employeeId));
                }

                /**
                 * 获取该驻勤点的人数信息
                 */
                long totalEmployeeInStation = cacheProvider.getSetSize(DocumentType.Station_Employee_Set, stationId);

                /**
                 * 将驻勤人数更新到驻勤点中
                 */
                cacheProvider.update(
                        DocumentType.SecurityStation, stationId,
                        Collections.singletonMap(NUM_OF_SECURITY_MEN, String.valueOf(totalEmployeeInStation))
                );
            });
        }


        /**
         * 将在职保安人员加入到公司的集合中
         */
        if (isLeave) {
            cacheProvider.removeRelative(DocumentType.Organization_Employee_Set, domainId, Collections.singleton(employeeId));
        } else {
            cacheProvider.addRelative(DocumentType.Organization_Employee_Set, domainId, Collections.singleton(employeeId));
        }

        /**
         * 获取该公司的保安人员数量
         */
        long totalEmployeeInOrganization = cacheProvider.getSetSize(DocumentType.Organization_Employee_Set, domainId);

        /**
         * 将在职保安人员添加到公司的属性中
         */
        cacheProvider.update(
                DocumentType.Organization, domainId,
                Collections.singletonMap(NUM_OF_SECURITY_MEN, String.valueOf(totalEmployeeInOrganization))
        );
    }

    private static boolean checkIfValidSecurityStation(Map<String, String> flagData) {
        if (flagData == null) {
            return false;
        }


        return "生效".equalsIgnoreCase(flagData.get("item_attendancesitestate"));
    }

    private static Map<String, String> getUpdatedFieldsForStationMap(Map<String, String> station) {
        if (station == null || station.isEmpty()) {
            return null;
        }

        Map<String, String> map = new HashMap<>();
        map.put(FlinkConstants.EMPLOYEE_STATION_ID, DebeziumUtil.getMapValue(station, "id"));
        map.put(FlinkConstants.EMPLOYEE_STATION_NAME, DebeziumUtil.getMapValue(station, "item_attendancesitename"));
        map.put(FlinkConstants.EMPLOYEE_STATION_SUPERVISE_DEPART_ID, DebeziumUtil.getMapValue(station, "item_officepoliceaddressid"));
        map.put(FlinkConstants.EMPLOYEE_STATION_SUPERVISE_DEPART_NAME, DebeziumUtil.getMapValue(station, "item_officepoliceaddress"));

        return map;
    }
}
