package com.bcxin.tenant.domain.entities;

import java.sql.Timestamp;
import java.time.Instant;

import javax.persistence.AttributeOverride;
import javax.persistence.AttributeOverrides;
import javax.persistence.CascadeType;
import javax.persistence.Column;
import javax.persistence.Embedded;
import javax.persistence.Entity;
import javax.persistence.FetchType;
import javax.persistence.Id;
import javax.persistence.Inheritance;
import javax.persistence.InheritanceType;
import javax.persistence.JoinColumn;
import javax.persistence.OneToOne;
import javax.persistence.Table;
import javax.persistence.UniqueConstraint;

import com.bcxin.Infrastructures.entities.EntityAbstract;
import com.bcxin.Infrastructures.entities.IAggregate;
import com.bcxin.Infrastructures.enums.ApprovedStatus;
import com.bcxin.Infrastructures.exceptions.BadTenantException;
import com.bcxin.Infrastructures.utils.UUIDUtil;
import com.bcxin.tenant.domain.DomainConstraint;
import com.bcxin.tenant.domain.entities.valueTypes.ApprovedInformationValueType;
import com.bcxin.tenant.domain.entities.valueTypes.LocationValueType;
import com.bcxin.tenant.domain.events.OrganizationCreatedEvent;

import lombok.AccessLevel;
import lombok.Getter;
import lombok.Setter;

@Getter
@Setter(AccessLevel.PROTECTED)
//@Table(name = "tenant_organizations",
//        uniqueConstraints = @UniqueConstraint(name = DomainConstraint.UNIQUE_ORGANIZATION_NAME,columnNames = "name"))
//@Entity
@Inheritance(strategy = InheritanceType.JOINED)
public class OrganizationEntity extends EntityAbstract implements IAggregate {
    @Id
    @Column(length = 50)
    private String id;

    @Column(nullable = false,length = 30)
    private String code;

    /**
     * 组织名称
     */
    @Column(name = "name",nullable = false)
    private String name;

    /**
     * 注册地
     */
    @Embedded
    @AttributeOverrides({
            @AttributeOverride(name="province.code",
                    column=@Column(name="place_Of_register_province_code")),
            @AttributeOverride(name="province.name",
                    column=@Column(name="place_Of_register_province_name")),
            @AttributeOverride(name="city.code",
                    column=@Column(name="place_Of_register_city_code")),
            @AttributeOverride(name="city.name",
                    column=@Column(name="place_Of_register_city_name")),
            @AttributeOverride(name="district.code",
                    column=@Column(name="place_Of_register_district_code")),
            @AttributeOverride(name="district.name",
                    column=@Column(name="place_Of_register_district_name")),
            @AttributeOverride(name="address",
                    column=@Column(name="place_Of_register_address")),
    })
    private LocationValueType placeOfRegister;

    /**
     * 组织图标
     */
    @Column(name = "logo_path", length = 500)
    private String logoPath;

    /**
     * 经营地
     */
    @Embedded
    @AttributeOverrides({
            @AttributeOverride(name="province.code",
                    column=@Column(name="place_Of_business_province_code")),
            @AttributeOverride(name="province.name",
                    column=@Column(name="place_Of_business_province_name")),
            @AttributeOverride(name="city.code",
                    column=@Column(name="place_Of_business_city_code")),
            @AttributeOverride(name="city.name",
                    column=@Column(name="place_Of_business_city_name")),
            @AttributeOverride(name="district.code",
                    column=@Column(name="place_Of_business_district_code")),
            @AttributeOverride(name="district.name",
                    column=@Column(name="place_Of_business_district_name")),
            @AttributeOverride(name="address",
                    column=@Column(name="place_Of_business_address")),
    })
    private LocationValueType placeOfBusiness;

    /**
     * 行业类型
     */
    @Column(name = "industry_code")
    private String industryCode;

    /**
     * 机构类型
     */
    @Column(name = "institutional_code")
    private String institutionalCode;

    /**
     * 审批状态
     */
    @Embedded
    @Column(name = "approved_information")
    @AttributeOverrides({
            @AttributeOverride(name="status",
                    column=@Column(name="approved_information_status")),
            @AttributeOverride(name="lastUpdatedTime",
                    column=@Column(name="approved_information_last_updated_time")),
            @AttributeOverride(name="note",
                    column=@Column(name="approved_information_note")),
    })
    private ApprovedInformationValueType approvedInformationValueType;

    @Column(name = "created_time", nullable = false)
    private Timestamp createdTime;

    @Column(name = "last_updated_time", nullable = true)
    private Timestamp lastUpdatedTime;



    @OneToOne(cascade = CascadeType.ALL,fetch = FetchType.LAZY)
    @JoinColumn(name = "id", referencedColumnName = "id",nullable = false)
    private CompanyEntity company;


    public void change(LocationValueType placeOfRegister, LocationValueType placeOfBusiness) {
        this.setPlaceOfRegister(placeOfRegister);
        this.setPlaceOfBusiness(placeOfBusiness);

        this.setLastUpdatedTime(Timestamp.from(Instant.now()));
    }

    public void changeLogoPath(String logoPath) {
        this.setLogoPath(logoPath);
        this.setLastUpdatedTime(Timestamp.from(Instant.now()));
    }

    /**
     *需改审批状态
     * @param status
     * @param note
     */
    public void approve(ApprovedStatus status, String note) {
        if (this.getApprovedInformationValueType()!=null &&
                this.getApprovedInformationValueType().getStatus() == ApprovedStatus.Passed) {
            throw new BadTenantException("企业已经审批通过，无需重复审批");
        }

        ApprovedInformationValueType approvedInformation = ApprovedInformationValueType.create(status, note);
        this.setApprovedInformationValueType(approvedInformation);

        /*
        this.recordEvent(OrganizationApprovedChangedEvent.create(
                this.getId(),
                approvedInformation.getStatus(),
                this.getPlaceOfRegister(),
                this.getPlaceOfBusiness(),
                institutionalCode));

         */

        this.setLastUpdatedTime(Timestamp.from(Instant.now()));
    }

    public void upgrade(String unifySocialCreditCode, String unifySocialCreditCodeFile)
    {
        CompanyEntity company = this.getCompany();
        if(company==null) {
            company = CompanyEntity.create(this, unifySocialCreditCode, unifySocialCreditCodeFile);
        }

        company.change(unifySocialCreditCode,unifySocialCreditCodeFile);

        this.setCompany(company);
    }

    protected OrganizationEntity() {
        this.setCreatedTime(Timestamp.from(Instant.now()));
    }

    protected OrganizationEntity(String industryCode, String institutionalCode, String name) {
        this();
        this.setId(UUIDUtil.getShortUuid());
        this.setCode(UUIDUtil.getShortUuid());

        this.setInstitutionalCode(institutionalCode);
        this.setIndustryCode(industryCode);
        this.setName(name);

        this.recordEvent(OrganizationCreatedEvent.create(this, this.getName()));

    }

    public static OrganizationEntity create(String industryCode, String institutionalCode, String name) {
        OrganizationEntity organization = new OrganizationEntity(industryCode, institutionalCode, name);



        return organization;
    }
}
