package com.bcxin.platform.web.controller.wallet;

import com.alibaba.fastjson.JSON;
import com.bcxin.platform.common.exception.V5BusinessException;
import com.bcxin.platform.common.utils.Result;
import com.bcxin.platform.domain.wallet.ComWallet;
import com.bcxin.platform.domain.wallet.ComWalletAccount;
import com.bcxin.platform.domain.wallet.ComWalletProcess;
import com.bcxin.platform.domain.wallet.ComWalletTrade;
import com.bcxin.platform.dto.CurrentUser;
import com.bcxin.platform.dto.wallet.*;
import com.bcxin.platform.service.oauth.RedisLockUtil;
import com.bcxin.platform.service.wallet.ComWalletChangeService;
import com.bcxin.platform.service.wallet.ComWalletService;
import com.bcxin.platform.util.file.BcxinFileUtils;
import com.bcxin.platform.util.log.Log;
import com.bcxin.platform.web.controller.base.BaseController;
import com.github.pagehelper.util.StringUtil;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import org.redisson.api.RLock;
import org.redisson.api.RedissonClient;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RestController;

import javax.annotation.Resource;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import java.io.IOException;
import java.util.Map;

/**
 * 企业钱包服务的controller层
 * by llc 2019-06-20
 */
@RestController
@RequestMapping("/wallet")
@Api(tags = {""}, description = "企业钱包服务")
public class ComWalletController extends BaseController {

	@Autowired
	private ComWalletService comWalletService;

	@Autowired
	private ComWalletChangeService comWalletChangeService;

	@Autowired
	private RedisLockUtil redisLockUtil;

//	@Resource
//	private RedissonClient redissonClient;

	private final static String TRANSFERKEY = "Transfer:";

	/**
	 * 获取企业钱包服务开通情况
	 * 【百课堂在调用，勿加权限校验】
	 *
	 * @param comWallet
	 * @author llc
	 * @date 2019-06-21
	 */
	@Log(operation = Log.OPERA_TYPE_SELECT, modle = "企业钱包服务", title = "获取企业钱包服务开通情况", systemType = Log.SYSTEMTYPE_PHONE)
	@ApiOperation(value = "获取企业钱包服务开通情况", httpMethod = "GET", notes = "获取企业钱包服务开通情况", response = Map.class)
	@GetMapping(value = "get-com-wallet-service-open-info")
	public Result getComWalletServiceOpenInfo(ComWallet comWallet) {
		/* 【百课堂在调用，勿加权限校验】 */
		return comWalletService.getComWalletServiceOpenInfo(comWallet).setBack(getRequest());
	}

	/**
	 * 获取企业钱包服务当前开通步骤(因富民银行开户流程调整作废)
	 *
	 * @param comWalletProcess
	 * @author llc
	 * @date 2019-06-21
	 */
	@Log(operation = Log.OPERA_TYPE_SELECT, modle = "企业钱包服务", title = "获取企业钱包服务当前开通步骤", systemType = Log.SYSTEMTYPE_PHONE)
	@ApiOperation(value = "获取企业钱包服务当前开通步骤", httpMethod = "GET", notes = "获取企业钱包服务当前开通步骤", response = Map.class)
	@GetMapping(value = "get-com-wallet-current-process")
	public Result getComWalletCurrentProcess(ComWalletProcess comWalletProcess) {
		return comWalletService.getComWalletCurrentProcess(comWalletProcess).setBack(getRequest());
	}


	/**
	 * 企业业务发送验证码
	 * 【百课堂在调用，勿加权限校验】
	 *
	 * @param dto
	 * @author llc
	 * @date 2019-06-22
	 */
	@Log(operation = Log.OPERA_TYPE_SELECT, modle = "企业钱包服务", title = "发送验证码", systemType = Log.SYSTEMTYPE_PHONE)
	@ApiOperation(value = "发送验证码", httpMethod = "POST", notes = "发送验证码", response = ComWalletAccount.class)
	@PostMapping(value = "send_verify_code")
	public Result sendVerifyCode(ComWalletAccountDTO dto) {
		/* 【百课堂在调用，勿加权限校验】 */
		return comWalletService.sendVerifyCode(dto).setBack(getRequest());
	}


	/**
	 * 保存开户资料
	 *
	 * @param dto
	 * @author llc
	 * @date 2019-06-21
	 * @update by llc 2020-09-10 保存开户资料第一步、第二步整合成保存开户资料
	 */
	@Log(operation = Log.OPERA_TYPE_SAVE, modle = "企业钱包服务", title = "保存开户资料", systemType = Log.SYSTEMTYPE_PHONE)
	@ApiOperation(value = "保存开户资料", httpMethod = "POST", notes = "保存开户资料", response = Result.class)
	@PostMapping(value = "save-account-info")
	public Result saveAccountInfo(ComWalletAccountDTO dto) {
		CurrentUser currentUser = new CurrentUser(getRequest());
		if (currentUser.getPerId() == null) {
			return Result.tokenExpired("用户登录超时或者未登录").setBack(getRequest());
		}
		return comWalletService.saveAccountInfo(dto).setBack(getRequest());
	}


	/**
	 * 保存开户资料第二步（作废）
	 *
	 * @param comWalletAccount
	 * @author llc
	 * @date 2019-06-21
	 */
	@Log(operation = Log.OPERA_TYPE_SAVE, modle = "企业钱包服务", title = "保存开户资料第二步", systemType = Log.SYSTEMTYPE_PHONE)
	@ApiOperation(value = "保存开户资料第二步", httpMethod = "POST", notes = "保存开户资料第二步", response = ComWalletAccount.class)
	@PostMapping(value = "save-account-info-step-two")
	public Result saveAccountInfoStepTwo(ComWalletAccount comWalletAccount) {
		return comWalletService.saveAccountInfoStepTwo(comWalletAccount).setBack(getRequest());
	}

	/**
	 * 获取企业钱包账户
	 * 【百课堂在调用，勿加权限校验】
	 *
	 * @param comWalletAccount
	 * @author llc
	 * @date 2019-06-21
	 */
	@Log(operation = Log.OPERA_TYPE_SELECT, modle = "企业钱包服务", title = "获取企业钱包账户", systemType = Log.SYSTEMTYPE_PHONE)
	@ApiOperation(value = "获取企业钱包账户", httpMethod = "GET", notes = "获取企业钱包账户", response = ComWalletAccount.class)
	@GetMapping(value = "get-com-wallet-account")
	public Result getComWalletAccount(ComWalletAccount comWalletAccount) {
		/* 【百课堂在调用，勿加权限校验】 */
		return comWalletService.getComWalletAccount(comWalletAccount).setBack(getRequest());
	}

	/**
	 * <b> 获取会员档案认证地址 </b>
	 * @author ZXF
	 * @create 2023/02/17 0017 16:01
	 * @version
	 * @注意事项 </b>
	 */
	@Log(operation = Log.OPERA_TYPE_SELECT, modle = "企业钱包服务", title = "获取会员档案认证地址", systemType = Log.SYSTEMTYPE_PHONE)
	@ApiOperation(value = "获取会员档案认证地址", httpMethod = "POST", notes = "获取会员档案认证地址", response = Map.class)
	@PostMapping(value = "get-account-confirm")
	public Result getAccountConfirm(ComWalletAccount comWalletAccount) {
		/* 【百课堂在调用，勿加权限校验】 */
		return comWalletService.getAccountConfirm(comWalletAccount).setBack(getRequest());
	}

	/**
	 * 获取企业钱包账户金额
	 * 【百课堂在调用，勿加权限校验】
	 *
	 * @param comWalletAccount
	 * @author llc
	 * @date 2019-06-24
	 */
	@Log(operation = Log.OPERA_TYPE_SELECT, modle = "企业钱包服务", title = "获取企业钱包账户金额", systemType = Log.SYSTEMTYPE_PHONE)
	@ApiOperation(value = "获取企业钱包账户金额", httpMethod = "GET", notes = "获取企业钱包账户金额", response = Map.class)
	@GetMapping(value = "get-com-wallet-account-amount")
	public Result getComWalletAccountAmount(ComWalletAccount comWalletAccount) {
		/* 【百课堂在调用，勿加权限校验】 */
		return comWalletService.getComWalletAccountAmount(comWalletAccount).setBack(getRequest());
	}


	/**
	 * 获取企业钱包账户交易明细
	 *
	 * @param dto
	 * @author llc
	 * @date 2019-06-21
	 */
	@Log(operation = Log.OPERA_TYPE_SELECT, modle = "企业钱包服务", title = "获取企业钱包账户交易明细", systemType = Log.SYSTEMTYPE_PHONE)
	@ApiOperation(value = "获取企业钱包账户交易明细", httpMethod = "GET", notes = "获取企业钱包账户交易明细", response = Map.class)
	@GetMapping(value = "get-com-wallet-account-trade-detail")
	public Result getComWalletAccountTradeDetail(ComWalletAccountDTO dto) {
		return comWalletService.thisComWalletAccountTradeDetail(dto).setBack(getRequest());
//		return comWalletService.getComWalletAccountTradeDetail(dto).setBack(getRequest());
	}


	/**
	 * 将钱包收支明细保存到本地-任务
	 *
	 * @author lp
	 * @date 2019-12-09
	 */
	@Log(operation = Log.OPERA_TYPE_SELECT, modle = "企业钱包服务", title = "将钱包收支明细保存到本地", systemType = Log.SYSTEMTYPE_PHONE)
	@ApiOperation(value = "将钱包收支明细保存到本地", httpMethod = "POST", notes = "将钱包收支明细保存到本地", response = Map.class)
	@PostMapping(value = "com-wallet-account-trade-detail-task")
	public Result comWalletAccountTradeDetailTask(ComWalletAccountDTO dto) {
		return comWalletService.comWalletAccountTradeDetailTask(dto.getStartDate(), dto.getEndDate()).setBack(getRequest());
	}


	/**
	 * 获取企业钱包充值token
	 *
	 * @param dto
	 * @author llc
	 * @date 2019-06-25
	 */
	@Log(operation = Log.OPERA_TYPE_SELECT, modle = "企业钱包服务", title = "获取企业钱包充值token", systemType = Log.SYSTEMTYPE_PHONE)
	@ApiOperation(value = "获取企业钱包充值token", httpMethod = "POST", notes = "获取企业钱包充值token", response = Map.class)
	@PostMapping(value = "get-wallet-recharge-token")
	public Result getWalletRechargeToken(ComWalletAccountDTO dto) {
		return comWalletService.getWalletRechargeToken(dto).setBack(getRequest());
	}

	/**
	 * <b> 企业钱包开户回调 </b>
	 * @author ZXF
	 * @create 2023/02/06 0006 17:00
	 * @version
	 * @注意事项 </b>
	 */
	@Log(operation = Log.OPERA_TYPE_SAVE, modle = "企业钱包服务", title = "企业钱包开户回调", systemType = Log.SYSTEMTYPE_PHONE)
	@ApiOperation(value = "企业钱包开户回调", httpMethod = "POST", notes = "企业钱包开户回调", response = Map.class)
	@PostMapping(value = "com-wallet-register-callback")
	public Result comWalletRegisterCallback(String token) throws V5BusinessException, Exception {
		return comWalletService.comWalletRegisterCallback(token).setBack(getRequest());
	}

	/**
	 * 企业钱包充值回调
	 *
	 * @param token
	 * @author llc
	 * @date 2019-06-25
	 */
	@Log(operation = Log.OPERA_TYPE_SAVE, modle = "企业钱包服务", title = "企业钱包充值回调", systemType = Log.SYSTEMTYPE_PHONE)
	@ApiOperation(value = "企业钱包充值回调", httpMethod = "POST", notes = "企业钱包充值回调", response = Map.class)
	@PostMapping(value = "com-wallet-recharge-callback")
	public Result comWalletRechargeCallback(String token) throws V5BusinessException, Exception {
		return comWalletService.comWalletRechargeCallback(token).setBack(getRequest());
	}

	/**
	 * <b> 根据业务编码或业务交易流水号查询交易信息 </b>
	 * @author ZXF
	 * @create 2023/03/23 0023 10:23
	 * @version
	 * @注意事项 </b>
	 */
	@Log(operation = Log.OPERA_TYPE_SELECT, modle = "企业钱包服务", title = "根据业务编码或业务交易流水号查询交易信息", systemType = Log.SYSTEMTYPE_PHONE)
	@ApiOperation(value = "根据业务编码或业务交易流水号查询交易信息", httpMethod = "GET", notes = "根据业务编码或业务交易流水号查询交易信息", response = Map.class)
	@GetMapping(value = "find-wallet-trade-info-by-business")
	public Result getComWalletTradeBySeqAndCode(ComWalletTrade comWalletTrade) {
		return comWalletService.getComWalletTradeBySeqAndCode(comWalletTrade).setBack(getRequest());
	}

	/**
	 * 获取企业充值交易状态
	 *
	 * @param comWalletTrade
	 * @author llc
	 * @date 2019-06-28
	 */
	@Log(operation = Log.OPERA_TYPE_SELECT, modle = "企业钱包服务", title = "获取企业充值交易状态", systemType = Log.SYSTEMTYPE_PHONE)
	@ApiOperation(value = "获取企业充值交易状态", httpMethod = "GET", notes = "获取企业充值交易状态", response = Map.class)
	@GetMapping(value = "get-wallet-trade-info")
	public Result getWalletTradeInfo(ComWalletTrade comWalletTrade) {
		return comWalletService.getWalletTradeInfo(comWalletTrade).setBack(getRequest());
	}


	/**
	 * 企业钱包账户提现
	 *
	 * @param dto
	 * @author llc
	 * @date 2019-06-25
	 */
	@Log(operation = Log.OPERA_TYPE_SAVE, modle = "企业钱包服务", title = "企业钱包账户提现", systemType = Log.SYSTEMTYPE_PHONE)
	@ApiOperation(value = "企业钱包账户提现", httpMethod = "POST", notes = "企业钱包账户提现", response = Map.class)
	@PostMapping(value = "com-wallet-account-extract")
	public Result comWalletAccountExtract(ComWalletAccountDTO dto) {
		if (dto.getComId() == null) {
			throw new V5BusinessException(Result.ERROR, "企业ID不能为空");
		}

		String lockId = TRANSFERKEY+dto.getComId();
		try {
			// 开始新建前要通过redis分布式锁防止多方操作
			if (redisLockUtil.lock(lockId)) {
				return comWalletService.comWalletAccountExtract(dto).setBack(getRequest());
			} else {
				return Result.fail("操作失败，请稍后重试");
			}
		} finally {
			redisLockUtil.unlock(lockId);
		}
		/*RLock lock = redissonClient.getLock(TRANSFERKEY+dto.getComId());
		lock.lock();
		try {
			return comWalletService.comWalletAccountExtract(dto).setBack(getRequest());
		} catch (V5BusinessException e){
			return Result.fail(e.getMsg());
		} finally {
			lock.unlock();
		}*/
	}

	/**
	 * <b> 批量提现临时接口 </b>
	 * @author ZXF
	 * @create 2023/09/01 0001 15:51
	 * @version
	 * @注意事项 </b>
	 */
	@Log(operation = Log.OPERA_TYPE_SAVE, modle = "企业钱包服务", title = "批量提现临时接口", systemType = Log.SYSTEMTYPE_PHONE)
	@ApiOperation(value = "批量提现临时接口", httpMethod = "POST", notes = "批量提现临时接口", response = Map.class)
	@PostMapping(value = "batch-extract")
	public Result batchExtract(ComWallet comWallet) {
		return comWalletService.batchExtract(comWallet).setBack(getRequest());
	}


	/**
	 * <b> 企业账户金额冻结/解冻 </b>
	 * @author ZXF
	 * @create 2023/03/29 0029 17:20
	 * @version
	 * @注意事项 </b>
	 */
	@Log(operation = Log.OPERA_TYPE_SAVE, modle = "企业钱包服务", title = "企业账户金额冻结/解冻", systemType = Log.SYSTEMTYPE_PHONE)
	@ApiOperation(value = "企业消费转账接口", httpMethod = "POST", notes = "企业账户金额冻结/解冻", response = Map.class)
	@PostMapping(value = "com-wallet-account-lock")
	public Result comWalletAccountLock(ComWalletTransferDTO dto) {
		if (StringUtil.isEmpty(dto.getTradeType())) {
			throw new V5BusinessException(Result.ERROR, "交易类型不能为空");
		}
		if (dto.getOutId() == null) {
			throw new V5BusinessException(Result.ERROR, "企业出账ID不能为空");
		}
		if (dto.getInId() == null) {
			throw new V5BusinessException(Result.ERROR, "企业入账ID不能为空");
		}
		String comId = "05".equals(dto.getTradeType())?dto.getInId():dto.getOutId();
		String lockId = TRANSFERKEY+comId;
		try {
			// 开始新建前要通过redis分布式锁防止多方操作
			if (redisLockUtil.lock(lockId)) {
				return comWalletService.comWalletAccountLock(dto).setBack(getRequest());
			} else {
				return Result.fail("操作失败，请稍后重试");
			}
		} finally {
			redisLockUtil.unlock(lockId);
		}
	}

	/**
	 * 企业消费转账接口
	 * 【百课堂在调用，勿加权限校验】
	 *
	 * @author lp
	 * @date 2019-08-20
	 */
	@Log(operation = Log.OPERA_TYPE_SAVE, modle = "企业钱包服务", title = "企业消费转账接口", systemType = Log.SYSTEMTYPE_PHONE)
	@ApiOperation(value = "企业消费转账接口", httpMethod = "POST", notes = "企业消费转账接口", response = Map.class)
	@PostMapping(value = "com-wallet-account-transfer")
	public Result comWalletAccountTransfer(ComWalletTransferDTO dto) {
		if (dto.getOutId() == null) {
			throw new V5BusinessException(Result.ERROR, "企业出账ID不能为空");
		}
		if (dto.getInId() == null) {
			throw new V5BusinessException(Result.ERROR, "企业入账ID不能为空");
		}
		if (StringUtil.isEmpty(dto.getTradeType())) {
			throw new V5BusinessException(Result.ERROR, "交易类型不能为空");
		}

		String lockId = TRANSFERKEY+dto.getOutId();
		try {
			// 开始新建前要通过redis分布式锁防止多方操作
			if (redisLockUtil.lock(lockId)) {
				return comWalletService.comWalletAccountTransfer(dto).setBack(getRequest());
			} else {
				return Result.fail("操作失败，请稍后重试");
			}
		} finally {
			redisLockUtil.unlock(lockId);
		}
	}

	/**
	 * <b> 企业异步转账接口 </b>
	 * @author ZXF
	 * @create 2023/04/06 0006 10:53
	 * @version
	 * @注意事项 </b>
	 */
	@Log(operation = Log.OPERA_TYPE_SAVE, modle = "企业钱包服务", title = "企业消费转账接口", systemType = Log.SYSTEMTYPE_PHONE)
	@ApiOperation(value = "企业消费转账接口", httpMethod = "POST", notes = "企业消费转账接口", response = Map.class)
	@PostMapping(value = "sync-com-wallet-account-transfer")
	public Result syncComWalletAccountTransfer(ComWalletTransferDTO dto) {
		return comWalletService.syncComWalletAccountTransfer(dto).setBack(getRequest());
	}

	/**
	 * <b> 企业消费平台分账接口 </b>
	 * @author ZXF
	 * @create 2023/02/27 0027 16:02
	 * @version
	 * @注意事项 </b>
	 */
	@Log(operation = Log.OPERA_TYPE_SAVE, modle = "企业钱包服务", title = "企业消费平台分账接口", systemType = Log.SYSTEMTYPE_PHONE)
	@ApiOperation(value = "企业消费平台分账接口", httpMethod = "POST", notes = "企业消费平台分账接口", response = Map.class)
	@PostMapping(value = "sync-plat-wallet-account-transfer")
	public Result syncPlatWalletAccountTransfer(ComWalletTransferDTO dto) {
		return comWalletService.syncPlatWalletAccountTransfer(dto).setBack(getRequest());
	}

	/**
	 * <b> 企业消费平台分账接口 </b>
	 * @author ZXF
	 * @create 2023/02/27 0027 16:02
	 * @version
	 * @注意事项 </b>
	 */
	@Log(operation = Log.OPERA_TYPE_SAVE, modle = "企业钱包服务", title = "企业消费平台分账接口", systemType = Log.SYSTEMTYPE_PHONE)
	@ApiOperation(value = "企业消费平台分账接口", httpMethod = "POST", notes = "企业消费平台分账接口", response = Map.class)
	@PostMapping(value = "plat-wallet-account-transfer")
	public Result platWalletAccountTransfer(ComWalletTransferDTO dto) {
		return comWalletService.platWalletAccountTransfer(dto).setBack(getRequest());
	}


	/**
	 * 企业消费分账接口
	 *
	 * @author lp
	 * @date 2019-08-20
	 */
	@Log(operation = Log.OPERA_TYPE_SAVE, modle = "企业钱包服务", title = "企业消费分账接口", systemType = Log.SYSTEMTYPE_PHONE)
	@ApiOperation(value = "企业消费分账接口", httpMethod = "POST", notes = "企业消费分账接口", response = Map.class)
	@PostMapping(value = "com-wallet-account-consume-ledger")
	public Result comWalletAccountConsumeLedger(HttpServletRequest request) {
		String data = request.getParameter("data");
		ComWalletConsumeLedgerDTO dto = JSON.parseObject(data, ComWalletConsumeLedgerDTO.class);
		return comWalletService.comWalletAccountConsumeLedger(dto).setBack(getRequest());
	}

	/**
	 * 电子回单
	 *
	 * @param dto
	 * @author lp
	 * @date 2019-07-08
	 */
	@Log(operation = Log.OPERA_TYPE_SAVE, modle = "企业钱包服务", title = "电子回单", systemType = Log.SYSTEMTYPE_PHONE)
	@ApiOperation(value = "电子回单", httpMethod = "GET", notes = "电子回单", response = Map.class)
	@GetMapping(value = "com-account-trade-voucher")
	public void getComAccountTradeVoucher(ComWalletTradeDTO dto, HttpServletResponse response) throws IOException {
//		CurrentUser currentUser = new CurrentUser(getRequest());
//		if (dto.getComId() == null) {
//			dto.setComId(currentUser.getComId());
//		}
		BcxinFileUtils.downloadBase64(response, comWalletService.getComAccountTradeVoucher(dto));
	}


	/**
	 * 交易凭证
	 *
	 * @param dto
	 * @author lp
	 * @date 2019-07-08
	 */
	@Log(operation = Log.OPERA_TYPE_SAVE, modle = "企业钱包服务", title = "交易凭证", systemType = Log.SYSTEMTYPE_PHONE)
	@ApiOperation(value = "交易凭证", httpMethod = "GET", notes = "交易凭证", response = Map.class)
	@GetMapping(value = "com-account-pay-voucher")
	public void getComAccountPayVoucher(ComWalletTradeDTO dto, HttpServletResponse response) throws IOException {
		BcxinFileUtils.downloadBase64(response, comWalletService.getComAccountPayVoucher(dto));
	}

	/**
	 * 企业钱包账户对账数据生成
	 *
	 * @author llc
	 * @date 2019-09-16
	 */
	@Log(operation = Log.OPERA_TYPE_SAVE, modle = "企业钱包服务", title = "企业钱包账户对账数据生成", systemType = Log.SYSTEMTYPE_PHONE)
	@ApiOperation(value = "企业钱包账户对账数据生成", httpMethod = "POST", notes = "企业钱包账户对账数据生成", response = Map.class)
	@PostMapping(value = "create-com-wallet-reconciliation")
	public Result createComWalletReconciliation() {
		return comWalletService.createComWalletReconciliation("").setBack(getRequest());
	}


	/**
	 * 获取企业钱包账户对账数据
	 *
	 * @author llc
	 * @date 2019-09-19
	 */
	@Log(operation = Log.OPERA_TYPE_SAVE, modle = "企业钱包服务", title = "获取企业钱包账户对账数据", systemType = Log.SYSTEMTYPE_PHONE)
	@ApiOperation(value = "获取企业钱包账户对账数据", httpMethod = "POST", notes = "获取企业钱包账户对账数据", response = Map.class)
	@PostMapping(value = "get-com-wallet-reconciliation")
	public Result getComWalletReconciliation(ComWalletTradeDTO dto) {
		CurrentUser currentUser = new CurrentUser(getRequest());
		if (currentUser.getPerId() == null) {
			return Result.tokenExpired("用户登录超时或者未登录").setBack(getRequest());
		}
		return comWalletService.getComWalletReconciliation(dto).setBack(getRequest());
	}

	/**
	 * 导出企业钱包账户对账数据
	 *
	 * @author llc
	 * @date 2019-09-20
	 */
	@Log(operation = Log.OPERA_TYPE_SAVE, modle = "企业钱包服务", title = "导出企业钱包账户对账数据", systemType = Log.SYSTEMTYPE_PHONE)
	@ApiOperation(value = "导出企业钱包账户对账数据", httpMethod = "POST", notes = "导出企业钱包账户对账数据", response = Map.class)
	@GetMapping(value = "export-com-wallet-reconciliation")
	public Result exprokComWalletReconciliation(ComWalletTradeDTO dto) {
		CurrentUser currentUser = new CurrentUser(getRequest());
		if (currentUser.getPerId() == null) {
			return Result.tokenExpired("用户登录超时或者未登录").setBack(getRequest());
		}
		return comWalletService.exportComWalletReconciliation(dto, getResponse()).setBack(getRequest());
	}


	/**
	 * 查询待发放补贴
	 *
	 * @author llc
	 * @date 2020-10-16
	 */
	@Log(operation = Log.OPERA_TYPE_SELECT, modle = "企业钱包服务", title = "查询待发放补贴", systemType = Log.SYSTEMTYPE_PHONE)
	@ApiOperation(value = "查询待发放补贴", httpMethod = "POST", notes = "查询待发放补贴", response = Result.class)
	@PostMapping(value = "get-dff-subsidy-list")
	public Result getDffSubsidyList(ComWalletTransferDTO dto) {
		CurrentUser currentUser = new CurrentUser(getRequest());
		if (currentUser.getPerId() == null) {
			return Result.tokenExpired("用户登录超时或者未登录").setBack(getRequest());
		}
		return comWalletService.getDffSubsidyList(dto).setBack(getRequest());
	}

	/**
	 * 发送补贴验证码
	 *
	 * @author llc
	 * @date 2020-10-22
	 */
	@Log(operation = Log.OPERA_TYPE_SAVE, modle = "企业钱包服务", title = "发送补贴验证码", systemType = Log.SYSTEMTYPE_BACKMANAGE)
	@ApiOperation(value = "发送补贴验证码", httpMethod = "POST", notes = "发送补贴验证码", response = Result.class)
	@PostMapping(value = "send-subsidy-verification-code")
	public Result sendsubSidyVerificationCode(ComWalletTransferDTO dto) {
		CurrentUser currentUser = new CurrentUser(getRequest());
		if (currentUser.getPerId() == null) {
			return Result.tokenExpired("用户登录超时或者未登录").setBack(getRequest());
		}
		return comWalletService.sendsubSidyVerificationCode(dto).setBack(getRequest());
	}

	/**
	 * 获取百保盾补贴详情页面地址
	 *
	 * @author llc
	 * @date 2020-10-14
	 */
	@Log(operation = Log.OPERA_TYPE_SELECT, modle = "企业钱包服务", title = "获取百保盾补贴详情页面地址", systemType = Log.SYSTEMTYPE_PHONE)
	@ApiOperation(value = "获取百保盾补贴详情页面地址", httpMethod = "POST", notes = "获取百保盾补贴详情页面地址", response = Result.class)
	@PostMapping(value = "get-bbd-subsidy-page-url")
	public Result getBbdSubsidyPageUrl(ComWalletTransferDTO dto) {
		CurrentUser currentUser = new CurrentUser(getRequest());
		if (currentUser.getPerId() == null) {
			return Result.tokenExpired("用户登录超时或者未登录").setBack(getRequest());
		}
		return comWalletService.getBbdSubsidyPageUrl(dto).setBack(getRequest());
	}


	/**
	 * 发放保安员资格证补贴
	 *
	 * @author llc
	 * @date 2020-10-14
	 */
	@Log(operation = Log.OPERA_TYPE_SAVE, modle = "企业钱包服务", title = "发放保安员资格证补贴", systemType = Log.SYSTEMTYPE_PHONE)
	@ApiOperation(value = "发放保安员资格证补贴", httpMethod = "POST", notes = "发放保安员资格证补贴", response = Result.class)
	@PostMapping(value = "grant-bayzgz-subsidy")
	public Result grantBayZgzSubsidy(ComWalletTransferDTO dto) {
		CurrentUser currentUser = new CurrentUser(getRequest());
		if (currentUser.getPerId() == null) {
			return Result.tokenExpired("用户登录超时或者未登录").setBack(getRequest());
		}
		return comWalletService.grantBayZgzSubsidy(dto).setBack(getRequest());
	}


	/**
	 * 发放职业技能培训补贴
	 *
	 * @author llc
	 * @date 2020-10-14
	 */
	@Log(operation = Log.OPERA_TYPE_SAVE, modle = "企业钱包服务", title = "发放职业技能培训补贴", systemType = Log.SYSTEMTYPE_PHONE)
	@ApiOperation(value = "发放职业技能培训补贴", httpMethod = "POST", notes = "发放职业技能培训补贴", response = Result.class)
	@PostMapping(value = "grant-zyjnpx-subsidy")
	public Result grantZyjnpxSubsidy(ComWalletTransferDTO dto) {
		CurrentUser currentUser = new CurrentUser(getRequest());
		if (currentUser.getPerId() == null) {
			return Result.tokenExpired("用户登录超时或者未登录").setBack(getRequest());
		}
		return comWalletService.grantZyjnpxSubsidy(dto).setBack(getRequest());
	}


	/**
	 * 查询已发放补贴
	 *
	 * @author llc
	 * @date 2020-10-16
	 */
	@Log(operation = Log.OPERA_TYPE_SELECT, modle = "企业钱包服务", title = "查询已发放补贴", systemType = Log.SYSTEMTYPE_PHONE)
	@ApiOperation(value = "查询已发放补贴", httpMethod = "POST", notes = "查询已发放补贴", response = Result.class)
	@PostMapping(value = "get-yff-subsidy-list")
	public Result getYffSubsidyList(ComWalletTransferDTO dto) {
		CurrentUser currentUser = new CurrentUser(getRequest());
		if (currentUser.getPerId() == null) {
			return Result.tokenExpired("用户登录超时或者未登录").setBack(getRequest());
		}
		return comWalletService.getYffSubsidyList(dto).setBack(getRequest());
	}

	/**
	 * 查询已发放补贴信息
	 *
	 * @author llc
	 * @date 2020-10-22
	 */
	@Log(operation = Log.OPERA_TYPE_SELECT, modle = "企业钱包服务", title = "查询已发放补贴信息", systemType = Log.SYSTEMTYPE_PHONE)
	@ApiOperation(value = "查询已发放补贴信息", httpMethod = "POST", notes = "查询已发放补贴信息", response = Result.class)
	@PostMapping(value = "get-yff-subsidy-info")
	public Result getYffSubsidyInfo(ComWalletTransferDTO dto) {
		CurrentUser currentUser = new CurrentUser(getRequest());
		if (currentUser.getPerId() == null) {
			return Result.tokenExpired("用户登录超时或者未登录").setBack(getRequest());
		}
		return comWalletService.getYffSubsidyInfo(dto).setBack(getRequest());
	}

	/**
	 * <b> 保存钱包银行卡信息变更 </b>
	 *
	 * @author ZXF
	 * @create 2019/09/17 0017 15:45
	 * @version
	 * @注意事项 </b>
	 */
	@Log(operation = Log.OPERA_TYPE_SAVE, modle = "企业钱包信息变更", title = "保存钱包银行卡信息变更", systemType = Log.SYSTEMTYPE_PHONE)
	@ApiOperation(value = "保存钱包银行卡信息变更", httpMethod = "POST", notes = "保存钱包银行卡信息变更", response = ComWalletChangeDTO.class)
	@PostMapping(value = "save-bank-card-change-event")
	public Result saveBankCardChangeEvent(ComWalletChangeDTO dto) throws V5BusinessException {
		CurrentUser currentUser = new CurrentUser(getRequest());
		if (dto.getComId() == null) {
			dto.setComId(currentUser.getComId());
		}
		return comWalletChangeService.saveBankCardChangeEvent(dto).setBack(getRequest());
	}

}
