package com.bcxin.platform.web.controller.grant;

import com.bcxin.platform.common.exception.V5BusinessException;
import com.bcxin.platform.common.utils.Result;
import com.bcxin.platform.domain.grant.ConfigBankOutletsWorkDay;
import com.bcxin.platform.domain.grant.PerBankOutletsAppointment;
import com.bcxin.platform.dto.CurrentUser;
import com.bcxin.platform.dto.grant.ConfigBankOutletsWorkDayTimePeriodDto;
import com.bcxin.platform.dto.grant.PerJointlyCardDTO;
import com.bcxin.platform.service.grant.PerJointlyCardService;
import com.bcxin.platform.util.log.Log;
import com.bcxin.platform.web.controller.base.BaseController;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import org.springframework.web.bind.annotation.*;

import javax.annotation.Resource;
import java.io.UnsupportedEncodingException;
import java.util.List;
import java.util.Map;

/**
 * 个人联名卡服务
 * @author llc
 * @date 2020-07-21
 */
@RestController
@RequestMapping("/task/per-jointly-card")
@Api(tags = { "" }, description = "个人联名卡服务")
public class PerJointlyCardController extends BaseController {

	@Resource
	private PerJointlyCardService perJointlyCardService;


	/**
	 * 获取企业人员开通联名卡情况
	 * @param dto
	 * @author llc
	 * @date 2020-07-21
	 */
	@Log(operation = Log.OPERA_TYPE_SELECT, modle = "个人联名卡服务", title = "获取企业人员开通联名卡情况", systemType = Log.SYSTEMTYPE_PHONE)
	@ApiOperation(value = "获取企业人员开通联名卡情况", httpMethod = "GET", notes = "获取企业人员开通联名卡情况", response = Map.class)
	@GetMapping(value = "get-per-open-jointly-info")
	public Result getPerOpenJointlyInfo(PerJointlyCardDTO dto) {
		CurrentUser currentUser = new CurrentUser(getRequest());
		if (currentUser.getPerId() == null) {
			return Result.tokenExpired("用户登录超时或者未登录").setBack(getRequest());
		}
		return perJointlyCardService.getPerOpenJointlyInfo(dto).setBack(getRequest());
	}

	/**
	 * 获取企业未开通联名卡人员列表
	 * @param dto
	 * @author llc
	 * @date 2020-07-21
	 */
	@Log(operation = Log.OPERA_TYPE_SELECT, modle = "个人联名卡服务", title = "获取企业未开通联名卡人员列表", systemType = Log.SYSTEMTYPE_PHONE)
	@ApiOperation(value = "获取企业未开通联名卡人员列表", httpMethod = "GET", notes = "获取企业未开通联名卡人员列表", response = Map.class)
	@GetMapping(value = "get-no-open-jointly-per-list")
	public Result getNoOpenJointlyPerList(PerJointlyCardDTO dto) {
		CurrentUser currentUser = new CurrentUser(getRequest());
		if (currentUser.getPerId() == null) {
			return Result.tokenExpired("用户登录超时或者未登录").setBack(getRequest());
		}
		return perJointlyCardService.getNoOpenJointlyPerList(dto).setBack(getRequest());
	}

	/**
	 * 导出企业未开通联名卡人员
	 * @param dto
	 * @author llc
	 * @date 2020-09-27
	 */
	@GetMapping(value = "/export-no-open-jointly-per")
	@ApiOperation(value = "导出企业未开通联名卡人员", httpMethod = "GET", notes = "无", response = Result.class)
	@Log(operation = Log.OPERA_TYPE_SELECT, modle = "用户管理", title = "导出企业未开通联名卡人员", systemType = Log.SYSTEMTYPE_BACKMANAGE)
	public Result exportNoOpenJointlyPer(PerJointlyCardDTO dto) {
		CurrentUser currentUser = new CurrentUser(getRequest());
		if (currentUser.getPerId() == null) {
			return Result.tokenExpired("用户登录超时或者未登录").setBack(getRequest());
		}
		Result result = perJointlyCardService.exportNoOpenJointlyPer(dto,getResponse()).setBack(getRequest());
		return result;
	}

	/**
	 * 获取企业已开通联名卡人员列表
	 * @param dto
	 * @author llc
	 * @date 2020-07-21
	 */
	@Log(operation = Log.OPERA_TYPE_SELECT, modle = "个人联名卡服务", title = "获取企业已开通联名卡人员列表", systemType = Log.SYSTEMTYPE_PHONE)
	@ApiOperation(value = "获取企业已开通联名卡人员列表", httpMethod = "GET", notes = "获取企业已开通联名卡人员列表", response = Map.class)
	@GetMapping(value = "get-open-jointly-per-list")
	public Result getOpenJointlyPerList(PerJointlyCardDTO dto) {
		CurrentUser currentUser = new CurrentUser(getRequest());
		if (currentUser.getPerId() == null) {
			return Result.tokenExpired("用户登录超时或者未登录").setBack(getRequest());
		}
		return perJointlyCardService.getOpenJointlyPerList(dto).setBack(getRequest());
	}

	/**
	 * 导出企业已开通联名卡人员
	 * @param dto
	 * @author llc
	 * @date 2020-07-22
	 */
	@GetMapping(value = "/export-open-jointly-per")
	@ApiOperation(value = "导出企业已开通联名卡人员", httpMethod = "GET", notes = "无", response = Result.class)
	@Log(operation = Log.OPERA_TYPE_SELECT, modle = "用户管理", title = "导出企业已开通联名卡人员", systemType = Log.SYSTEMTYPE_BACKMANAGE)
	public Result exportOpenJointlyPer(PerJointlyCardDTO dto) {
		CurrentUser currentUser = new CurrentUser(getRequest());
		if (currentUser.getPerId() == null) {
			return Result.tokenExpired("用户登录超时或者未登录").setBack(getRequest());
		}
		Result result = perJointlyCardService.exportOpenJointlyPer(dto,getResponse()).setBack(getRequest());
		return result;
	}

	/**
	 * 批量提醒开通
	 * @param dto
	 * @author llc
	 * @date 2020-07-21
	 */
	@Log(operation = Log.OPERA_TYPE_SELECT, modle = "个人联名卡服务", title = "批量提醒开通", systemType = Log.SYSTEMTYPE_PHONE)
	@ApiOperation(value = "批量提醒开通", httpMethod = "POST", notes = "批量提醒开通", response = Map.class)
	@PostMapping(value = "batch-notice")
	public Result batchNotice(PerJointlyCardDTO dto) {
		CurrentUser currentUser = new CurrentUser(getRequest());
		if (currentUser.getPerId() == null) {
			return Result.tokenExpired("用户登录超时或者未登录").setBack(getRequest());
		}
		return perJointlyCardService.batchNotice(dto).setBack(getRequest());
	}



	/**
	 *  保安员联名卡批量同步
	 * @param list
	 * @auth llc
	 * @date 2020-07-22
	 * @return
	 * @throws Exception
	 */
	@PostMapping("/batch-sync-per-jointly")
	@ApiOperation(value = "保安员联名卡批量同步", notes = "保安员联名卡批量同步")
	@Log(operation = Log.OPERA_TYPE_UPDATE, modle = "个人联名卡服务", title = "保安员联名卡批量同步", systemType = Log.SYSTEMTYPE_BACKMANAGE)
	public Result batchSyncPerJointly(@RequestBody List<Map<String,String>> list) throws V5BusinessException {
		Result result = perJointlyCardService.batchSyncPerJointly(list).setBack(getRequest());
		return result;
	}

	/**
	 * 根据用户ID获取人员联名卡信息
	 *
	 * @param perId
	 * @return
	 * @author llc
	 * @date 2020-07-29
	 * 作为服务提供者,接收参数必须用 @RequestBody或者@RequestParam，单参数用@RequestParam
	 */
	@Log(operation = Log.OPERA_TYPE_SELECT, modle = "个人联名卡服务", title = "根据用户ID获取人员联名卡信息", systemType = Log.SYSTEMTYPE_BACKMANAGE)
	@PostMapping("/get-per-jointly-by-perid")
	@ApiOperation(value = "根据用户ID获取人员联名卡信息")
	public Result getPerJointlyByPerId(@RequestParam("perId") Long perId) throws V5BusinessException {
		Result result = perJointlyCardService.getPerJointlyByPerId(perId).setBack(getRequest());
		return result;
	}


	/**
	 * 获取保安员联名卡信息
	 *
	 * @param dto
	 * @return
	 * @author llc
	 * @date 2020-07-30
	 */
	@Log(operation = Log.OPERA_TYPE_SELECT, modle = "个人联名卡服务", title = "获取保安员联名卡信息", systemType = Log.SYSTEMTYPE_BACKMANAGE)
	@PostMapping("/get-per-jointly-card")
	@ApiOperation(value = "获取保安员联名卡信息")
	public Result getPerJointlyCard(PerJointlyCardDTO dto) throws V5BusinessException {
		CurrentUser currentUser = new CurrentUser(getRequest());
		if (currentUser.getPerId() == null) {
			return Result.tokenExpired("用户登录超时或者未登录").setBack(getRequest());
		}
		dto.setPerId(currentUser.getPerId());
		Result result = perJointlyCardService.getPerJointlyCard(dto).setBack(getRequest());
		return result;
	}


	/**
	 * 获取银行网点列表
	 *
	 * @param dto
	 * @return
	 * @author llc
	 * @date 2020-07-31
	 */
	@Log(operation = Log.OPERA_TYPE_SELECT, modle = "个人联名卡服务", title = "获取银行网点列表", systemType = Log.SYSTEMTYPE_BACKMANAGE)
	@PostMapping("/get-bank-outlets-list")
	@ApiOperation(value = "获取银行网点列表")
	public Result getBankOutletsList(PerJointlyCardDTO dto) throws V5BusinessException {
		CurrentUser currentUser = new CurrentUser(getRequest());
		if (currentUser.getPerId() == null) {
			return Result.tokenExpired("用户登录超时或者未登录").setBack(getRequest());
		}
		Result result = perJointlyCardService.getBankOutletsList(dto).setBack(getRequest());
		return result;
	}

	/**
	 * 获取银行网点工作日列表
	 *
	 * @param configBankOutletsWorkDay
	 * @return
	 * @author llc
	 * @date 2020-08-21
	 */
	@Log(operation = Log.OPERA_TYPE_SELECT, modle = "个人联名卡服务", title = "获取银行网点工作日列表", systemType = Log.SYSTEMTYPE_BACKMANAGE)
	@PostMapping("/get-bank-outlets-work-day-list")
	@ApiOperation(value = "获取银行网点工作日列表")
	public Result getBankOutletsWorkDayList(ConfigBankOutletsWorkDay configBankOutletsWorkDay) throws V5BusinessException {
		CurrentUser currentUser = new CurrentUser(getRequest());
		if (currentUser.getPerId() == null) {
			return Result.tokenExpired("用户登录超时或者未登录").setBack(getRequest());
		}
		Result result = perJointlyCardService.getBankOutletsWorkDayList(configBankOutletsWorkDay).setBack(getRequest());
		return result;
	}


	/**
	 * 获取银行网点工作日列表
	 *
	 * @param dto
	 * @return
	 * @author llc
	 * @date 2020-08-21
	 */
	@Log(operation = Log.OPERA_TYPE_SELECT, modle = "个人联名卡服务", title = "获取银行网点工作日时间段列表", systemType = Log.SYSTEMTYPE_BACKMANAGE)
	@PostMapping("/get-bank-outlets-work-day-time-period-list")
	@ApiOperation(value = "获取银行网点工作日时间段列表")
	public Result getBankOutletsWorkDayTimePeriodList(ConfigBankOutletsWorkDayTimePeriodDto dto) throws V5BusinessException {
		CurrentUser currentUser = new CurrentUser(getRequest());
		if (currentUser.getPerId() == null) {
			return Result.tokenExpired("用户登录超时或者未登录").setBack(getRequest());
		}
		Result result = perJointlyCardService.getBankOutletsWorkDayTimePeriodList(dto).setBack(getRequest());
		return result;
	}

	/**
	 * 预约银行网点
	 *
	 * @param perBankOutletsAppointment
	 * @return
	 * @author llc
	 * @date 2020-08-21
	 */
	@Log(operation = Log.OPERA_TYPE_SAVE, modle = "个人联名卡服务", title = "预约银行网点", systemType = Log.SYSTEMTYPE_BACKMANAGE)
	@PostMapping("/reserve-bank-outlets")
	@ApiOperation(value = "预约银行网点")
	public Result reserveBankOutlets(PerBankOutletsAppointment perBankOutletsAppointment) throws V5BusinessException {
		CurrentUser currentUser = new CurrentUser(getRequest());
		if (currentUser.getPerId() == null) {
			return Result.tokenExpired("用户登录超时或者未登录").setBack(getRequest());
		}
		perBankOutletsAppointment.setPerId(currentUser.getPerId());
		Result result = perJointlyCardService.reserveBankOutlets(perBankOutletsAppointment).setBack(getRequest());
		return result;
	}

	/**
	 * 查询个人预约记录
	 *
	 * @param perBankOutletsAppointment
	 * @return
	 * @author llc
	 * @date 2020-08-21
	 */
	@Log(operation = Log.OPERA_TYPE_SELECT, modle = "个人联名卡服务", title = "查询个人预约记录", systemType = Log.SYSTEMTYPE_BACKMANAGE)
	@PostMapping("/get-per-bank-outlets-appointment")
	@ApiOperation(value = "查询个人预约记录")
	public Result getPerBankOutletsAppointment(PerBankOutletsAppointment perBankOutletsAppointment) throws V5BusinessException {
		CurrentUser currentUser = new CurrentUser(getRequest());
		if (currentUser.getPerId() == null) {
			return Result.tokenExpired("用户登录超时或者未登录").setBack(getRequest());
		}
		perBankOutletsAppointment.setPerId(currentUser.getPerId());
		Result result = perJointlyCardService.getPerBankOutletsAppointment(perBankOutletsAppointment).setBack(getRequest());
		return result;
	}

	/**
	 * 取消银行网点预约
	 *
	 * @param perBankOutletsAppointment
	 * @return
	 * @author llc
	 * @date 2020-08-21
	 */
	@Log(operation = Log.OPERA_TYPE_DEL, modle = "个人联名卡服务", title = "取消银行网点预约", systemType = Log.SYSTEMTYPE_BACKMANAGE)
	@PostMapping("/cancel-reserve-bank-outlets")
	@ApiOperation(value = "取消银行网点预约")
	public Result cancelReserveBankOutlets(PerBankOutletsAppointment perBankOutletsAppointment) throws V5BusinessException {
		CurrentUser currentUser = new CurrentUser(getRequest());
		if (currentUser.getPerId() == null) {
			return Result.tokenExpired("用户登录超时或者未登录").setBack(getRequest());
		}
		perBankOutletsAppointment.setPerId(currentUser.getPerId());
		Result result = perJointlyCardService.cancelReserveBankOutlets(perBankOutletsAppointment).setBack(getRequest());
		return result;
	}

	/**
	 * <b> 获取联名卡申请、查询地址 </b>
	 * @author ZXF
	 * @create 2020/09/04 0004 14:30
	 * @version
	 * @注意事项 </b>
	 */
	@Log(operation = Log.OPERA_TYPE_DEL, modle = "个人联名卡服务", title = "获取联名卡申请、查询地址", systemType = Log.SYSTEMTYPE_BACKMANAGE)
	@PostMapping("/get-jointly-card-apply-url")
	@ApiOperation(value = "获取联名卡申请、查询地址")
	public Result getJointlyCardApplyUrl() throws UnsupportedEncodingException {
		CurrentUser currentUser = new CurrentUser(getRequest());
		if (currentUser.getPerId() == null) {
			return Result.tokenExpired("用户登录超时或者未登录").setBack(getRequest());
		}
		return perJointlyCardService.getJointlyCardApplyUrl(currentUser.getPerId()).setBack(getRequest());
	}

	/**
	 * <b> 获取联名卡申请条件 </b>
	 * @author ZXF
	 * @create 2020/09/09 0009 13:48
	 * @version
	 * @注意事项 </b>
	 */
	@Log(operation = Log.OPERA_TYPE_DEL, modle = "个人联名卡服务", title = "获取联名卡申请条件", systemType = Log.SYSTEMTYPE_BACKMANAGE)
	@PostMapping("/get-jointly-card-apply-conditions")
	@ApiOperation(value = "获取联名卡申请条件")
	public Result getJointlyCardApplyConditions() throws UnsupportedEncodingException {
		CurrentUser currentUser = new CurrentUser(getRequest());
		if (currentUser.getPerId() == null) {
			return Result.tokenExpired("用户登录超时或者未登录").setBack(getRequest());
		}
		return perJointlyCardService.getJointlyCardApplyConditions(currentUser.getPerId(),currentUser.getComId()).setBack(getRequest());
	}

}
