package com.bcxin.platform.service.wallet;

import com.bcxin.platform.common.exception.V5BusinessException;
import com.bcxin.platform.common.utils.Result;
import com.bcxin.platform.dto.order.ComOrderPayDTO;
import com.bcxin.platform.dto.wallet.ComWalletAccountDTO;
import com.bcxin.platform.dto.wallet.ComWalletTransferDTO;
import com.bcxin.platform.mapper.wallet.ComWalletAccountMapper;
import com.bcxin.platform.service.cache.TaskCacheService;
import com.bcxin.platform.service.common.CommonService;
import com.bcxin.platform.service.oauth.RedisUtil;
import com.bcxin.platform.util.constants.CommonConst;
import com.bcxin.platform.util.constants.Constants;
import com.bcxin.platform.util.constants.PublicConst;
import com.github.pagehelper.util.StringUtil;
import org.springframework.data.redis.core.StringRedisTemplate;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import javax.annotation.Resource;
import java.math.BigDecimal;
import java.util.Map;
import java.util.Objects;
import java.util.UUID;
import java.util.concurrent.TimeUnit;

/**
 * 企业支付服务，目前支持富民
 *
 * @author lp
 * @date 2020-01-15
 */
@Service
@Transactional("transactionManager")
public class ComWalletPaymentServiceImpl implements ComWalletPaymentService {

    @Resource
    private ComWalletAccountMapper comWalletAccountMapper;

    @Resource
    private TaskCacheService taskCacheService;

    @Resource
    private CommonService commonService;

    @Resource
    private ComWalletService comWalletService;

    @Resource
    private StringRedisTemplate stringRedisTemplate;

    @Resource
    private RedisUtil redisUtil;

    /**
     * 支付校验
     *
     * @param dto
     * @return
     * @throws V5BusinessException
     */
    @Override
    @Transactional("transactionManager")
    public Result payValidate(ComOrderPayDTO dto) throws V5BusinessException {
        if (dto.getActualPayAmount() == null) {
            return Result.fail("支付金额不能为空");
        }
        //if (BigDecimal.ZERO.compareTo(dto.getActualPayAmount()) == 0) {
        //    return Result.fail("支付金额不能为0");
        //}
        if (dto.getInComId() == null) {
            return Result.fail("入账方Id不能为空");
        }
        if (dto.getComId() == null) {
            return Result.fail("出账方Id不能为空");
        }
        ComWalletAccountDTO inComAccount = comWalletAccountMapper.getComWalletAccountByComId(dto.getInComId());
        ComWalletAccountDTO outComAccount = comWalletAccountMapper.getComWalletAccountByComId(dto.getComId());
        if (inComAccount == null) {
            return Result.fail("入账方未开通企业钱包，无法收款，请联系客服开通！");
        }
        if (StringUtil.isEmpty(inComAccount.getWalletAccountNo())) {
            return Result.fail("入账方未开通企业钱包，无法收款，请联系客服开通！");
        }
        if (outComAccount == null) {
            return Result.fail("您的公司未开通企业钱包，请先开通！");
        }
        if (StringUtil.isEmpty(outComAccount.getWalletAccountNo())) {
            return Result.fail("您的公司未开通企业钱包，请先开通！");
        }
        /*
         * 判断当前环境是否是测试环境，如果是则跳过校验环境
         */
        String envi = commonService.getSystemConfig("PLATFORM_SIGN");
        if (Objects.equals(envi, Constants.ENVI_TEST)) {
            return Result.success(Result.SUCCESS_QUERY_MSG);
        }
        /* 请求余额接口 */
        Result balanceResult = comWalletService.getComWalletAccountAmount(outComAccount);
        if (balanceResult.getData() != null) {
            Map<String, Object> map = (Map<String, Object>) balanceResult.getData();
            if (map.get("availableAmount") == null || StringUtil.isEmpty(map.get("availableAmount").toString())) {
                return Result.fail("获取公司账户余额失败，请联系客服！");
            }
            if (new BigDecimal(map.get("availableAmount").toString().replace(",",""))
                    .compareTo(dto.getActualPayAmount().setScale(2, BigDecimal.ROUND_HALF_DOWN)) < 0) {
                return Result.fail("您公司钱包余额不足，请先充值！");
            }
        }
        return Result.success(Result.SUCCESS_QUERY_MSG);
    }


    /**
     * 获得支付手机号
     *
     * @param dto
     * @return
     * @throws V5BusinessException
     */
    @Override
    public Result getPayPhone(ComOrderPayDTO dto) throws V5BusinessException {
        ComWalletAccountDTO outComAccount = comWalletAccountMapper.getComWalletAccountByComId(dto.getComId());
        return Result.success(Result.SUCCESS_QUERY_MSG, outComAccount.getMobilePhone());
    }

    /**
     * 发送支付短信接口
     *
     * @param dto
     * @return
     * @throws V5BusinessException
     */
    @Override
    public Result sendPayVerifyCode(ComOrderPayDTO dto) throws V5BusinessException {
        ComWalletAccountDTO outComAccount = comWalletAccountMapper.getComWalletAccountByComId(dto.getComId());

        String verifyCode = commonService.sendCode(outComAccount.getMobilePhone());

        /* 验证码3分钟失效 */
        stringRedisTemplate.opsForValue().set(redisUtil.REDIS_PREFIX_KEY + ":PAY:" + dto.getComOrderId() + outComAccount.getMobilePhone()
                , verifyCode, PublicConst.THREE_MIN_TIMEOUT_MILLISECONDS, TimeUnit.MILLISECONDS);

        /* 30分钟失效，防止重复订单提交 */
        stringRedisTemplate.opsForValue().set(redisUtil.REDIS_PREFIX_KEY + ":PAY:" + dto.getComOrderId() + dto.getComId()
                , UUID.randomUUID().toString(), PublicConst.HALF_HOUR_TIMEOUT_SECONDS, TimeUnit.SECONDS);

        return new Result();
    }

    /**
     * 实际支付接口
     *
     * @param dto
     * @return
     * @throws V5BusinessException
     */
    @Override
    @Transactional("transactionManager")
    public Result actualPay(ComOrderPayDTO dto) throws V5BusinessException {
        if (dto.getComId() == null) {
            Result result = new Result();
            result.setRetType(Result.ERROR);
            result.setMsg(Result.ERROR_MSG + "comId不能为空。");
            return result;
        }
        ComWalletAccountDTO outComAccount = comWalletAccountMapper.getComWalletAccountByComId(dto.getComId());
        /* 判断验证码是否输入正确 */
        String cacheVerifyCode = stringRedisTemplate.opsForValue().get(redisUtil.REDIS_PREFIX_KEY + ":PAY:" + dto.getComOrderId() + outComAccount.getMobilePhone());
        if (StringUtil.isEmpty(cacheVerifyCode)) {
            Result result = new Result();
            result.setRetType(Result.ERROR);
            result.setMsg(Result.ERROR_MSG + "验证码已失效，请重新获取。");
            return result;
        }
        if (!cacheVerifyCode.equals(dto.getActiveCode())) {
            Result result = new Result();
            result.setRetType(Result.ERROR);
            result.setMsg(Result.ERROR_MSG + "验证码不正确，请重新输入。");
            return result;
        }
        /* 判断只能支付一次的缓存标识 */
        String token = stringRedisTemplate.opsForValue().get(redisUtil.REDIS_PREFIX_KEY + ":PAY:" + dto.getComOrderId() + dto.getComId());
        if (StringUtil.isEmpty(token)) {
            Result result = new Result();
            result.setRetType(Result.ERROR);
            result.setMsg(Result.ERROR_MSG + "请勿重复提交订单或订单支付有效时间失败，请重新获取验证码进行支付。");
            return result;
        }
        /**
         * 判断当前环境是否是测试环境，如果是则跳过校验环境
         */
        String envi = commonService.getSystemConfig("PLATFORM_SIGN");
        if (Objects.equals(envi, Constants.ENVI_TEST)) {
            return Result.success(Result.SUCCESS_MSG);
        }
        if((BigDecimal.ZERO.compareTo(dto.getActualPayAmount()) == 0)){
            return Result.success(Result.SUCCESS_MSG);
        }
        /* 转账 */
        ComWalletTransferDTO transferDTO = new ComWalletTransferDTO();
        transferDTO.setOutId(dto.getComId().toString());
        transferDTO.setInId(dto.getInComId().toString());
        transferDTO.setAmount(dto.getActualPayAmount().setScale(2, BigDecimal.ROUND_HALF_DOWN).toString());
        transferDTO.setComment(dto.getComment());
        transferDTO.setNeedActiveCode(CommonConst.N);
        Result result = comWalletService.comWalletAccountTransfer(transferDTO);
        if (Result.ERROR.equals(result.getRetType())) {
            return result;
        }
        /* 清除缓存标识，不让重复提交 */
        stringRedisTemplate.delete(redisUtil.REDIS_PREFIX_KEY + ":PAY:" + dto.getComOrderId() + outComAccount.getMobilePhone());
        /* 清除验证码 */
        stringRedisTemplate.delete(redisUtil.REDIS_PREFIX_KEY + ":PAY:" + dto.getComOrderId() + dto.getComId());

        return Result.success(Result.SUCCESS_MSG);
    }


}