package com.bcxin.platform.service.grant;

import cn.hutool.http.HttpUtil;
import com.alibaba.fastjson.JSON;
import com.bcxin.platform.common.exception.V5BusinessException;
import com.bcxin.platform.common.utils.IdWorker;
import com.bcxin.platform.common.utils.Result;
import com.bcxin.platform.domain.grant.ComBhAccount;
import com.bcxin.platform.dto.grant.ComBhAccountDTO;
import com.bcxin.platform.mapper.grant.ComBhAccountMapper;
import com.bcxin.platform.service.common.CommonService;
import com.bcxin.platform.util.JwtUtil;
import com.bcxin.platform.util.ObjectUtils;
import com.bcxin.platform.util.codes.BhPaymentServiceConst;
import com.bcxin.platform.util.constants.CommonConst;
import com.github.pagehelper.util.StringUtil;
import com.google.common.collect.Maps;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import javax.annotation.Resource;
import java.math.BigDecimal;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

/**
 * 企业渤海银行
 *
 * @author llc
 * @date 2020-08-11
 */
@Service
@Transactional("transactionManager")
public class ComBhAccountServiceImpl implements ComBhAccountService {

    @Resource
    private ComBhAccountMapper comBhAccountMapper;

    @Resource
    private CommonService commonService;

    @Resource
    private IdWorker idWorker;

    /**
     * 获取企业渤海账户
     *
     * @param comBhAccount
     * @return
     * @author llc
     * @date 2020-08-11
     */
    @Override
    public Result getComBhAccount(ComBhAccount comBhAccount) throws V5BusinessException {
        comBhAccount = comBhAccountMapper.getComBhAccountByComId(comBhAccount.getComId());
        if(comBhAccount == null){
            return Result.success(Result.SUCCESS_QUERY_MSG);
        }
        /*** 调用支付平台接口获取渤海银行账户状态 ***/
        Map<Object,Object> p = Maps.newHashMap();
        p.put("MerUserId",comBhAccount.getComId().toString());//平台会员号
        String token = JwtUtil.createJWT(p);//加密生成token
        Map<String,Object> requestMap = Maps.newHashMap();
        requestMap.put("token",token);
        String platUrl = commonService.getSystemConfig(CommonConst.PAYMENT_PLATFORM_URL);
        String resultStr = HttpUtil.post(platUrl + BhPaymentServiceConst.GET_ACCOUNT_INFO, requestMap);
        Result result = JSON.parseObject(resultStr, Result.class);
        /*** 如果返回成功 ***/
        ComBhAccountDTO dto = new ComBhAccountDTO();
        ObjectUtils.copyProperties(dto, comBhAccount);
        if (Result.SUCCESS.equals(result.getRetType())) {
            Map<String, Object> map = (Map<String, Object>) result.getData();
            dto.setAccountStatus(map.get("Status").toString());
        }
        return Result.success(Result.SUCCESS_QUERY_MSG, dto);
    }


    /**
     * 获取企业钱包账户金额
     *
     * @param comBhAccount
     * @return
     * @author llc
     * @date 2020-08-12
     */
    @Override
    public Result getComBhAccountAmount(ComBhAccount comBhAccount) throws V5BusinessException {
        /*** 调用支付平台接口获取账户金额 ***/
        Map<Object, Object> p = new HashMap<>();
        p.put("MerUserId",comBhAccount.getComId().toString());//平台会员号
        p.put("VirlAcctType","01");//平台会员账户类型，用于校验交易密码01:基本户；02:保证金户； 04:只收丌付户；05:融资与用户
        String token = JwtUtil.createJWT(p);//加密生成token
        Map<String,Object> requestMap = Maps.newHashMap();
        requestMap.put("token",token);
        String platUrl = commonService.getSystemConfig(CommonConst.PAYMENT_PLATFORM_URL);
        String resultStr = HttpUtil.post(platUrl + BhPaymentServiceConst.GET_ACCOUNT_AMOUNT, requestMap);
        Result result = JSON.parseObject(resultStr, Result.class);
        Map<String, Object> resultMap = new HashMap<>();
        /*** 如果返回成功 ***/
        if (Result.SUCCESS.equals(result.getRetType())) {
            Map<String, Object> map = (Map<String, Object>) result.getData();
            /*** 账户总额 ***/
            resultMap.put("totalAmount", map.get("TotalAmt"));
            /*** 可用余额 ***/
            resultMap.put("availableAmount", map.get("AviAmt"));
            /*** 冻结金额 ***/
            resultMap.put("frozenAmount", map.get("CongealAmt"));
        } else {
            /*** 返回失败则抛错 ***/
            return Result.fail(result.getMsg());
        }
        return Result.success(Result.SUCCESS_QUERY_MSG, resultMap);
    }

    /**
     * 企业开户发送验证码
     * @param dto
     * @author llc
     * @date 2020-08-25
     * @return
     */
    @Override
    public Result sendOpenAccountVerifyCode(ComBhAccountDTO dto) throws V5BusinessException {
        if (StringUtil.isEmpty(dto.getIdCardNo())) {
            throw new V5BusinessException(Result.ERROR, "法定代表人证件号码不能为空");
        }
        if (StringUtil.isEmpty(dto.getMobilePhone())) {
            throw new V5BusinessException(Result.ERROR, "手机号不能为空");
        }
        if (StringUtil.isEmpty(dto.getBankCardNo())) {
            throw new V5BusinessException(Result.ERROR, "银行卡号不能为空");
        }
        if (StringUtil.isEmpty(dto.getComName())) {
            throw new V5BusinessException(Result.ERROR, "企业名称不能为空");
        }
        /*** 调用支付平台接口完成开户验证码发送 ***/
        Map<Object, Object> p = new HashMap<>();
        p.put("MerUserId",dto.getComId().toString());//平台会员号
        p.put("UserType","2");//虚拟客户对公对私标识1:个人；2:企业；4:个体工商户
        p.put("CertNo",dto.getIdCardNo());//身份证
        p.put("MobilePhone",dto.getMobilePhone());//(TransCode =HQJQ 必输) 手机号
        p.put("AccountNo",dto.getBankCardNo());//(TransCode =HQJQ 必输) 绑定实体账户账号
        p.put("AcctName",dto.getComName());//(TransCode =HQJQ 必输) 绑定实体账户户名
        String token = JwtUtil.createJWT(p);//加密生成token
        Map<String,Object> requestMap = Maps.newHashMap();
        requestMap.put("token",token);
        String platUrl = commonService.getSystemConfig(CommonConst.PAYMENT_PLATFORM_URL);
        String resultStr = HttpUtil.post(platUrl + BhPaymentServiceConst.SEND_OPEN_ACCOUNT_VERIFY_CODE, requestMap);
        Result result = JSON.parseObject(resultStr, Result.class);
        /*** 如果返回成功 ***/
        if (Result.SUCCESS.equals(result.getRetType())) {
            return Result.success("验证码发送成功");
        } else {
            /*** 返回失败则抛错 ***/
            return Result.fail(result.getMsg());
        }
    }

    /**
     * 注册企业渤海账户
     * @param dto
     * @author llc
     * @date 2020-08-11
     * @return
     */
    @Override
    public Result saveComBhAccount(ComBhAccountDTO dto) throws V5BusinessException {
        if (StringUtil.isEmpty(dto.getComName())) {
            throw new V5BusinessException(Result.ERROR, "企业名称不能为空");
        }
        if (StringUtil.isEmpty(dto.getComCerType())) {
            throw new V5BusinessException(Result.ERROR, "企业证件类型不能为空");
        }
        if (StringUtil.isEmpty(dto.getComCerNo())) {
            throw new V5BusinessException(Result.ERROR, "企业证件号码不能为空");
        }
        if (StringUtil.isEmpty(dto.getLegalRepName())) {
            throw new V5BusinessException(Result.ERROR, "法定代表人姓名不能为空");
        }
        if (StringUtil.isEmpty(dto.getIdCardNo())) {
            throw new V5BusinessException(Result.ERROR, "法定代表人证件号码不能为空");
        }
        if (StringUtil.isEmpty(dto.getMobilePhone())) {
            throw new V5BusinessException(Result.ERROR, "手机号不能为空");
        }
        if (StringUtil.isEmpty(dto.getIsBhBank())) {
            throw new V5BusinessException(Result.ERROR, "请选择是否渤海银行");
        }
        if (StringUtil.isEmpty(dto.getBankNo())) {
            throw new V5BusinessException(Result.ERROR, "银行行号不能为空");
        }
        if (StringUtil.isEmpty(dto.getBankOutlets())) {
            throw new V5BusinessException(Result.ERROR, "银行网点不能为空");
        }
        if (StringUtil.isEmpty(dto.getBankCardNo())) {
            throw new V5BusinessException(Result.ERROR, "银行卡号不能为空");
        }
        if (StringUtil.isEmpty(dto.getActiveCode())) {
            throw new V5BusinessException(Result.ERROR, "验证码不能为空");
        }

        /*** 调用支付平台接口完成开户资料上传 ***/
        Map<Object, Object> p = new HashMap<>();
        p.put("MerUserId",dto.getComId().toString());//平台会员号
        p.put("VirlAcctType","01");// 平台会员账户类型，用于校验交易密码01:基本户；02:保证金户； 04:只收丌付户；05:融资与用户
        p.put("UserType","2");//虚拟客户对公对私标识1:个人；2:企业；4:个体工商户
        p.put("EleAcctName",dto.getComName());// 虚账户户名
        p.put("CertNo",dto.getIdCardNo());//身份证
        p.put("BusPermitId",dto.getComCerNo());//社会统一信用代码UserType=2 或 UserType=4 必输
        p.put("MobilePhone",dto.getMobilePhone());//手机号
        p.put("BossName",dto.getLegalRepName());//法人姓名UserType=2 或 UserType=4 必输
        p.put("AcctNo",dto.getBankCardNo());//绑定账户（银行账户）
        p.put("AcctName",dto.getComName());// 绑定账户户名(企业名称)
        p.put("VerifyNo",dto.getActiveCode());//手机动态验证码
        p.put("AcctBankId",dto.getBankNo());//开户行行号 对公用户绑行外卡时必输
        p.put("AcctBankName",dto.getBankOutlets());//开户行行名 对公用户绑行外卡时必输
        p.put("AcctBHFlag",dto.getIsBhBank());//绑定实体账户行内外标识：0:行外（非渤海银行）；1:行内（渤海银行）
        p.put("JQOrderId",null);//鉴权码(HQDMIF 返回时上送)UserType=1 或 UserType=4 绑定行外卡时 必输
        p.put("NonAuthFlag","1");//免双密标识0：是;1：否。商户允许免密时，以此字段为主；商户丌允许免密时，此字段无效
        p.put("izAddress",null);//企业实际经营地址
        p.put("VisaDate",null);//证件发出日期 yyyy-MM-dd
        p.put("LostDate",null);//证件过期日期 yyyy-MM-dd
        String token = JwtUtil.createJWT(p);//加密生成token
        Map<String, Object> requestMap = new HashMap<>();
        requestMap.put("token", token);
        String platUrl = commonService.getSystemConfig(CommonConst.PAYMENT_PLATFORM_URL);
        String resultStr = HttpUtil.post(platUrl + BhPaymentServiceConst.OPEN_BH_ACCOUNT, requestMap);
        Result result = JSON.parseObject(resultStr, Result.class);
        /*** 如果返回成功 ***/
        if (Result.SUCCESS.equals(result.getRetType())) {
            Map<String, Object> map = (Map<String, Object>) result.getData();
            /*** 支付平台返回的钱包账户号 ***/

            /*** 写入saas钱包账户 ***/
            Date now = new Date();
            dto.setCreateTime(now);
            dto.setBhVirlAcctNo(map.get("VirlAcctNo").toString());
            comBhAccountMapper.insertComBhAccount(dto);

        } else {
            /*** 返回失败则抛错 ***/
            return Result.fail(result.getMsg());
        }
        return Result.success(Result.SUCCESS_MSG);
    }


    /**
     * 修改企业渤海账户
     * @param dto
     * @author llc
     * @date 2020-08-11
     * @return
     */
    @Override
    public Result updateComBhAccount(ComBhAccountDTO dto) throws V5BusinessException {
        if (StringUtil.isEmpty(dto.getMobilePhone())) {
            throw new V5BusinessException(Result.ERROR, "手机号不能为空");
        }
        if (StringUtil.isEmpty(dto.getBankOutlets())) {
            throw new V5BusinessException(Result.ERROR, "银行不能为空");
        }
        if (StringUtil.isEmpty(dto.getBankCardNo())) {
            throw new V5BusinessException(Result.ERROR, "银行卡号不能为空");
        }
        if (StringUtil.isEmpty(dto.getActiveCode())) {
            throw new V5BusinessException(Result.ERROR, "验证码不能为空");
        }
        /*** 调用支付平台接口完成开户资料上传 ***/
        Map<Object, Object> p = new HashMap<>();
        p.put("MerUserId",dto.getComId().toString());//平台会员号
        p.put("VerifyNo",dto.getActiveCode());//手机劢态验证码
        p.put("VirlAcctType","01");//平台会员账户类型，用于校验交易密码01:基本户；02:保证金户； 04:只收丌付户；05:融资与用户
        p.put("TransCodeIdx","XXBG");//来源交易:XNKH:信息绑定;XXBG:信息发更
        p.put("AccountNo",dto.getBankCardNo());//绑定实体账户账号
        p.put("AcctName",dto.getComName());// 绑定账户户名(企业名称)
        p.put("AcctBankId",dto.getBankNo());//开户行行号 对公用户绑行外卡时必输
        p.put("AcctBankName",dto.getBankOutlets());//开户行行名 对公用户绑行外卡时必输
        p.put("AcctBHFlag",dto.getIsBhBank());//绑定实体账户行内外标识：0:行外（非渤海银行）；1:行内（渤海银行）
        p.put("MobilePhone",dto.getMobilePhone());//手机号
        p.put("JQOrderId",null);//鉴权码(HQDMIF 返回时上送)UserType=1 或 UserType=4 绑
        String token = JwtUtil.createJWT(p);//加密生成token
        Map<String, Object> requestMap = new HashMap<>();
        requestMap.put("token", token);
        String platUrl = commonService.getSystemConfig(CommonConst.PAYMENT_PLATFORM_URL);
        String resultStr = HttpUtil.post(platUrl + BhPaymentServiceConst.UPDATE_BH_ACCOUNT, requestMap);
        Result result = JSON.parseObject(resultStr, Result.class);
        /*** 如果返回成功 ***/
        if (Result.SUCCESS.equals(result.getRetType())) {
            comBhAccountMapper.updateComBhAccount(dto);
        } else {
            /*** 返回失败则抛错 ***/
            return Result.fail(result.getMsg());
        }
        return Result.success(Result.SUCCESS_MSG);
    }


    /**
     * 查询渤海银行列表
     *
     * @param comBhAccount
     * @return
     * @author llc
     * @date 2020-08-11
     */
    @Override
    public Result getBhBankList(ComBhAccount comBhAccount) throws V5BusinessException {
        /*** 调用支付平台接口获取银行列表 ***/

        return Result.success(Result.SUCCESS_QUERY_MSG, comBhAccount);
    }

    /**
     * 企业业务发送验证码
     *
     * @param dto
     * @return
     * @author llc
     * @date 2020-08-11
     */
    @Override
    public Result sendVerifyCode(ComBhAccountDTO dto) throws V5BusinessException {
        /*** 取TransCode
         * 交易码     交易码解释     对应 TransCode
         XNCZGIF    虚拟充值              XNCZ
         XNTXGIF    虚拟提现              XNTX
         XNZZGIF    虚账户转账            XNZZ
         DJJDGIF    冻结解冻              DJJD
         JDZZGIF    解冻转账              JDZZ
         YEDDQRGIF  支付订单确认          DDQR
         SHFKQRGIF  确认收货付款          SHFKQR
         ZFTKQRGIF  退款 ZFTKQR
         XXBDGIF    账户信息绑定、变更    HQJQ
         XNTK       线下入金退款
         FXJHRQ     反向激活重发
         FXJHTB     反向激活验证
         MYQY       免验签约
         XNXH       销户
         BZXXBD     对公信息变更
         DFXZTZ     代发明细文件上传通知  DFXZ
         */
        if (StringUtil.isEmpty(dto.getMobilePhone())) {
            throw new V5BusinessException(Result.ERROR, "手机号不能为空");
        }
        if (StringUtil.isEmpty(dto.getTransCode())) {
            throw new V5BusinessException(Result.ERROR, "交易码不能为空");
        }else{
            /*** 如果交易类型为账户信息绑定、变更需要校验户名和卡号不能为空 ***/
            if(dto.getTransCode().equals("HQJQ")){
                if (StringUtil.isEmpty(dto.getComName())) {
                    throw new V5BusinessException(Result.ERROR, "户名(企业名)不能为空");
                }
                if (StringUtil.isEmpty(dto.getBankCardNo())) {
                    throw new V5BusinessException(Result.ERROR, "银行卡号不能为空");
                }
            }
            /*** 如果为交易短信，金额不能为空或者0 ***/
            if(dto.getTransCode().equals("XNCZ") || dto.getTransCode().equals("XNTX")|| dto.getTransCode().equals("XNZZ") ){
                if (dto.getAmount() == null) {
                    return Result.fail("金额不能为空");
                }
                if(dto.getAmount() != null && BigDecimal.ZERO.compareTo(dto.getAmount()) == 0){
                    return Result.fail("金额不能为0");
                }
            }
        }

        /*** 调用支付平台接口完成验证码发送 ***/
        Map<Object, Object> p = new HashMap<>();
        p.put("MerUserId",dto.getComId().toString());//平台会员号
        p.put("TransCode",dto.getTransCode());// 交易码（参照文档动态验证码附表）
        p.put("MobilePhone",dto.getMobilePhone());//(TransCode =HQJQ 必输) 手机号
        p.put("AccountNo",dto.getBankCardNo());//(TransCode =HQJQ 必输) 绑定实体账户账号
        p.put("AcctName",dto.getComName());//(TransCode =HQJQ 必输) 绑定实体账户户名
        p.put("TransAmount",dto.getAmount());//交易金额，如果是动账交易金额则必输
        String token = JwtUtil.createJWT(p);
        Map<String, Object> requestMap = new HashMap<>();
        requestMap.put("token", token);
        String platUrl = commonService.getSystemConfig(CommonConst.PAYMENT_PLATFORM_URL);
        String resultStr = HttpUtil.post(platUrl + BhPaymentServiceConst.SEND_VERIFY_CODE, requestMap);
        Result result = JSON.parseObject(resultStr, Result.class);
        /*** 如果返回成功 ***/
        if (Result.SUCCESS.equals(result.getRetType())) {
            return Result.success("验证码发送成功");
        } else {
            /*** 返回失败则抛错 ***/
            return Result.fail(result.getMsg());
        }
    }

    /**
     * 渤海银账户充值
     *
     * @param dto
     * @return
     * @author llc
     * @date 2020-08-12
     */
    @Override
    public Result bhAccountRecharge(ComBhAccountDTO dto) throws V5BusinessException {
        if (StringUtil.isEmpty(dto.getActiveCode())) {
            return Result.fail("手机验证码不能为空");
        }
        if (dto.getAmount() == null) {
            return Result.fail("金额不能为空");
        }
        if(dto.getAmount() != null && BigDecimal.ZERO.compareTo(dto.getAmount()) == 0){
            return Result.fail("金额不能为0");
        }
        /*** 调用支付平台接口进行充值 ***/
        Map<Object, Object> p = new HashMap<>();
        p.put("MerUserId",dto.getComId().toString());//平台会员号
        p.put("VirlAcctType","01");//台会员账户类型 01:基本户;02:保证金户;03:结算户;04:只收丌付户;05:融资与用户
        p.put("Amount",dto.getAmount().toString());//金额(精确2位小数)
        p.put("VerifyNo",dto.getActiveCode());//验证码
        String token = JwtUtil.createJWT(p);
        Map<String, Object> requestMap = new HashMap<>();
        requestMap.put("token", token);
        String platUrl = commonService.getSystemConfig(CommonConst.PAYMENT_PLATFORM_URL);
        String resultStr = HttpUtil.post(platUrl + BhPaymentServiceConst.ACCOUNT_RECHARGE, requestMap);
        Result result = JSON.parseObject(resultStr, Result.class);
        /*** 如果返回成功 ***/
        if (Result.SUCCESS.equals(result.getRetType())) {
            return Result.success("充值成功");
        } else {
            /*** 返回失败则抛错 ***/
            return Result.fail(result.getMsg());
        }
    }


    /**
     * 渤海银账户提现
     *
     * @param dto
     * @return
     * @author llc
     * @date 2020-08-12
     */
    @Override
    public Result bhAccountWithdrawa(ComBhAccountDTO dto) throws V5BusinessException {
        if (StringUtil.isEmpty(dto.getActiveCode())) {
            return Result.fail("手机验证码不能为空");
        }
        if (dto.getAmount() == null) {
            return Result.fail("金额不能为空");
        }
        if(dto.getAmount() != null && BigDecimal.ZERO.compareTo(dto.getAmount()) == 0){
            return Result.fail("金额不能为0");
        }
        /*** 调用支付平台接口进行提现 ***/
        Map<Object, Object> p = new HashMap<>();
        p.put("MerUserId",dto.getComId().toString());//平台会员号
        p.put("VirlAcctType","01");//平台会员账户类型 01:基本户;02:保证金户;03:结算户
        p.put("Amount",dto.getAmount().toString());//金额
        p.put("VerifyNo",dto.getActiveCode());//手机劢态验证码
        p.put("Memo",null);//附言
        String token = JwtUtil.createJWT(p);
        Map<String, Object> requestMap = new HashMap<>();
        requestMap.put("token", token);
        String platUrl = commonService.getSystemConfig(CommonConst.PAYMENT_PLATFORM_URL);
        String resultStr = HttpUtil.post(platUrl + BhPaymentServiceConst.ACCOUNT_WITHDRAWA, requestMap);
        Result result = JSON.parseObject(resultStr, Result.class);
        /*** 如果返回成功 ***/
        if (Result.SUCCESS.equals(result.getRetType())) {
            return Result.success("提现成功");
        } else {
            /*** 返回失败则抛错 ***/
            return Result.fail(result.getMsg());
        }
    }


    /**
     * 获取账户历史明细明细列表
     *
     * @param dto
     * @return
     * @author llc
     * @date 2020-08-12
     */
    @Override
    public Result getAccountHistoryDetailList(ComBhAccountDTO dto) throws V5BusinessException {
        if (StringUtil.isEmpty(dto.getStartDate())) {
            return Result.fail("开始日期不能为空");
        }
        if (StringUtil.isEmpty(dto.getEndDate())) {
            return Result.fail("结束日期不能为空");
        }
        /*** 调用支付平台获取交易明细列表 ***/
        Map<Object,Object> p = Maps.newHashMap();
        p.put("MerUserId",dto.getComId().toString());//平台会员号
        p.put("VirlAcctType","01");//虚账户类型：00-平台虚拟过渡户，01-基本户，02-保证金户，03-结算户，04-只收丌付，05-融资与用户
        p.put("BeginDate",dto.getStartDate());//开始日期
        p.put("EndDate",dto.getEndDate());//结束日期
        String token = JwtUtil.createJWT(p);
        Map<String,Object> requestMap = Maps.newHashMap();
        requestMap.put("token",token);
        String platUrl = commonService.getSystemConfig(CommonConst.PAYMENT_PLATFORM_URL);
        String resultStr = HttpUtil.post(platUrl + BhPaymentServiceConst.GET_ACCOUNT_HISTORY_DETAIL_LIST, requestMap);
        Result result = JSON.parseObject(resultStr, Result.class);
        /*** 如果返回成功 ***/
        if (Result.SUCCESS.equals(result.getRetType())) {
            result.setMsg(Result.SUCCESS_QUERY_MSG);
            return result;
        } else {
            /*** 返回失败则抛错 ***/
            return Result.fail(result.getMsg());
        }
    }

    /**
     * 修改手机
     * @param dto
     * @author llc
     * @date 2020-10-12
     * @return
     */
    @Override
    public Result updateMobilePhone(ComBhAccountDTO dto) throws V5BusinessException {
        if (StringUtil.isEmpty(dto.getMobilePhone())) {
            throw new V5BusinessException(Result.ERROR, "手机号不能为空");
        }
        if (StringUtil.isEmpty(dto.getActiveCode())) {
            throw new V5BusinessException(Result.ERROR, "验证码不能为空");
        }
        ComBhAccount comBhAccount = comBhAccountMapper.getComBhAccountByComId(dto.getComId());

        /*** 调用支付平台接口完成开户资料上传 ***/
        Map<Object, Object> p = new HashMap<>();
        p.put("MerUserId",dto.getComId().toString());//平台会员号
        p.put("VerifyNo",dto.getActiveCode());//手机劢态验证码
        p.put("VirlAcctType","01");//平台会员账户类型，用于校验交易密码01:基本户；02:保证金户； 04:只收丌付户；05:融资与用户
        p.put("TransCodeIdx","XXBG");//来源交易:XNKH:信息绑定;XXBG:信息发更
        p.put("AccountNo",comBhAccount.getBankCardNo());//绑定实体账户账号
        p.put("AcctName",comBhAccount.getComName());// 绑定账户户名(企业名称)
        p.put("AcctBankId",comBhAccount.getBankNo());//开户行行号 对公用户绑行外卡时必输
        p.put("AcctBankName",comBhAccount.getBankOutlets());//开户行行名 对公用户绑行外卡时必输
        p.put("AcctBHFlag",comBhAccount.getIsBhBank());//绑定实体账户行内外标识：0:行外（非渤海银行）；1:行内（渤海银行）
        p.put("MobilePhone",dto.getMobilePhone());//手机号
        p.put("JQOrderId",null);//鉴权码(HQDMIF 返回时上送)UserType=1 或 UserType=4 绑
        String token = JwtUtil.createJWT(p);//加密生成token
        Map<String, Object> requestMap = new HashMap<>();
        requestMap.put("token", token);
        String platUrl = commonService.getSystemConfig(CommonConst.PAYMENT_PLATFORM_URL);
        String resultStr = HttpUtil.post(platUrl + BhPaymentServiceConst.UPDATE_BH_ACCOUNT, requestMap);
        Result result = JSON.parseObject(resultStr, Result.class);
        /*** 如果返回成功 ***/
        if (Result.SUCCESS.equals(result.getRetType())) {
            comBhAccountMapper.updateComBhAccount(dto);
        } else {
            /*** 返回失败则抛错 ***/
            return Result.fail(result.getMsg());
        }
        return Result.success(Result.SUCCESS_MSG);
    }

    /**
     * <b> 企业渤海账户列表查询 </b>
     * @author ZXF
     * @create 2020/08/17 0017 14:38
     * @version
     * @注意事项 </b>
     */
    @Override
    public List<ComBhAccountDTO> pageComBhAccountList(ComBhAccountDTO dto) {
        if(StringUtil.isNotEmpty(dto.getKeyword())){
            dto.setKeyword(dto.getKeyword().trim());
        }
        return comBhAccountMapper.pageComBhAccountList(dto);
    }

    /**
     * <b> 获取账户历史明细明细列表 </b>
     * @author ZXF
     * @create 2020/08/17 0017 14:38
     * @version
     * @注意事项 </b>
     */
    @Override
    public List pagePaymentDetailsList(ComBhAccountDTO dto) throws V5BusinessException {
        if (StringUtil.isEmpty(dto.getStartDate())) {
            throw new V5BusinessException(Result.ERROR,"开始日期不能为空");
        }
        if (StringUtil.isEmpty(dto.getEndDate())) {
            throw new V5BusinessException(Result.ERROR,"结束日期不能为空");
        }
        /*** 调用支付平台获取交易明细列表 ***/
        Map<Object,Object> p = Maps.newHashMap();
        p.put("MerUserId",dto.getComId().toString());//平台会员号
        p.put("VirlAcctType","01");//虚账户类型：00-平台虚拟过渡户，01-基本户，02-保证金户，03-结算户，04-只收丌付，05-融资与用户
        p.put("BeginDate",dto.getStartDate());//开始日期
        p.put("EndDate",dto.getEndDate());//结束日期
        String token = JwtUtil.createJWT(p);
        Map<String,Object> requestMap = Maps.newHashMap();
        requestMap.put("token",token);
        String platUrl = commonService.getSystemConfig(CommonConst.PAYMENT_PLATFORM_URL);
        String resultStr = HttpUtil.post(platUrl + BhPaymentServiceConst.GET_ACCOUNT_HISTORY_DETAIL_LIST, requestMap);
        Result result = JSON.parseObject(resultStr, Result.class);
        /*** 如果返回成功 ***/
        if (Result.SUCCESS.equals(result.getRetType())) {
            List listW = (List)result.getData();
            List listN = (List)listW.get(0);
            return listN;
        } else {
            throw new V5BusinessException(Result.ERROR,result.getMsg());
        }
    }

    /**
     * <b> 获取企业渤海账户 </b>
     * @author ZXF
     * @create 2020/08/17 0017 14:38
     * @version
     * @注意事项 </b>
     */
    @Override
    public ComBhAccountDTO getComBhAccount(String comId){
        ComBhAccount comBhAccount = comBhAccountMapper.getComBhAccountByComId(Long.parseLong(comId));
        if(comBhAccount == null){
            return null;
        }
        /*** 调用支付平台接口获取渤海银行账户状态 ***/
        Map<Object,Object> p = Maps.newHashMap();
        p.put("MerUserId",comBhAccount.getComId().toString());//平台会员号
        String token = JwtUtil.createJWT(p);//加密生成token
        Map<String,Object> requestMap = Maps.newHashMap();
        requestMap.put("token",token);
        String platUrl = commonService.getSystemConfig(CommonConst.PAYMENT_PLATFORM_URL);
        String resultStr = HttpUtil.post(platUrl + BhPaymentServiceConst.GET_ACCOUNT_INFO, requestMap);
        Result result = JSON.parseObject(resultStr, Result.class);
        /*** 如果返回成功 ***/
        ComBhAccountDTO cDto = new ComBhAccountDTO();
        ObjectUtils.copyProperties(cDto, comBhAccount);
        if (Result.SUCCESS.equals(result.getRetType())) {
            Map<String, Object> map = (Map<String, Object>) result.getData();
            String status = map.get("Status").toString();
            cDto.setAccountStatus("0".equals(status)?"正常":"7".equals(status)?"客户绑卡未激活":"8".equals(status)?"平台户开户处理中":"9".equals(status)?"普通客户开户处理中":"");
        }
        return cDto;
    }

    /**
     * <b> 修改发薪权限 </b>
     * @author ZXF
     * @create 2020/09/16 0016 13:31
     * @version
     * @注意事项 </b>
     */
    @Override
    public Result updateOpenSalaryAuth(ComBhAccountDTO dto){
        if(dto.getComId() == null){
            return Result.fail("comId不能为空");
        }
        if(StringUtil.isEmpty(dto.getIsOpenSalaryAuth())){
            return Result.fail("isOpenSalaryAuth不能为空");
        }
        comBhAccountMapper.updateOpenSalaryAuth(dto);
        return Result.success("1".equals(dto.getIsOpenSalaryAuth())?"发薪权限已开通！":"发薪权限已关闭！");
    }
}