import { Feature, LineString } from 'geojson';
import { Coord, Units } from '@turf/helpers';

/**
 * Calculates the distance between a given point and the nearest point on a
 * line. Sometimes referred to as the cross track distance.
 *
 * @function
 * @param {Feature<Point>|Array<number>} pt Feature or Geometry
 * @param {Feature<LineString>} line GeoJSON Feature or Geometry
 * @param {Object} [options={}] Optional parameters
 * @param {string} [options.units="kilometers"] can be anything supported by turf/convertLength
 * (ex: degrees, radians, miles, or kilometers)
 * @param {string} [options.method="geodesic"] whether to calculate the distance based on geodesic (spheroid) or
 * planar (flat) method. Valid options are 'geodesic' or 'planar'.
 * @returns {number} distance between point and line
 * @example
 * var pt = turf.point([0, 0]);
 * var line = turf.lineString([[1, 1],[-1, 1]]);
 *
 * var distance = turf.pointToLineDistance(pt, line, {units: 'miles'});
 * //=69.11854715938406
 */
declare function pointToLineDistance(pt: Coord, line: Feature<LineString> | LineString, options?: {
    units?: Units;
    method?: "geodesic" | "planar";
}): number;

export { pointToLineDistance as default, pointToLineDistance };
