"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
var crypto = require("crypto");
var fs = require("fs");
var fsExtra = require("fs-extra");
var path = require("path");
var tsc = require("typescript");
var logger_1 = require("./logger");
function getTSJestConfig(globals) {
    return globals && globals['ts-jest'] ? globals['ts-jest'] : {};
}
exports.getTSJestConfig = getTSJestConfig;
function formatTscParserErrors(errors) {
    return errors.map(function (s) { return JSON.stringify(s, null, 4); }).join('\n');
}
function readCompilerOptions(configPath) {
    configPath = path.resolve(configPath);
    var loaded = tsc.readConfigFile(configPath, function (file) {
        var read = tsc.sys.readFile(file);
        if (!read) {
            throw new Error("ENOENT: no such file or directory, open '" + configPath + "'");
        }
        return read;
    });
    if (loaded.error) {
        throw new Error(JSON.stringify(loaded.error, null, 4));
    }
    var basePath = path.dirname(configPath);
    var parsedConfig = tsc.parseJsonConfigFileContent(loaded.config, tsc.sys, basePath);
    if (parsedConfig.errors.length > 0) {
        var formattedErrors = formatTscParserErrors(parsedConfig.errors);
        throw new Error("Some errors occurred while attempting to read from " + configPath + ": " + formattedErrors);
    }
    return parsedConfig.options;
}
function getStartDir() {
    var grandparent = path.resolve(__dirname, '..', '..');
    if (grandparent.endsWith('/node_modules')) {
        return process.cwd();
    }
    return '.';
}
function getPathToClosestTSConfig(startDir, previousDir) {
    if (!startDir) {
        return getPathToClosestTSConfig(getStartDir());
    }
    var tsConfigPath = path.join(startDir, 'tsconfig.json');
    var startDirPath = path.resolve(startDir);
    var previousDirPath = path.resolve(previousDir || '/');
    if (startDirPath === previousDirPath || fs.existsSync(tsConfigPath)) {
        return tsConfigPath;
    }
    return getPathToClosestTSConfig(path.join(startDir, '..'), startDir);
}
function getTSConfigPathFromConfig(globals) {
    var defaultTSConfigFile = getPathToClosestTSConfig();
    if (!globals) {
        return defaultTSConfigFile;
    }
    var tsJestConfig = getTSJestConfig(globals);
    if (tsJestConfig.tsConfigFile) {
        return tsJestConfig.tsConfigFile;
    }
    return defaultTSConfigFile;
}
function mockGlobalTSConfigSchema(globals) {
    var configPath = getTSConfigPathFromConfig(globals);
    return { 'ts-jest-babel-7': { tsConfigFile: configPath } };
}
exports.mockGlobalTSConfigSchema = mockGlobalTSConfigSchema;
var tsConfigCache = {};
function getTSConfig(globals, collectCoverage) {
    if (collectCoverage === void 0) { collectCoverage = false; }
    var configPath = getTSConfigPathFromConfig(globals);
    logger_1.logOnce("Reading tsconfig file from path " + configPath);
    var skipBabel = getTSJestConfig(globals).skipBabel;
    var tsConfigCacheKey = JSON.stringify([
        skipBabel,
        collectCoverage,
        configPath,
    ]);
    if (tsConfigCacheKey in tsConfigCache) {
        return tsConfigCache[tsConfigCacheKey];
    }
    var config = readCompilerOptions(configPath);
    logger_1.logOnce('Original typescript config before modifications: ', config);
    delete config.sourceMap;
    config.inlineSourceMap = true;
    config.inlineSources = true;
    delete config.outDir;
    if (configPath === path.join(getStartDir(), 'tsconfig.json')) {
        config.module = tsc.ModuleKind.CommonJS;
    }
    config.module = config.module || tsc.ModuleKind.CommonJS;
    config.jsx = config.jsx || tsc.JsxEmit.React;
    if (config.allowSyntheticDefaultImports && !skipBabel) {
        config.module = tsc.ModuleKind.ES2015;
    }
    tsConfigCache[tsConfigCacheKey] = config;
    return config;
}
exports.getTSConfig = getTSConfig;
function cacheFile(jestConfig, filePath, src) {
    if (!jestConfig.testRegex || !filePath.match(jestConfig.testRegex)) {
        var outputFilePath = path.join(jestConfig.cacheDirectory, '/ts-jest-babel-7/', crypto
            .createHash('md5')
            .update(filePath)
            .digest('hex'));
        fsExtra.outputFileSync(outputFilePath, src);
    }
}
exports.cacheFile = cacheFile;
function injectSourcemapHook(filePath, typeScriptCode, src) {
    var start = src.length > 12 ? src.substr(1, 10) : '';
    var filePathParam = JSON.stringify(filePath);
    var codeParam = JSON.stringify(typeScriptCode);
    var sourceMapHook = "require('ts-jest-babel-7').install(" + filePathParam + ", " + codeParam + ")";
    return start === 'use strict'
        ? "'use strict';" + sourceMapHook + ";" + src
        : sourceMapHook + ";" + src;
}
exports.injectSourcemapHook = injectSourcemapHook;
function runTsDiagnostics(filePath, compilerOptions) {
    var program = tsc.createProgram([filePath], compilerOptions);
    var allDiagnostics = tsc.getPreEmitDiagnostics(program);
    var formattedDiagnostics = allDiagnostics.map(function (diagnostic) {
        if (diagnostic.file) {
            var _a = diagnostic.file.getLineAndCharacterOfPosition(diagnostic.start), line = _a.line, character = _a.character;
            var message = tsc.flattenDiagnosticMessageText(diagnostic.messageText, '\n');
            return path.relative(process.cwd(), diagnostic.file.fileName) + " (" + (line + 1) + "," + (character + 1) + "): " + message + "\n";
        }
        return "" + tsc.flattenDiagnosticMessageText(diagnostic.messageText, '\n');
    });
    if (formattedDiagnostics.length) {
        throw new Error(formattedDiagnostics.join(''));
    }
}
exports.runTsDiagnostics = runTsDiagnostics;
//# sourceMappingURL=data:application/json;base64,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