"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
var assert = require("assert");
var fs = require("fs");
var ini = require("ini");
var path_1 = require("path");
// 获取 gitinfo
var GitInfo = /** @class */ (function () {
    function GitInfo(args) {
        this.gitPath = args && args.gitPath;
        assert(this.gitPath, 'options.gitPath is required');
        assert(fs.existsSync(this.gitPath), this.gitPath + " should exist");
    }
    Object.defineProperty(GitInfo.prototype, "gitInfo", {
        get: function () {
            return {
                // git 地址
                repository: this.getRepository(),
                // 当前分支
                branch: this.getBranch(),
                // commitId
                commitId: this.getCommitId(),
                // 提交人信息
                userInfo: this.getUserInfo(),
            };
        },
        enumerable: false,
        configurable: true
    });
    // 获取 commitId
    GitInfo.prototype.getCommitId = function () {
        var headFileContent = this._getHeadFileContent();
        var match = headFileContent.match(/ref:\s(\S+$)/);
        var branchPathInfo = match ? match[1] : '';
        if (!branchPathInfo) {
            this._error('Get commit info error');
        }
        var branchInfoPath = path_1.join(this.gitPath, branchPathInfo);
        if (!fs.existsSync(branchInfoPath)) {
            this._error("Commit info file not exist: " + branchInfoPath);
        }
        return this._readFile(branchInfoPath);
    };
    // 获取当前分支
    GitInfo.prototype.getBranch = function () {
        var headFileContent = this._getHeadFileContent();
        var match = headFileContent.match(/refs\/heads\/(\S+$)/);
        var branchName = match && match[1];
        if (!branchName) {
            this._error('Get branch name not found');
        }
        return branchName || '';
    };
    // 获取当前 url
    GitInfo.prototype.getRepository = function () {
        var repository = '';
        try {
            var config = this._getConfigFileContent();
            repository = config['remote "origin"'].url;
        }
        catch (e) {
            this._error("Get project url error. " + e.message);
        }
        return repository;
    };
    // 获取 userInfo
    GitInfo.prototype.getUserInfo = function () {
        var commitId = this.getCommitId();
        var logFileContent = this._getLogFileContent();
        var regexp = new RegExp(commitId + "\\s(.*>)");
        var match = logFileContent.match(regexp);
        var userInfo = match && match[1];
        if (!userInfo) {
            this._error('Get user info error');
        }
        return userInfo || '';
    };
    // 解析日志
    GitInfo.prototype._getLogFileContent = function () {
        var headFileContent = this._getHeadFileContent();
        var match = headFileContent.match(/ref:\s(\S+$)/);
        var branchPathInfo = match ? match[1] : '';
        if (!branchPathInfo) {
            this._error('Get commit info error');
        }
        var logInfoPath = path_1.join(this.gitPath, 'logs', branchPathInfo);
        if (!fs.existsSync(logInfoPath)) {
            this._error("log info file not exist: " + logInfoPath);
        }
        return this._readFile(logInfoPath);
    };
    // 解析 config
    GitInfo.prototype._getConfigFileContent = function () {
        var configFilePath = path_1.join(this.gitPath, './config');
        if (!fs.existsSync(configFilePath)) {
            this._error("config info file not exist: " + configFilePath);
        }
        try {
            var config = ini.parse(this._readFile(configFilePath));
            return config;
        }
        catch (e) {
            this._error("Get project5 url error. " + e.message);
        }
    };
    // 获取 HEAD
    GitInfo.prototype._getHeadFileContent = function () {
        var headFilePath = path_1.join(this.gitPath, './HEAD');
        if (!fs.existsSync(headFilePath)) {
            this._error("HEAD file not exist: " + headFilePath);
        }
        return this._readFile(headFilePath);
    };
    GitInfo.prototype._readFile = function (filePath) {
        return fs.readFileSync(filePath, 'utf-8').trim();
    };
    GitInfo.prototype._error = function (message) {
        var err = new Error(message);
        err.name = 'GITINFO_ERROR';
        throw err;
    };
    return GitInfo;
}());
exports.default = GitInfo;
