"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
var tslib_1 = require("tslib");
var assert = require("assert");
var uuid_1 = require("uuid");
var debug = require("debug");
var path_1 = require("path");
var fs_1 = require("fs");
var utils_1 = require("./utils");
var satori_1 = require("./adapter/satori");
var log = debug('bigbrother-webpack-plugin:BigBrotherWebpackPlugin');
var BigBrotherWebpackPlugin = /** @class */ (function () {
    function BigBrotherWebpackPlugin(options) {
        assert(options, 'options is required');
        this.cwd = options.cwd;
        assert(this.cwd, 'options.cwd is required');
        this.tool = options.tool;
        assert(this.tool, 'options.tool is required');
        this.buildConfig = options.buildConfig || {};
        this.callback = options.callback || (function () { });
        try {
            this.satori = new satori_1.default();
        }
        catch (_) {
            log(_);
        }
    }
    BigBrotherWebpackPlugin.prototype.apply = function (compiler) {
        var _this = this;
        if (compiler.hooks) {
            compiler.hooks.done.tap('monitor', function (stats) {
                // 若有错误，不上传依赖树，需要延迟一秒（防止进程退出导致构建失败信息上报不了）
                if (stats && stats.hasErrors()) {
                    _this.simepleReport(stats);
                    return;
                }
                process.env.NODE_ENV === 'production'
                    ? _this.report(stats)
                    : _this.simepleReport(stats);
            });
            return;
        }
        // webpack 1
        compiler.plugin('done', function (stats) {
            _this.simepleReport(stats);
        });
    };
    BigBrotherWebpackPlugin.prototype.report = function (stats) {
        return tslib_1.__awaiter(this, void 0, void 0, function () {
            var data, fun, _1;
            return tslib_1.__generator(this, function (_a) {
                switch (_a.label) {
                    case 0:
                        _a.trys.push([0, 2, , 3]);
                        return [4 /*yield*/, this.getReportData(stats)];
                    case 1:
                        data = _a.sent();
                        fun = process.env.NODE_ENV === 'development' ? 'postDev' : 'postBuild';
                        // callback
                        this.callback(data, { stats: stats });
                        this.satori[fun](data, stats);
                        return [3 /*break*/, 3];
                    case 2:
                        _1 = _a.sent();
                        log(_1);
                        return [3 /*break*/, 3];
                    case 3: return [2 /*return*/];
                }
            });
        });
    };
    // 不需要 deps tree 的 report
    BigBrotherWebpackPlugin.prototype.simepleReport = function (stats) {
        var _a = process.versions, node = _a.node, _b = _a.alinode, alinode = _b === void 0 ? '' : _b;
        try {
            var data = {
                buildTimeRef: this.getUniqBuildId(),
                cwd: this.cwd,
                success: !stats.hasErrors(),
                startedAt: stats.startTime,
                endedAt: stats.endTime,
                tool: this.tool,
                pkg: utils_1.getPkg(this.cwd),
                buildConfig: this.buildConfig,
                artifact: utils_1.getArtifactInfo(stats),
                errors: stats.compilation.errors.slice(0, 2).map(function (_a) {
                    var name = _a.name, _b = _a.message, message = _b === void 0 ? '' : _b;
                    return ({
                        name: name,
                        message: message.substr(0, 200),
                    });
                }),
                stages: [],
                node: {
                    node: node,
                    alinode: alinode || '',
                },
                git: utils_1.getGitInfo(this.cwd),
                platform: utils_1.getPlatformInfo(),
                envs: process.env,
                hardware: utils_1.getHardware(),
                shameimaruUrl: '',
                xiaoshengStatsUrl: '',
            };
            // callback
            this.callback(data, { stats: stats, type: 'simple' });
            var fun = process.env.NODE_ENV === 'development' ? 'postDev' : 'postBuild'; // 若 NODE_ENV 为空，以 build 处理
            this.satori[fun](data, stats);
        }
        catch (_) {
            log(_);
        }
    };
    BigBrotherWebpackPlugin.prototype.getReportData = function (stats) {
        return tslib_1.__awaiter(this, void 0, void 0, function () {
            var _a, node, _b, alinode, data, _c, _d;
            return tslib_1.__generator(this, function (_e) {
                switch (_e.label) {
                    case 0:
                        _a = process.versions, node = _a.node, _b = _a.alinode, alinode = _b === void 0 ? '' : _b;
                        data = {
                            buildTimeRef: this.getUniqBuildId(),
                            cwd: this.cwd,
                            success: !stats.hasErrors(),
                            startedAt: stats.startTime,
                            endedAt: stats.endTime,
                            tool: this.tool,
                            pkg: utils_1.getPkg(this.cwd),
                            buildConfig: this.buildConfig,
                            artifact: utils_1.getArtifactInfo(stats),
                            errors: stats.compilation.errors.slice(0, 2).map(function (_a) {
                                var name = _a.name, _b = _a.message, message = _b === void 0 ? '' : _b;
                                return ({
                                    name: name,
                                    message: message.substr(0, 200),
                                });
                            }),
                            stages: [],
                            node: {
                                node: node,
                                alinode: alinode || '',
                            },
                            git: utils_1.getGitInfo(this.cwd),
                            platform: utils_1.getPlatformInfo(),
                            envs: process.env,
                            hardware: utils_1.getHardware(),
                            shameimaruUrl: '',
                            xiaoshengStatsUrl: '',
                        };
                        _c = data;
                        return [4 /*yield*/, this.getDepTreeOssUrl(data)];
                    case 1:
                        _c.shameimaruUrl = _e.sent();
                        _d = data;
                        return [4 /*yield*/, this.getStatsOssUrl()];
                    case 2:
                        _d.xiaoshengStatsUrl = _e.sent();
                        return [2 /*return*/, data];
                }
            });
        });
    };
    BigBrotherWebpackPlugin.prototype.getDepTreeOssUrl = function (data) {
        return tslib_1.__awaiter(this, void 0, void 0, function () {
            var depTreeOssUrl, depTree, _2;
            return tslib_1.__generator(this, function (_a) {
                switch (_a.label) {
                    case 0:
                        _a.trys.push([0, 4, , 5]);
                        if (!data.platform.type) return [3 /*break*/, 3];
                        return [4 /*yield*/, utils_1.getDepTree(this.cwd)];
                    case 1:
                        depTree = _a.sent();
                        return [4 /*yield*/, this.satori.uploadShameimaru(depTree)];
                    case 2:
                        depTreeOssUrl = _a.sent();
                        _a.label = 3;
                    case 3: return [3 /*break*/, 5];
                    case 4:
                        _2 = _a.sent();
                        log(_2);
                        depTreeOssUrl = '';
                        return [3 /*break*/, 5];
                    case 5: return [2 /*return*/, depTreeOssUrl];
                }
            });
        });
    };
    BigBrotherWebpackPlugin.prototype.getStatsOssUrl = function () {
        return tslib_1.__awaiter(this, void 0, void 0, function () {
            var statsOssUrl, statsFilePath, _3;
            return tslib_1.__generator(this, function (_a) {
                switch (_a.label) {
                    case 0:
                        statsOssUrl = '';
                        statsFilePath = path_1.join(process.env.HOME, 'dist', 'bundlestats.json');
                        if (!fs_1.existsSync(statsFilePath)) {
                            statsFilePath = path_1.join(process.env.HOME, 'build', 'bundlestats.json');
                        }
                        if (!fs_1.existsSync(statsFilePath)) {
                            return [2 /*return*/, statsOssUrl];
                        }
                        _a.label = 1;
                    case 1:
                        _a.trys.push([1, 3, , 4]);
                        return [4 /*yield*/, this.satori.uploadStatusFile(statsFilePath)];
                    case 2:
                        statsOssUrl = _a.sent();
                        return [3 /*break*/, 4];
                    case 3:
                        _3 = _a.sent();
                        log(_3);
                        statsOssUrl = '';
                        return [3 /*break*/, 4];
                    case 4: return [2 /*return*/, statsOssUrl];
                }
            });
        });
    };
    BigBrotherWebpackPlugin.prototype.getUniqBuildId = function () {
        return process.env.CLOUD_BUILD_ID || uuid_1.v4();
    };
    return BigBrotherWebpackPlugin;
}());
exports.default = BigBrotherWebpackPlugin;
