package com.bcxin.ins.task.base.config;

import com.bcxin.ins.common.constants.GlobalConstants;
import com.bcxin.ins.spring.util.SpringContextHolder;
import com.bcxin.ins.entity.common.ComTaskCron;
import com.bcxin.ins.util.toolbox.StrUtil;
import com.google.common.collect.Maps;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.scheduling.concurrent.ConcurrentTaskScheduler;
import org.springframework.scheduling.support.CronTrigger;
import org.springframework.scheduling.support.ScheduledMethodRunnable;

import java.util.Map;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.ScheduledFuture;
/**
 * <b> 定时任务配置 </b>
 * @author ZXF
 * @create 2018/10/16 0016 13:45
 * @version
 * @注意事项 </b>
 */
public class TaskConfig {

    /** logger日志. */
    private Logger logger = LoggerFactory.getLogger(TaskConfig.class);
    public static  Map<Long, ComTaskCron> TASKS = Maps.newConcurrentMap();
    public static  Map<Long, ScheduledFuture<?>> SCHEDULED_FUTURE = Maps.newConcurrentMap();

    private static TaskConfig taskConfig;
    private TaskConfig (){}
    public static synchronized TaskConfig getInstance() {
        if (taskConfig == null) {
            taskConfig = new TaskConfig();
        }
        return taskConfig;
    }

    /**
     * 多线程执行
     */
    public final ConcurrentTaskScheduler ct = new ConcurrentTaskScheduler(Executors.newScheduledThreadPool(StrUtil.toInteger(GlobalConstants.getConfig("task.pool"))));

    /**
     * 定时任务注册对象
     */
//    public ContextLifecycleScheduledTaskRegistrar contextLifecycleScheduledTaskRegistrar = SpringContextHolder.getBean(ContextLifecycleScheduledTaskRegistrar.class);

    /**
     * 启动一个计划任务.
     * @param task 当前进行的任务.
     */
    public void start(ComTaskCron task) {
        if (task.getCom_task_cron_id()==null) {
            throw new RuntimeException("任务ID不能为空");
        }
        if (StrUtil.isBlank(task.getCron_expression())) {
            throw new RuntimeException("任务的调度表达式不能为空.");
        }
        if(SCHEDULED_FUTURE.get(task.getCom_task_cron_id())!=null){
//            throw new RuntimeException("任务已启动无法重复操作.");
            return;
        }
        ScheduledFuture<?> scheduledFuture = null;
        try {
            scheduledFuture = ct.schedule(new ScheduledMethodRunnable(SpringContextHolder.getBean(task.getTarget_object().trim()),task.getTask_method()),
                    new CronTrigger(task.getCron_expression()));
            SCHEDULED_FUTURE.put(task.getCom_task_cron_id(), scheduledFuture);
            TASKS.put(task.getCom_task_cron_id(), task);
            logger.info("任务: " + task.getName() + "启动成功");
        } catch (NoSuchMethodException e) {
            logger.info("任务: " + task.getName() + "启动失败,没找到对应的方法");
        }
    }

    /**
     * 停止一个计划任务.
     * @param taskId 任务编号.
     */
    public void stop(Long taskId) {
        logger.info("正在停止任务 " + taskId);
        if (taskId==null) {
            throw new RuntimeException("任务ID不能为空");
        }

        try {
            ScheduledFuture<?> scheduledFuture = SCHEDULED_FUTURE.remove(taskId);
            TASKS.remove(taskId);
            if (scheduledFuture == null) {
//                throw new RuntimeException("任务ID:"+taskId+"不存在");
                return;
            } else {
                if (!scheduledFuture.isCancelled()) {
                    Thread.sleep(100);
                    /** false 表示当前任务若正在执行，则待其执行结束，再结束此任务. */
                    scheduledFuture.cancel(true);
                }
            }
        } catch (Exception e) {
            throw new RuntimeException("任务停止失败："+e.getMessage());
        }
    }

    /**
     * 重新设置当前任务的执行频率.
     *
     * @param taskId
     *            任务编号.
     */
    public void resetTrigger(Long taskId, String cronExpression) {
        logger.info("正在修改当前任务 " + taskId + "执行频率.");
        if (taskId ==null) {
            throw new RuntimeException("任务ID不能为空");
        }

        if (StrUtil.isBlank(cronExpression)) {
            throw new RuntimeException("任务的调度表达式不能为空.");
        }

        ComTaskCron task = TASKS.get(taskId);
        if (task != null) {
            if (cronExpression.equals(task.getCron_expression())) {
                return;
            }
            ScheduledFuture<?> scheduledFuture = SCHEDULED_FUTURE.remove(taskId);
            scheduledFuture.cancel(false);
            try {
                task.setCron_expression(cronExpression);
                scheduledFuture = ct.schedule(new ScheduledMethodRunnable(SpringContextHolder.getBean(task.getTarget_object().trim()).getClass(),task.getTask_method()),
                        new CronTrigger(task.getCron_expression()));
                SCHEDULED_FUTURE.put(taskId, scheduledFuture);
            } catch (NoSuchMethodException e) {
                logger.info("任务: " + task.getName() + "设置失败,没找到对应的方法");
            }
        }
    }

    /**
     * 仅执行一次.
     *
     * @param task 所要执行任务.
     */
    public void onlyOneTime(ComTaskCron task) {
        try {
            ct.execute(new ScheduledMethodRunnable(SpringContextHolder.getBean(task.getTarget_object().trim()).getClass(),task.getTask_method()), 2000);
        } catch (NoSuchMethodException e) {
            logger.info("任务: " + task.getName() + "启动失败,没找到对应的方法");
        }
    }

    /**
     * 销毁线程池中的任务.
     */
    public void destrory() {
        logger.info("正在终止自动任务的线程池资源.");
        ScheduledExecutorService scheduledExecutor = (ScheduledExecutorService) ct.getConcurrentExecutor();
        try {
            scheduledExecutor.shutdownNow();
        } catch (Exception e) {
            logger.info("自动任务的线程池资源清理发生异常.", e);
        } finally {
            logger.info("自动任务的线程池资源清理完成.");
        }
    }

}
