package com.bcxin.rest.web.apis.filters;

import java.io.IOException;
import java.nio.charset.StandardCharsets;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Map;

import javax.servlet.FilterChain;
import javax.servlet.ServletException;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

import org.springframework.boot.autoconfigure.http.HttpMessageConverters;
import org.springframework.core.Ordered;
import org.springframework.core.annotation.Order;
import org.springframework.http.HttpHeaders;
import org.springframework.http.HttpStatus;
import org.springframework.http.ResponseEntity;
import org.springframework.stereotype.Component;
import org.springframework.util.StringUtils;
import org.springframework.web.filter.OncePerRequestFilter;

import com.bcxin.Infrastructures.TenantContext;
import com.bcxin.Infrastructures.TenantUserContext;
import com.bcxin.Infrastructures.components.CacheProvider;
import com.bcxin.Infrastructures.components.JsonProvider;
import com.bcxin.Infrastructures.exceptions.BadTenantException;
import com.bcxin.Infrastructures.exceptions.TenantExceptionAbstract;
import com.bcxin.Infrastructures.exceptions.UnAuthorizedTenantException;
import com.bcxin.Infrastructures.utils.JwtUtil;
import com.bcxin.api.interfaces.tenants.UserRpcProvider;
import com.bcxin.api.interfaces.tenants.responses.UserOrganBasicGetResponse;
import com.bcxin.rest.web.apis.ResponseBuilder;
import com.bcxin.rest.web.apis.caches.UserOrganBasic;
import com.bcxin.rest.web.apis.utils.ServletRequestUtil;

@Component
@Order(Ordered.HIGHEST_PRECEDENCE-1)
public class AuthFilter extends OncePerRequestFilter {
    private final static Collection<String> excludedUrls = new ArrayList<>();
    static {
        excludedUrls.add("/identity/sign-in");
        excludedUrls.add("/tenant/organizations/registration");
        excludedUrls.add("/swagger-ui/");
        excludedUrls.add("/v2/api-docs");
        excludedUrls.add("/common/resources");
        excludedUrls.add("/identity/wechat-in");
        excludedUrls.add("/identity/prepare-reset-password");
        excludedUrls.add("/identity/reset-password");
        excludedUrls.add("/identity/wechat-in");
    }

    /**
     * todo 统一修复序列化方式
     */
    private final JsonProvider jsonProvider;
    private final UserRpcProvider userRpcProvider;

    public AuthFilter(JsonProvider jsonProvider, HttpMessageConverters converters,
                      CacheProvider cacheProvider, UserRpcProvider userRpcProvider) {
        this.jsonProvider = jsonProvider;
        this.userRpcProvider = userRpcProvider;
    }

    @Override
    protected void doFilterInternal(HttpServletRequest request, HttpServletResponse response, FilterChain filterChain) throws ServletException, IOException {
        String token = null;
        try {
            TenantUserContext userContext = TenantContext.getInstance().getUserContext();
            if(userContext.get()==null || !StringUtils.hasLength(userContext.get().getToken())) {
                throw new UnAuthorizedTenantException("当前无有效Authorization信息");
            }

            /**
             * 还是需要查询用户信息
             */
            this.doCheckSelectedOrganization(request);

            filterChain.doFilter(request, response);
        } catch (Exception ex) {
            HttpStatus status = HttpStatus.BAD_REQUEST;
            if (ex instanceof TenantExceptionAbstract) {
                status = HttpStatus.UNAUTHORIZED;
            }

            ResponseEntity responseEntity =
                    ResponseBuilder.build(status, String.format("无效Token(%s), 详细:%s", token, ex.getMessage()));
            response.setStatus(responseEntity.getStatusCode().value());

            String error = this.jsonProvider.getJson(responseEntity.getBody());
            response.setHeader(HttpHeaders.CONTENT_TYPE, "application/json");

            response.getOutputStream().write(error.getBytes(StandardCharsets.UTF_8));
            response.flushBuffer();
        }
    }

    @Override
    protected boolean shouldNotFilter(HttpServletRequest request) throws ServletException {
        /**
         * 重置当前用户上下文信息
         */
        TenantContext.getInstance().getUserContext().reset();

        try {
            String token = ServletRequestUtil.getBearerToken(request);
            String tenantUserId = JwtUtil.getUserIdFromToken(token);
            /**
             * 初始化当前token和对应的tenant user id
             */
            TenantContext.getInstance().getUserContext().init(token, tenantUserId);
        } catch (Exception ex) {
            //todo 获取token或者解析token失败
        	ex.printStackTrace();
        }

        String requestUri = request.getRequestURI();
        if (!StringUtils.hasLength(requestUri)) {
            return true;
        }

        boolean matchedResult =
                excludedUrls.stream().anyMatch(ii -> requestUri.startsWith(String.format("%s%s", request.getContextPath(), ii)));

        return matchedResult || requestUri.contains(String.format("%s/swagger", request.getContextPath())) || requestUri.contains("/download/resources");
    }

    private void doCheckSelectedOrganization(HttpServletRequest request) {
        String requestUri = request.getRequestURI();
        if (StringUtils.hasLength(requestUri) && requestUri.contains("organizations/") && !requestUri.contains("/approved") && !requestUri.contains("/organizations/search")) {
            String[] pathSections = requestUri.split("/");
            String selectedOrganId = null;
            for (int index = 0; index < pathSections.length; index++) {
                if ("organizations".equals(pathSections[index]) && (index + 1) < pathSections.length) {
                    selectedOrganId = pathSections[index + 1];
                    break;
                }
            }

            if (StringUtils.hasLength(selectedOrganId)) {
                String finalSelectedOrganId = selectedOrganId;
                String tenantUserId = TenantContext.getInstance().getUserContext().get().getId();

                //todo 修复
                UserOrganBasicGetResponse organBasicGetResponse = this.userRpcProvider.getByIdAndOrganId(
                        finalSelectedOrganId,
                        tenantUserId);

                if (organBasicGetResponse == null) {
                    throw new UnAuthorizedTenantException(String.format("当前用户(%s)不属于本公司(%s)或者已经办理离职手续!", tenantUserId, finalSelectedOrganId));
                }

                UserOrganBasic basic = UserOrganBasic.create(
                        organBasicGetResponse.getOrganizationId(),
                        organBasicGetResponse.getEmployeeId(),
                        organBasicGetResponse.getName(),
                        organBasicGetResponse.getMasterSlaveType(),
                        organBasicGetResponse.getAdditional());
                TenantContext.getInstance().getUserContext()
                        .assignDetail(basic.getName(), basic.getOrganizationId(), basic.getEmployeeId(), basic.getMasterSlaveType());
            }
        }
    }
}
