package com.bcxin.rest.web.apis.controllers;

import com.bcxin.Infrastructures.Pagination;
import com.bcxin.Infrastructures.components.ExcelProvider;
import com.bcxin.Infrastructures.components.WebFileProvider;
import com.bcxin.api.interfaces.commons.CommonImportResponse;
import com.bcxin.api.interfaces.tenants.EmployeeRpcProvider;
import com.bcxin.api.interfaces.tenants.criterias.EmployeeCriteria;
import com.bcxin.api.interfaces.tenants.criterias.EmployeeLeaveCriteria;
import com.bcxin.api.interfaces.tenants.requests.employees.*;
import com.bcxin.api.interfaces.tenants.responses.EmployeeDetailGetResponse;
import com.bcxin.api.interfaces.tenants.responses.EmployeeGetResponse;
import com.bcxin.api.interfaces.tenants.responses.EmployeeLeaveGetResponse;
import com.bcxin.rest.web.apis.ExtractDataComponent;
import com.bcxin.rest.web.apis.requests.BatchImportDataRequest;
import io.swagger.annotations.ApiOperation;
import org.springframework.http.ResponseEntity;
import org.springframework.util.CollectionUtils;
import org.springframework.util.StringUtils;
import org.springframework.web.bind.annotation.*;
import javax.servlet.http.HttpServletRequest;
import javax.validation.Valid;
import java.util.Collection;
import java.util.Collections;
import java.util.Map;
import java.util.stream.Collectors;

@RestController
@RequestMapping("/tenant/organizations/{organizationId}/employees")
public class EmployeeController extends ControllerAbstract {
    private final EmployeeRpcProvider employeeRpcProvide;
    private final ExtractDataComponent extractDataComponent;

    public EmployeeController(EmployeeRpcProvider employeeRpcProvide,
                              WebFileProvider webFileProvider,
                              ExcelProvider excelProvider, ExtractDataComponent extractDataComponent) {
        this.employeeRpcProvide = employeeRpcProvide;
        this.extractDataComponent = extractDataComponent;
    }

    @ApiOperation(value = "转移职员到其他部门", response = Void.class)
    @PostMapping("/department-movement")
    public ResponseEntity post_department_movement(@PathVariable String organizationId,
                                                   @RequestBody BatchMoveEmployeesRequest request) {
        this.employeeRpcProvide.batchMove(organizationId, request);

        return this.ok();
    }

    @ApiOperation(value = "新增企业职员信息", response = String.class)
    @PostMapping
    public ResponseEntity post_employees(@PathVariable String organizationId,
                                         @Valid @RequestBody EmployeeRequest request) {
        String employeeId = this.employeeRpcProvide.create(organizationId, request);

        return this.ok(employeeId);
    }

    @ApiOperation(value = "批量导入保安员", response = CommonImportResponse.class)
    @PostMapping("/batch-import")
    public ResponseEntity bathImport(
            HttpServletRequest servletRequest,
            @PathVariable String organizationId,
            @RequestBody BatchImportDataRequest request) {
        Collection<BatchImportEmployeeRequest> batchImportEmployeeRequests =
                extractDataComponent.extract(2, request.getPath(), excelMapValues -> {
                    return translate2BatchImportEmployeeRequests(excelMapValues);
                });

        CommonImportResponse response = this.employeeRpcProvide.batchImport(organizationId, request.getPath(), batchImportEmployeeRequests);

        response.buildDownloadUrl(
                String.format("%s/download/resources/%s/employee", servletRequest.getContextPath(), response.getResourceId())
        );

        return this.ok(response);
    }

    @ApiOperation(value = "批量离职", response = Void.class)
    @PostMapping("/batch-leave")
    public ResponseEntity batchLeave(@PathVariable String organizationId,
                                     @RequestBody BatchLeaveEmployeeRequest request) {
        this.employeeRpcProvide.batchLeave(organizationId, request);

        return this.ok();
    }

    @ApiOperation(value = "复职", response = Void.class)
    @PostMapping("/go-back")
    public ResponseEntity goBack(@PathVariable String organizationId,
                                 @RequestBody BackEmployeeRequest request) {
        this.employeeRpcProvide.back(organizationId, request);

        return this.ok();
    }

    @ApiOperation(value = "批量设置上级", response = Void.class)
    @PostMapping("/batch-assign-superior")
    public ResponseEntity batchAssignSuperior(@PathVariable String organizationId,
                                              @RequestBody BatchAssignEmployeeSuperiorRequest request) {
        this.employeeRpcProvide.batchAssignSuperiors(organizationId, request);

        return this.ok();
    }


    @ApiOperation(value = "获取已离职的职员列表")
    @PostMapping("/search-leave")
    public ResponseEntity get_leave_employees(@PathVariable String organizationId,
                                              @RequestBody EmployeeLeaveCriteria criteria) {
        Pagination<EmployeeLeaveGetResponse> data = this.employeeRpcProvide.findLeaves(organizationId, criteria);

        return this.ok(data);
    }


    @ApiOperation(value = "获取企业职员列表", response = EmployeeGetResponse.class)
    @PostMapping("/search")
    public ResponseEntity get_employees(@PathVariable String organizationId, @Valid @RequestBody EmployeeCriteria criteria) {
        Pagination<EmployeeGetResponse> data = this.employeeRpcProvide.find(organizationId, criteria);

        return this.ok(data);
    }

    @ApiOperation(value = "获取职员/用户的基本信息", response = EmployeeDetailGetResponse.class)
    @GetMapping("/{id}")
    public ResponseEntity<EmployeeDetailGetResponse> get(@PathVariable String organizationId, @PathVariable String id) {
        EmployeeDetailGetResponse detailGetResponse = this.employeeRpcProvide.getByOrganIdAndId(organizationId, id);

        return this.ok(detailGetResponse);
    }

    @ApiOperation(value = "修改职员信息", response = Void.class)
    @PutMapping("/{id}")
    public ResponseEntity put(@PathVariable String organizationId, @PathVariable String id, @RequestBody UpdateEmployeeRequest request) {
        this.employeeRpcProvide.update(organizationId, id, request);

        return this.ok();
    }

    /**
     * 转换为对象
     * 先跳过前两行
     *
     * @param excelMapValues
     * @return
     */
    private Collection<BatchImportEmployeeRequest> translate2BatchImportEmployeeRequests(
            Collection<Map<Integer, String>> excelMapValues) {
        if (CollectionUtils.isEmpty(excelMapValues)) {
            return Collections.emptyList();
        }

        Collection<BatchImportEmployeeRequest> requests = excelMapValues.parallelStream().map(ii -> {
            String name = StringUtils.trimWhitespace(ii.get(0));
            String telephone = StringUtils.trimWhitespace(ii.get(1));
            String departNames = StringUtils.trimWhitespace(ii.get(2));
            String occupationType = StringUtils.trimWhitespace(ii.get(3));
            String hiredDate = StringUtils.trimWhitespace(ii.get(4));
            String credentialType = StringUtils.trimWhitespace(ii.get(5));
            String credentialNumber = StringUtils.trimWhitespace(ii.get(6));


            if (StringUtils.hasLength(name) && StringUtils.hasLength(telephone)) {

                return BatchImportEmployeeRequest.create(name, telephone, departNames, occupationType, hiredDate, credentialType, credentialNumber);
            }

            return null;
        }).filter(ii -> ii != null).collect(Collectors.toList());

        return requests;
    }
}
