package com.bcxin.rest.web.apis.controllers;
import com.bcxin.api.interfaces.identities.IdentityRpcProvider;
import com.bcxin.api.interfaces.identities.requests.BindWechatRequest;
import com.bcxin.api.interfaces.tenants.EmployeeRpcProvider;
import com.bcxin.api.interfaces.tenants.UserRpcProvider;
import com.bcxin.api.interfaces.tenants.requests.tenantUsers.SyncTenantUserLocationRequest;
import com.bcxin.api.interfaces.tenants.requests.tenantUsers.SyncCidRequest;
import com.bcxin.api.interfaces.tenants.requests.tenantUsers.UpdateCredentialRequest;
import com.bcxin.api.interfaces.tenants.requests.tenantUsers.UpdateTenantPhotoRequest;
import com.bcxin.api.interfaces.tenants.responses.MyDepartImContactGetResponse;
import com.bcxin.api.interfaces.tenants.responses.MyOrganizationProfileGetResponse;
import com.bcxin.api.interfaces.tenants.responses.UserDetailGetResponse;
import com.bcxin.api.interfaces.tenants.responses.UserProfileGetResponse;
import com.bcxin.rest.web.apis.responses.ApiDepartmentTreeGetResponse;
import io.swagger.annotations.*;
import org.springframework.http.ResponseEntity;
import org.springframework.web.bind.annotation.*;

import javax.validation.Valid;
import java.util.Collection;
import java.util.stream.Collectors;

/**
 * 获取当前用户信息
 */
@Api("当前用户信息")
@RestController
@RequestMapping("/tenant/users/current")
public class CurrentUserController extends ControllerAbstract {
    private final UserRpcProvider userRpcProvider;
    private final EmployeeRpcProvider employeeRpcProvider;
    private final IdentityRpcProvider identityRpcProvider;

    public CurrentUserController(UserRpcProvider userRpcProvider,
                                 EmployeeRpcProvider employeeRpcProvider,
                                 IdentityRpcProvider identityRpcProvider) {
        this.userRpcProvider = userRpcProvider;
        this.employeeRpcProvider = employeeRpcProvider;
        this.identityRpcProvider = identityRpcProvider;
    }

    /**
     * @return
     */
    @ApiImplicitParams({
            @ApiImplicitParam(
                    name = "Authorization",
                    value = "Bearer 租户token",
                    paramType = "header",
                    dataType = "String"
            )
    })
    @ApiResponses({
            @ApiResponse(code = 200, message = "返回当前租户的基本信息",
                    response = UserProfileGetResponse.class,
            examples = @Example({
                    @ExampleProperty(mediaType = "id",value = "xxxx"),
                    @ExampleProperty(mediaType = "name",value = "张三"),
                    @ExampleProperty(mediaType = "lonLatJson",value = "{lat:11.222,lon:2323232}"),
                    @ExampleProperty(mediaType = "birthdate",value = "2021-01-01"),
                    @ExampleProperty(mediaType = "checkedStatus",value = "34323"),
                    @ExampleProperty(mediaType = "authenticateStatus",value = "34323"),
                    @ExampleProperty(mediaType = "authenticatedResult",value = "34323"),
                    @ExampleProperty(mediaType = "stature",value = "1.8"),
                    @ExampleProperty(mediaType = "userType",value = "xxx")
            })),
            @ApiResponse(code = 404, message = "找不到租户信息", response = Valid.class)
    })
    @ApiOperation("获取当前租户(不是公司的职员)的profile")
    @GetMapping("/profile")
    public ResponseEntity<UserProfileGetResponse> profile() {
        UserProfileGetResponse response =
                this.userRpcProvider.get(this.getCurrentUserId());

        return this.ok(response);
    }

    @GetMapping("/detail")
    public ResponseEntity<UserDetailGetResponse> detail() {
        UserDetailGetResponse response =
                this.userRpcProvider.getDetail(this.getCurrentUserId());

        return this.ok(response);
    }



    @ApiImplicitParams({
            @ApiImplicitParam(
                    name = "Authorization",
                    value = "Bearer 租户token",
                    paramType = "header",
                    dataType = "String"
            )
    })
    @ApiOperation("获取当前租户的组织列表")
    @ApiResponses({
            @ApiResponse(code = 200, message = "返回当前租户的组织列表", response = ApiDepartmentTreeGetResponse.class,responseContainer = "list"),
            @ApiResponse(code = 404, message = "找不到租户信息", response = Valid.class)
    })
    @GetMapping("/organizations")
    public ResponseEntity<Collection<MyOrganizationProfileGetResponse>> myOrgans() {
        Collection<MyOrganizationProfileGetResponse> responses =
                this.userRpcProvider.getById(this.getCurrentUserId());

        return this.ok(responses);
    }

    @GetMapping("/organizations/{organizationId}/contacts")
    public ResponseEntity myOrganContact(@PathVariable String organizationId) {
        Collection<MyDepartImContactGetResponse> responses =
                this.employeeRpcProvider.getMyImContacts(organizationId,this.getCurrentUserId());

        return this.ok(responses.stream().sorted((t1, t2) -> t2.getDisplayOrder() - t1.getDisplayOrder()).collect(Collectors.toList()));
    }

    @PutMapping("/sync-location")
    public ResponseEntity SyncLocation(@RequestBody SyncTenantUserLocationRequest request) {
        this.userRpcProvider.update(this.getCurrentUserId(), request);

        return this.ok();
    }

    @PutMapping("/sync-device")
    public ResponseEntity SyncLocation(@RequestBody SyncCidRequest request) {
        this.userRpcProvider.updateDeviceId(this.getCurrentUserId(), request.getCid());

        return this.ok();
    }

    @ApiImplicitParams({
            @ApiImplicitParam(
                    name = "Authorization",
                    value = "Bearer 租户token",
                    paramType = "header",
                    dataType = "String"
            )
    })
    @ApiOperation("上传证件信息")
    @ApiResponses({
            @ApiResponse(code = 200, message = "无", response = Valid.class),
            @ApiResponse(code = 404, message = "当前用户无效", response = Valid.class),
            @ApiResponse(code = 400, message = "如果有保安员的职业身份, 则必须上传身份证且大于18岁", response = Valid.class)
    })
    @PostMapping("/update-credential")
    public ResponseEntity UpdateCredential(@RequestBody UpdateCredentialRequest request) {
        this.userRpcProvider.updateCredential(this.getCurrentUserId(), request);

        return this.ok();
    }

    @ApiImplicitParams({
            @ApiImplicitParam(
                    name = "Authorization",
                    value = "Bearer 租户token",
                    paramType = "header",
                    dataType = "String"
            )
    })
    @ApiOperation("上传一寸免冠照片")
    @ApiResponses({
            @ApiResponse(code = 200, message = "无", response = Valid.class),
            @ApiResponse(code = 404, message = "当前用户无效", response = Valid.class)
    })
    @PostMapping("/upload-photo")
    public ResponseEntity updatePhotos(@RequestBody UpdateTenantPhotoRequest request) {
        this.userRpcProvider.updatePhoto(this.getCurrentUserId(), request);

        return this.ok();
    }

    @ApiImplicitParams({
            @ApiImplicitParam(
                    name = "Authorization",
                    value = "Bearer 租户token",
                    paramType = "header",
                    dataType = "String"
            )
    })
    @ApiOperation("绑定微信")
    @ApiResponses({
            @ApiResponse(code = 200, message = "无", response = Valid.class),
            @ApiResponse(code = 404, message = "当前用户无效", response = Valid.class)
    })
    @PostMapping("/wechat/binding")
    public ResponseEntity bindWechat(@RequestBody BindWechatRequest request) {
        this.identityRpcProvider.bindWechat(request);

        return this.ok();
    }

    @ApiImplicitParams({
            @ApiImplicitParam(
                    name = "Authorization",
                    value = "Bearer 租户token",
                    paramType = "header",
                    dataType = "String"
            )
    })
    @ApiOperation("解除绑定微信")
    @ApiResponses({
            @ApiResponse(code = 200, message = "无", response = Valid.class),
            @ApiResponse(code = 404, message = "当前用户无效", response = Valid.class)
    })
    @PostMapping("/wechat/unbinding")
    public ResponseEntity unbindWechat() {
        this.identityRpcProvider.unbindWechat(this.getCurrentUserId());

        return this.ok();
    }

    @ApiImplicitParams({
            @ApiImplicitParam(
                    name = "Authorization",
                    value = "Bearer 租户token",
                    paramType = "header",
                    dataType = "String"
            )
    })
    @ApiOperation("重新进行实名认证")
    @ApiResponses({
            @ApiResponse(code = 200, message = "无", response = Valid.class),
            @ApiResponse(code = 404, message = "当前用户无效", response = Valid.class)
    })
    @PostMapping("/re-real-name-authenticate")
    public ResponseEntity reRealNameAuthenticate() {
        this.userRpcProvider.reRealNameAuthenticate(this.getCurrentUserId());

        return this.ok();
    }
}
