package com.bcxin.rest.web.apis.controllers;

import com.bcxin.Infrastructures.Pagination;
import com.bcxin.api.interfaces.commons.CommonImportResponse;
import com.bcxin.api.interfaces.tenants.ContractRpcProvider;
import com.bcxin.api.interfaces.tenants.criterias.ContractCriteria;
import com.bcxin.api.interfaces.tenants.requests.contracts.BatchImportContractRequest;
import com.bcxin.api.interfaces.tenants.requests.contracts.CreateContractRequest;
import com.bcxin.api.interfaces.tenants.requests.contracts.UpdateContractRequest;
import com.bcxin.api.interfaces.tenants.responses.ContractGetResponse;
import com.bcxin.rest.web.apis.ExtractDataComponent;
import com.bcxin.rest.web.apis.requests.BatchImportDataRequest;
import org.springframework.http.ResponseEntity;
import org.springframework.util.CollectionUtils;
import org.springframework.util.StringUtils;
import org.springframework.web.bind.annotation.*;

import javax.servlet.http.HttpServletRequest;
import java.util.Collection;
import java.util.Collections;
import java.util.Map;
import java.util.stream.Collectors;

@RestController
@RequestMapping("/tenant/organizations/{organizationId}/contracts")
public class ContractController extends ControllerAbstract {

    private final ContractRpcProvider contractRpcProvider;
    private final ExtractDataComponent extractDataComponent;

    public ContractController(ContractRpcProvider contractRpcProvider,
                              ExtractDataComponent extractDataComponent) {
        this.contractRpcProvider = contractRpcProvider;
        this.extractDataComponent = extractDataComponent;
    }

    @PostMapping
    public ResponseEntity post(@PathVariable String organizationId, @RequestBody CreateContractRequest request) {
        this.contractRpcProvider.create(organizationId, request);

        return this.ok();
    }

    @PutMapping("/{id}")
    public ResponseEntity post(@PathVariable String organizationId, @PathVariable String id,
                               @RequestBody UpdateContractRequest request) {
        this.contractRpcProvider.update(organizationId, id, request);

        return this.ok();
    }

    @DeleteMapping("/{id}")
    public ResponseEntity delete(@PathVariable String organizationId, @PathVariable String id) {
        this.contractRpcProvider.delete(organizationId, id);

        return this.ok();
    }

    @PostMapping("/search")
    public ResponseEntity search(@PathVariable String organizationId, @RequestBody ContractCriteria criteria) {
        Pagination<ContractGetResponse> contractGetResponsePageable =
                this.contractRpcProvider.find(organizationId, criteria);

        return this.ok(contractGetResponsePageable);
    }

    @PostMapping("/batch-import")
    public ResponseEntity batchImport(
            HttpServletRequest servletRequest,
            @PathVariable String organizationId,
                                      @RequestBody BatchImportDataRequest request ) {
        Collection<BatchImportContractRequest> batchImportContractRequests =
                extractDataComponent.extract(2, request.getPath(), excelMapValues -> {
                    return translate2BatchImportContractRequests(excelMapValues);
                });

        CommonImportResponse response =
                this.contractRpcProvider.batchImport(organizationId, request.getPath(), batchImportContractRequests);

        response.buildDownloadUrl(
                String.format("%s/download/resources/%s/contract", servletRequest.getContextPath(), response.getResourceId())
        );

        return this.ok(response);
    }

    /**
     * 转换为对象
     * 先跳过前两行
     *
     * @param excelMapValues
     * @return
     */
    private Collection<BatchImportContractRequest> translate2BatchImportContractRequests(
            Collection<Map<Integer, String>> excelMapValues) {
        if (CollectionUtils.isEmpty(excelMapValues)) {
            return Collections.emptyList();
        }

        Collection<BatchImportContractRequest> requests = excelMapValues.parallelStream().map(ii -> {
            String name = StringUtils.trimWhitespace(ii.get(0));
            String aName = StringUtils.trimWhitespace(ii.get(1));
            String bName = StringUtils.trimWhitespace(ii.get(2));
            String telephone = StringUtils.trimWhitespace(ii.get(3));
            String beginDate = StringUtils.trimWhitespace(ii.get(4));
            String endDate = StringUtils.trimWhitespace(ii.get(5));


            if (StringUtils.hasLength(name) && StringUtils.hasLength(telephone)) {

                return BatchImportContractRequest.create(name, aName, bName, telephone, beginDate, endDate);
            }

            return null;
        }).filter(ii -> ii != null).collect(Collectors.toList());

        return requests;
    }
}
