package com.bcxin.auth.system.util;

import org.apache.http.HttpEntity;
import org.apache.http.client.methods.CloseableHttpResponse;
import org.apache.http.client.methods.HttpGet;
import org.apache.http.impl.client.CloseableHttpClient;
import org.apache.http.impl.client.HttpClients;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.*;
import java.net.HttpURLConnection;
import java.net.URL;
import java.nio.file.Files;
import java.nio.file.StandardCopyOption;

/**
 * 文件处理类
 *
 * @author Ethan_Huang
 */
public class FileUtil {

	Logger logger = LoggerFactory.getLogger(FileUtil.class);

	/**
	 * 下载远程文件到本地
	 * @param remoteFilePath 文件路径
	 * @param localFilePath 保存本地路径
	 */
	public static void downloadFile(String remoteFilePath, String localFilePath) throws Exception{
		URL urlfile = null;
		HttpURLConnection httpUrl = null;
		BufferedInputStream bis = null;
		BufferedOutputStream bos = null;
		File f = new File(localFilePath);
		try{
			if(!f.getParentFile().exists()){
				f.getParentFile().mkdirs();
			}
			urlfile = new URL(remoteFilePath);
			httpUrl = (HttpURLConnection)urlfile.openConnection();
			httpUrl.connect();
			bis = new BufferedInputStream(httpUrl.getInputStream());
			bos = new BufferedOutputStream(new FileOutputStream(f));
			int len = 2048;
			byte[] b = new byte[len];
			while ((len = bis.read(b)) != -1){
				bos.write(b, 0, len);
			}
			bos.flush();
			bis.close();
			httpUrl.disconnect();
		}finally{
			try{
				bis.close();
				bos.close();
			}catch (IOException e){
				e.printStackTrace();
			}
		}
	}

	public static void downloadFile1(String remoteFilePath, String localFilePath) throws Exception {
		HttpURLConnection connection = null;
		InputStream inputStream = null;
		OutputStream outputStream = null;

		try {
			// 确保目录存在
			File targetFile = new File(localFilePath);
			File parentDir = targetFile.getParentFile();
			if (parentDir != null && !parentDir.exists()) {
				parentDir.mkdirs();
			}

			// 创建临时文件（避免部分下载导致文件损坏）
			File tempFile = File.createTempFile("download_", ".tmp", parentDir);

			URL url = new URL(remoteFilePath);
			connection = (HttpURLConnection) url.openConnection();

			// 设置避免内容编码修改（关键！）
			connection.setRequestProperty("Accept-Encoding", "identity");

			// 获取输入流（不进行缓冲，避免内存修改）
			inputStream = connection.getInputStream();

			// 使用NIO直接传输（避免缓冲修改）
			Files.copy(inputStream, tempFile.toPath(), StandardCopyOption.REPLACE_EXISTING);

			// 验证文件完整性（可选）
			long contentLength = connection.getContentLengthLong();
			if (contentLength > 0 && tempFile.length() != contentLength) {
				throw new IOException("下载文件不完整: " + tempFile.length() + "/" + contentLength + " bytes");
			}

			// 原子操作：重命名临时文件为目标文件
			Files.move(tempFile.toPath(), targetFile.toPath(), StandardCopyOption.REPLACE_EXISTING);

		} finally {
			// 安全关闭资源
			if (inputStream != null) {
				try { inputStream.close(); } catch (IOException e) {}
			}
			if (outputStream != null) {
				try { outputStream.close(); } catch (IOException e) {}
			}
			if (connection != null) {
				connection.disconnect();
			}
		}
	}

	public static void downloadWithHttpClient(String url, String localPath) throws IOException {
		try (CloseableHttpClient client = HttpClients.createDefault()) {
			HttpGet request = new HttpGet(url);
			try (CloseableHttpResponse response = client.execute(request)) {
				HttpEntity entity = response.getEntity();
				if (entity != null) {
					File target = new File(localPath);
					try (InputStream in = entity.getContent()) {
						Files.copy(in, target.toPath(), StandardCopyOption.REPLACE_EXISTING);
					}
				}
			}
		}
	}
}