package com.bcxin.ars.timer.dataSync;

import com.abcxin.smart.validator.annotation.DataSyncOutAnnotation;
import com.bcxin.ars.model.Config;
import com.bcxin.ars.model.sys.JobRunLog;
import com.bcxin.ars.service.sys.JobRunLogService;
import com.bcxin.ars.service.util.ConfigUtils;
import com.bcxin.ars.service.util.DataSyncContantsUtil;
import com.bcxin.ars.service.util.DataSyncUtil;
import com.bcxin.ars.util.Constants;
import com.bcxin.ars.util.DateUtil;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import java.io.File;
import java.util.Date;
import java.util.List;

/**
 * 内外网数据同步内到外
 * 需要同步的表在service实现类加上DataSyncAnnotation注解
 * service实现类中
 *  读取同步数据文件实现saveOrUpdateForDS或updateFlagForOutToIn
 *  生成同步数据文件实现searchForDataSynchronization
 * @author linqinglin
 * @date 2019/04/03 14:15
 */
@Service
public class DataSyncI2OService {

    private Logger logger = LoggerFactory.getLogger(DataSyncI2OService.class);

    @Autowired
    private ConfigUtils configUtils;

    @Autowired
    private DataSyncUtil dataSyncUtil;

    @Autowired
    private DataSyncContantsUtil dataSyncContantsUtil;

    @Autowired
    private JobRunLogService jobRunLogService;

    /***
     *  增量数据
     */
    private boolean lock = false;
    public void run() {
        if(lock){
            return;
        }
        lock = true;

        try {

            if (Constants.TIMEFLAG_VALUE.equals(configUtils.timeFlag)) {
                if (!Constants.DATASYNC_SWITCH_1.equals(configUtils.getValueByKey(Constants.DATASYNC_SWITCH))) {
                    return;
                }
                JobRunLog log = new JobRunLog();
                log.setCreateTime(new Date());
                log.setActive(true);
                log.setUpdateBy("jobSystem");
                //开始时间
                long startTime = System.currentTimeMillis();
                try {
                    if (configUtils.isIntranet()) {
                        createDataFile();
                    } else {
                        //外网读取同步文件
                        this.readDSFile(DataSyncOutAnnotation.class, dataSyncContantsUtil.ftpInFolder);
                    }
                }catch (Exception e){
                    logger.error(e.getMessage(),e);
                    log.setExceptionMsg(e.toString());
                }
                //日志信息
                long endTime = System.currentTimeMillis();
                log.setUpdateTime(new Date());
                log.setJobName(DataSyncI2OService.class.getName());
                log.setRunTime(new Date());
                log.setRunTimeLength((endTime - startTime) + "ms");
                jobRunLogService.insert(log);
            }
        }finally {
            lock = false;
        }
    }


    /**
     * 内网生成数据同步文件
     */
    private void createDataFile(){
        //内到外生成
        /***
         * 最后同步日期
         */
        String startDate = DateUtil.getYesterday();
        Config lastDateConfig = configUtils.getConfigByKey(Constants.LASTDATETIME);
        //配置为空或者等于当天日期
        if(lastDateConfig == null || DateUtil.convertDateToString(new Date(), DateUtil.FORMAT2).equals(lastDateConfig.getValue())){
        }else{
            startDate = lastDateConfig.getValue();
        }
        String dirDate = DateUtil.getCurrentDate();
        String filePath = dataSyncContantsUtil.ftpInFolder + dirDate + File.separatorChar;
        dataSyncUtil.createDSFile(startDate,filePath);

        /**
         * 更新"内到外最后同步日期" start
         */
        Config updateConfig = new Config();
        updateConfig.setKey(Constants.LASTDATETIME);
        updateConfig.setValue(DateUtil.getDateStrAfterMinute(-5));
        updateConfig.setUpdateTime(new Date());
        configUtils.updateForKey(updateConfig);

        /**
         * 更新"内到外最后同步日期" end
         */
    }

    /**
     * 外网读取同步数据文件
     */
    private void readDSFile(Class annotationClass, String filePath) {
        Date now = new Date();
        /***
         * 最后同步日期
         */
        Config lastDateConfig = configUtils.getConfigByKey(Constants.DATASYNC_LASTDATE);
        if (lastDateConfig != null) {
            //读取的时间为最后更新的时间
            String lastDate = lastDateConfig.getValue();
            List<Date> dates = DateUtil.getBetweenDates(DateUtil.convertStringToDate(lastDate), now);
            //加入当前日期
            dates.add(DateUtil.getBeginDayOfYesterday());
            dates.add(now);
            for (Date date : dates) {
                String dirFilePath = filePath + DateUtil.convertDateToString(date, DateUtil.FORMAT2);
                File dirFile = new File(dirFilePath);
                if (!dirFile.exists()) {
                    break;
                }
                File[] files = dirFile.listFiles();
                //读取是否有错误
                boolean errorFlag = false;
                for (int i = 0; i < files.length; i++) {
                    File file = files[i];
                    if(file.getName().endsWith(Constants.ZIP) && file.getName().startsWith(configUtils.getDataSyncStartFileName(DataSyncOutAnnotation.class))){
                        try {
                            dataSyncUtil.readDSFile(annotationClass,file);
                        } catch (Exception e) {
                            logger.error(e.getMessage(), e);
                            errorFlag = true;
                        }
                    }
                }

                //如果文件没有读取错误
                if (!errorFlag) {
                    Config updateConfig = new Config();
                    updateConfig.setKey(Constants.DATASYNC_LASTDATE);
                    updateConfig.setValue(DateUtil.convertDateToString(date,DateUtil.FORMAT2));
                    configUtils.updateForKey(updateConfig);
                }

            }
        }
    }
}
