package com.bcxin.backend.jobs;

import com.bcxin.backend.certificateSignatures.CertificateSignatureProvider;
import com.bcxin.backend.documentSignatrures.SignatureConfigProperty;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.boot.autoconfigure.condition.ConditionalOnProperty;
import org.springframework.scheduling.annotation.Scheduled;
import org.springframework.stereotype.Component;
import org.springframework.util.CollectionUtils;

import java.util.Collection;

/**
 * 资格认证签章功能定时任务
 */
@Component
@ConditionalOnProperty(prefix = "myapps.signature",name = "enable",havingValue = "true")
public class CertificateSignatureScheduleJob extends ScheduleJobAbstract {
    private final static Logger logger = LoggerFactory.getLogger(CertificateSignatureScheduleJob.class);
    private final SignatureConfigProperty signatureConfigProperty;

    public CertificateSignatureScheduleJob(SignatureConfigProperty signatureConfigProperty) {
        this.signatureConfigProperty = signatureConfigProperty;
    }

    /**
     * 步骤一、将待签章数据放入队列
     */
    @Scheduled(fixedRate = 1 * 60 * 1000)
    public void step1_prepareData() {
        Collection<CertificateSignatureProvider> certificateSignatureProviders
                = this.getBeans(signatureConfigProperty.getSignatureAreaCodes(),
                CertificateSignatureProvider.class
        );
        if (CollectionUtils.isEmpty(certificateSignatureProviders)) {
            logger.error("找不到符合实现:{}", signatureConfigProperty.getSignatureAreaCodes());
            return;
        }

        certificateSignatureProviders.parallelStream().forEach(certificateSignatureProvider -> {
            certificateSignatureProvider.step1_prepareData();
        });
    }


    /**
     * 步骤二、生成待签章的PDF
     */
    @Scheduled(cron = "${myapps.signature.create_pdf_schedule}") // 测试环境5分钟一次
    public void step2_generatePdf() {
        Collection<CertificateSignatureProvider> certificateSignatureProviders
                = this.getBeans(signatureConfigProperty.getSignatureAreaCodes(),
                CertificateSignatureProvider.class
        );
        if (CollectionUtils.isEmpty(certificateSignatureProviders)) {
            logger.error("找不到符合实现:{}", signatureConfigProperty.getSignatureAreaCodes());
            return;
        }

        certificateSignatureProviders.parallelStream().forEach(certificateSignatureProvider -> {
            certificateSignatureProvider.step2_generatePdf();
        });
    }

    /**
     * 步骤三、拷贝PDF文件到IN目录并执行签章操作; 对应于签章的
     * convertDto.getTempUrl().replace("/temp", "/in") 与 json.put("in", "/data/share/in"); 这边在服务器上是有映射关系的.
     * 签章后的目录为: json.put("out", "/data/share/out"); 对应于我们的convertDto.getTempUrl().replace("/temp", "/out")
     * <p>
     * 其中: convertDto.getTempUrl() 为rootPath + /uploads/pdf/dianziqianzhang/temp
     */
    @Scheduled(cron = "${myapps.signature.cgi_schedule}") // 测试环境3分钟一次
    public void step3_doSignature() {
        Collection<CertificateSignatureProvider> certificateSignatureProviders
                = this.getBeans(signatureConfigProperty.getSignatureAreaCodes(),
                CertificateSignatureProvider.class
        );
        if (CollectionUtils.isEmpty(certificateSignatureProviders)) {
            logger.error("找不到符合实现:{}", signatureConfigProperty.getSignatureAreaCodes());
            return;
        }

        certificateSignatureProviders.parallelStream().forEach(certificateSignatureProvider -> {
            certificateSignatureProvider.step3_doSignature();
        });
    }


    /**
     * <b> 步骤四: 针对签章完成的操作; 扫描解析out目录的文件, 复制到业务目录并更改签章状态 </b>
     */
    @Scheduled(cron = "${myapps.signature.move_schedule}") // 测试环境3分钟一次
    public void step4_doSignature() {
        Collection<CertificateSignatureProvider> certificateSignatureProviders
                = this.getBeans(signatureConfigProperty.getSignatureAreaCodes(),
                CertificateSignatureProvider.class
        );
        if (CollectionUtils.isEmpty(certificateSignatureProviders)) {
            logger.error("找不到符合实现:{}", signatureConfigProperty.getSignatureAreaCodes());
            return;
        }

        certificateSignatureProviders.parallelStream().forEach(certificateSignatureProvider -> {
            certificateSignatureProvider.step4_moveOutDocument2BusinessDirAndFinishSignature();
        });
    }

    @Scheduled(cron = "${myapps.signature.change_schedule}") // 测试环境1小时一次
    public void step5_doneForExceptionData() {
        Collection<CertificateSignatureProvider> certificateSignatureProviders
                = this.getBeans(signatureConfigProperty.getSignatureAreaCodes(),
                CertificateSignatureProvider.class
        );
        if (CollectionUtils.isEmpty(certificateSignatureProviders)) {
            logger.error("找不到符合实现:{}", signatureConfigProperty.getSignatureAreaCodes());
            return;
        }

        certificateSignatureProviders.parallelStream().forEach(certificateSignatureProvider -> {
            certificateSignatureProvider.step5_doneForExceptionData();
        });
    }
}
