package com.bcxin.tenant.open.backend.tasks.configs;

import com.bcxin.tenant.open.infrastructures.components.JsonProvider;
import com.bcxin.tenant.open.infrastructures.enums.PoliceEventCategory;
import com.bcxin.tenant.open.infrastructures.utils.ExceptionUtil;
import com.bcxin.tenant.open.infrastructures.utils.GisPointUtil;
import com.bcxin.tenant.open.infrastructures.utils.StringUtil;
import com.bcxin.tenant.open.infrastructures.valueTypes.GeoLocationValueType;
import com.bcxin.tenant.open.infrastructures.valueTypes.LonLatValueType;
import com.bcxin.tenant.open.infrastructures.constants.KafkaConstants;
import com.bcxin.tenant.open.jdks.*;
import com.bcxin.tenant.open.jdks.requests.*;
import com.bcxin.tenant.open.jdks.requests.enums.BroadcastMessageType;
import com.bcxin.tenant.open.jdks.responses.CompanyDetailResponse;
import com.bcxin.tenant.open.jdks.responses.EmployeeOverviewResponse;
import com.bcxin.tenant.open.jdks.responses.SearchNearByCompanyResponse;
import org.apache.http.client.methods.CloseableHttpResponse;
import org.apache.http.client.methods.HttpPost;
import org.apache.http.entity.StringEntity;
import org.apache.http.impl.client.CloseableHttpClient;
import org.apache.http.util.EntityUtils;
import org.apache.kafka.clients.consumer.ConsumerRecord;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.kafka.annotation.KafkaListener;
import org.springframework.kafka.support.Acknowledgment;
import org.springframework.stereotype.Component;
import org.springframework.util.CollectionUtils;

import java.util.Collection;
import java.util.Comparator;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

/**
 * 调度指挥的业务功能
 * 1, 警情上报预警
 * 2, 电子围栏预警
 */
@Component
public class KafkaComponent_Dispatch extends KafkaComponentAbstract {
    private static Logger logger = LoggerFactory.getLogger(KafkaComponent_Dispatch.class);

    private final SocketPublishRpcProvider socketPublishRpcProvider;
    private final PoliceIncidentsRpcProvider policeIncidentsRpcProvider;
    private final RdSecurityStationRailWriterRpcProvider securityStationRailWriterRpcProvider;
    private final JsonProvider jsonProvider;
    private final EmployeeReaderRpcProvider employeeReaderRpcProvider;
    private final CompanyReaderRpcProvider companyReaderRpcProvider;
    private final SecurityStationReaderRpcProvider securityStationReaderRpcProvider;
        private final AttendanceRpcProvider attendanceRpcProvider;
    private final CloseableHttpClient httpClient;

    public KafkaComponent_Dispatch(SocketPublishRpcProvider socketPublishRpcProvider,
                                   PoliceIncidentsRpcProvider policeIncidentsRpcProvider,
                                   RdSecurityStationRailWriterRpcProvider securityStationRailWriterRpcProvider,
                                   JsonProvider jsonProvider,
                                   EmployeeReaderRpcProvider employeeReaderRpcProvider,
                                   CompanyReaderRpcProvider companyReaderRpcProvider,
                                   SecurityStationReaderRpcProvider securityStationReaderRpcProvider,
                                   AttendanceRpcProvider attendanceRpcProvider,
                                   DispatchDataScopeRpcProvider dispatchDataScopeRpcProvider,
                                   RdSyncRpcWriterProvider syncRpcWriterProvider,
                                   CloseableHttpClient httpClient) {
        super(jsonProvider, securityStationReaderRpcProvider, dispatchDataScopeRpcProvider, syncRpcWriterProvider);
        this.socketPublishRpcProvider = socketPublishRpcProvider;
        this.policeIncidentsRpcProvider = policeIncidentsRpcProvider;
        this.securityStationRailWriterRpcProvider = securityStationRailWriterRpcProvider;
        this.jsonProvider = jsonProvider;
        this.employeeReaderRpcProvider = employeeReaderRpcProvider;
        this.companyReaderRpcProvider = companyReaderRpcProvider;
        this.securityStationReaderRpcProvider = securityStationReaderRpcProvider;
        this.attendanceRpcProvider = attendanceRpcProvider;
        this.httpClient = httpClient;
    }

    /**
     * 消费Sos警情上报数据
     * @param records
     * @param ack
     */
    @KafkaListener(
            id = "${spring.kafka.consumer.group-id}-dispatch",
            topics = {
                    KafkaConstants.TOPIC_NEW_POLICE_INCIDENT_REQUEST
            }, groupId = "${spring.kafka.consumer.group-id}-relative-changes")
    public void ackPoliceIncidents(List<ConsumerRecord<String, String>> records,
                                   Acknowledgment ack) {
        if (CollectionUtils.isEmpty(records)) {
            logger.warn("无效订阅警情数据:{}", records);
            return;
        }

        boolean allowed2CommitAtFinial = true;
        try {
            logger.error("ackPoliceIncidents 处理警情数据:{}",records.stream().map(ii->ii.value()).collect(Collectors.joining(",")));
            Collection<CreateSosPoliceIncidentQueueRequest> policeIncidentData
                    = records.stream().map(ii -> {
                CreateSosPoliceIncidentQueueRequest qr =
                        this.jsonProvider.toObject(
                                CreateSosPoliceIncidentQueueRequest.class, ii.value()
                        );
                return qr;
            }).collect(Collectors.toList());

            Collection<LonLatValueType> locations =
                    policeIncidentData.stream().map(ii -> {
                        LonLatValueType location = new LonLatValueType();
                        location.setLat(ii.getLatitude());
                        location.setLon(ii.getLongitude());

                        return location;
                    }).collect(Collectors.toUnmodifiableList());


            SearchNearByCompanyRequest request =
                    SearchNearByCompanyRequest.createForSecurityDepart(locations);
            Collection<SearchNearByCompanyResponse> nearResponses =
                    this.companyReaderRpcProvider.search(request);
            Collection<CreateSpecialSosRequest> specialSosRequests =
                    policeIncidentData.stream().map(ii -> {
                        CreateSpecialSosRequest sosRequest =
                                CreateSpecialSosRequest.create(
                                        ii.getId(),
                                        ii.getTenantUserId(),
                                        ii.getTenantEmployeeId(),
                                        ii.getOrganizationId(),
                                        ii.getSecurityStationId(),
                                        ii.getSecurityStationName(),
                                        ii.getLatitude(),
                                        ii.getLongitude(),
                                        ii.getAddress(),
                                        ii.getTenantUserName(),
                                        ii.getTenantUserIdCard(),
                                        ii.getStationAddress(),
                                        ii.getOrganizationName(),
                                        ii.getCreatedTime(),
                                        ii.getResourceType(),
                                        ii.getProjectId(),
                                        ii.getProjectName()
                                );

                        fillSuperviseData(ii, nearResponses, sosRequest);
                        return sosRequest;
                    }).collect(Collectors.toUnmodifiableList());
            this.policeIncidentsRpcProvider.create(specialSosRequests);
            Collection<BroadcastPoliceIndicentMessageContentItem> messageContents =
                    specialSosRequests.stream()
                            .map(ix -> {
                                return BroadcastPoliceIndicentMessageContentItem.create(
                                        String.valueOf(ix.getId()),
                                        PoliceEventCategory.Sos,
                                        ix.getTenantEmployeeId(),
                                        ix.getTenantUserName(),
                                        ix.getAddress(),
                                        ix.getSuperviseDepartId(),
                                        ix.getCreatedTime(),
                                        ix.getLatitude(),
                                        ix.getLongitude()
                                );
                            })
                            .sorted(Comparator.comparing(BroadcastPoliceIndicentMessageContentItem::getCreatedTime).reversed())
                            .collect(Collectors.toList());
            doBroadcastMessages(messageContents);
        } catch (Exception ex) {
            logger.error("消费警情上报数据发生异常", ex);
            allowed2CommitAtFinial = false;
        } finally {
            if (allowed2CommitAtFinial) {
                ack.acknowledge();
            }
        }
    }

    private void doBroadcastMessages(Collection<BroadcastPoliceIndicentMessageContentItem> messageContents) {
        if(CollectionUtils.isEmpty(messageContents)) {
            return;
        }

        try {
            Collection<String> empIds
                    = messageContents.stream().map(ii -> ii.getEmployeeId()).distinct()
                    .collect(Collectors.toList());

            Collection<EmployeeOverviewResponse> employeeOverviewResponses =
                    this.employeeReaderRpcProvider.getEmployeeOverviewByIds(empIds);
            /**
             * 填充人的设备信息
             */
            for (BroadcastPoliceIndicentMessageContentItem contentItem : messageContents) {
                EmployeeOverviewResponse employeeOverviewResponse
                        = employeeOverviewResponses.stream().filter(ii -> ii.getId().equalsIgnoreCase(contentItem.getEmployeeId()))
                        .findFirst().orElse(null);
                if (employeeOverviewResponse != null) {
                    contentItem.assignTencentUserInfo(
                            employeeOverviewResponse.getTencentUserId(),
                            employeeOverviewResponse.getSecurityStationId(),
                            employeeOverviewResponse.getSecurityStationName()
                    );
                }
            }

            this.socketPublishRpcProvider.dispatch(
                    BroadcastMessageRequest.create(BroadcastMessageType.PoliceIncidents,
                            BroadcastPoliceIndicentMessageContent.create(messageContents))
            );
        } catch (Exception ex) {
            logger.error("错误-广播消息(涉及的警情:{})到客户端socket:{}",
                    messageContents.stream().map(ii -> String.format("id=%s;message=%s",
                            ii.getId(), ii.getEmpName())).collect(Collectors.joining(";")),
                    ExceptionUtil.getStackMessage(ex));
        }
        finally {
            logger.error("推送的广播消息:{}", this.jsonProvider.getJson(messageContents));
        }
    }

    @KafkaListener(
            id = "${spring.kafka.consumer.group-id}-dispatch-location",
            topics = {
                    KafkaConstants.TOPIC_EMPLOYEE_SYNC_LOCATION
            }, groupId = "${spring.kafka.consumer.group-id}-sync-location")
    public void ackSyncEmployeeLocation(List<ConsumerRecord<String, String>> records,
                                        Acknowledgment ack) {
        if (CollectionUtils.isEmpty(records)) {
            logger.warn("无效订阅警情数据:{}", records);
            return;
        }

        boolean allowed2CommitAtFinial = true;
        try {
            Collection<SyncUpdateGeoRequest> syncUpdateGeoRequests
                    = records.stream()
                    .map(ix -> {
                        SyncUpdateGeoRequest data =
                                this.jsonProvider.toObject(SyncUpdateGeoRequest.class, ix.value());
                        return data;
                    })
                    .filter(ix -> ix != null)
                    .collect(Collectors.toList());

            if (!syncUpdateGeoRequests.isEmpty()) {
                securityStationRailWriterRpcProvider.doCheck(
                        syncUpdateGeoRequests
                );
            }
        } catch (Exception ex) {
            logger.error("消费警情上报数据发生异常", ex);
            allowed2CommitAtFinial = false;
        } finally {

            if (allowed2CommitAtFinial) {
                ack.acknowledge();
            }
        }
    }

    @KafkaListener(
            id = "${spring.kafka.consumer.group-id}-dispatch-attendance-v5",
            topics = {
                    KafkaConstants.TOPIC_DISPATCH_ATTENDANCE
            }, groupId = "${spring.kafka.consumer.group-id}-dispatch-attendance-v5",concurrency = "6")
    public void ackSyncTenantAttendance(List<ConsumerRecord<String, String>> records,
                                        Acknowledgment ack) {
        if (CollectionUtils.isEmpty(records)) {
            logger.warn("无效的签到数据:{}", records);
            return;
        }

        boolean allowed2CommitAtFinial = true;
        try {
            Collection<String> data =
                    records.stream().map(ii -> ii.value()).collect(Collectors.toList());
            this.attendanceRpcProvider.batchCommit(data);
        } catch (Exception ex) {
            logger.error("无效的签到数据({})发生异常", records.stream().map(ii -> ii.value()).collect(Collectors.joining(",")), ex);
            allowed2CommitAtFinial = false;
        } finally {
            if (allowed2CommitAtFinial) {
                ack.acknowledge();
            }
        }
    }


    @KafkaListener(
            id = "${spring.kafka.consumer.group-id}-TOPIC_DISPATCH_VGA_ROOMS",
            topics = {
                    KafkaConstants.TOPIC_DISPATCH_VGA_ROOMS
            }, groupId = "${spring.kafka.consumer.group-id}-TOPIC_DISPATCH_VGA_ROOMS")
    public void ackSync_VGA_ROOMS(List<ConsumerRecord<String, String>> records,
                                  Acknowledgment ack) {
        ack.acknowledge();
    }

    @KafkaListener(
            id = "${spring.kafka.consumer.group-id}-TOPIC_DISPATCH_TENCENT_CALLBACK",
            topics = {
                    KafkaConstants.TOPIC_DISPATCH_TENCENT_CALLBACK
            }, groupId = "${spring.kafka.consumer.group-id}-TOPIC_DISPATCH_TENCENT_CALLBACK")
    public void ackSync_DISPATCH_TENCENT_CALLBACK(
            List<ConsumerRecord<String, String>> records,
                                  Acknowledgment ack) {
        Collection<TencentQueueCallbackRequest> callbackRequests
                = records.stream()
                .map(ii -> this.jsonProvider.toObject(TencentQueueCallbackRequest.class, ii.value()))
                .collect(Collectors.toList());
        logger.error("收到腾讯回调信息:{};即将执行如下回调信息", this.jsonProvider.getJson(callbackRequests));
        try {
            callbackRequests.forEach(ii -> {
                for (String targetPublicUrl : ii.getPublicUrls()) {
                    String body = null;
                    try {
                        HttpPost post = new HttpPost(targetPublicUrl);
                        post.addHeader("Content-Type", "application/json");
                        Map bodyMap = ii.getData();

                        body = this.jsonProvider.getJson(bodyMap);
                        post.setEntity(new StringEntity(body));

                        try (CloseableHttpResponse response = this.httpClient.execute(post)) {
                            logger.error("广播腾讯回调: 推送url={}; 请求内容:{}, 状态:{};响应信息:{}",
                                    post.getURI(), body,
                                    response.getStatusLine(),
                                    EntityUtils.toString(response.getEntity())
                            );
                        }
                    } catch (Exception ex) {
                        //todo
                        logger.error("回调信息发生异常(url={})", targetPublicUrl, body, ex);
                    }
                }
            });
        } finally {
            ack.acknowledge();
        }
    }


    private CreateSpecialSosRequest fillSuperviseData(CreateSosPoliceIncidentQueueRequest sr,
                                                      Collection<SearchNearByCompanyResponse> nrs,CreateSpecialSosRequest csr) {
        csr.setSuperviseDepartId(sr.getRecommendedSuperviseDepartId());
        csr.setSuperviseDepartName(sr.getRecommendedSuperviseDepartName());
        if (!CollectionUtils.isEmpty(nrs)) {
            SearchNearByCompanyResponse selectedNearByCompany =
                    nrs.stream().filter(ii -> ii.getFilteredLongitude() == sr.getLongitude() &&
                                    ii.getFilteredLatitude() == sr.getLatitude()).findFirst()
                            .orElse(null);
            if (selectedNearByCompany != null) {

                if (!StringUtil.isEmpty(sr.getRecommendedSuperviseDepartId())) {
                    CompanyDetailResponse companyDetailResponse =
                            this.companyReaderRpcProvider.get(sr.getRecommendedSuperviseDepartId(),false);

                    if (companyDetailResponse != null &&
                            GeoLocationValueType.isValidLocation(companyDetailResponse.getLatitude(),companyDetailResponse.getLongitude())) {
                        double distance = GisPointUtil.getDistance(
                                csr.getLatitude(), csr.getLongitude(),
                                companyDetailResponse.getLatitude(),
                                companyDetailResponse.getLongitude()
                        );

                        if (distance > selectedNearByCompany.getDistance()) {
                            csr.setSuperviseDepartId(selectedNearByCompany.getId());
                            csr.setSuperviseDepartName(selectedNearByCompany.getName());
                        }
                    }
                }
            }
        }

        return csr;
    }
}
