package com.bcxin.flink.streaming.computing.task.jobs;

import com.bcxin.event.core.FlinkJobAbstract;
import com.bcxin.event.core.definitions.JdbcDefinition;
import com.bcxin.event.core.definitions.KafkaTopicConnectorDefinition;
import com.bcxin.event.core.exceptions.BadEventException;
import com.bcxin.event.core.jdbc.JdbcNameParameterSqlParser;
import com.bcxin.event.core.jdbc.JdbcNameParameterSqlParserImpl;
import com.bcxin.event.core.jdbc.ParseSqlParameter;
import com.bcxin.flink.streaming.computing.task.StreamingJobContext;
import com.bcxin.flink.streaming.computing.task.dtos.DebeziumJsonNodeDto;
import com.bcxin.flink.streaming.computing.task.dtos.JobStreamingJdbcSinkData;
import com.bcxin.flink.streaming.cores.StreamingCoreEnvironments;
import com.bcxin.flink.streaming.cores.TableConstant;
import com.bcxin.flink.streaming.cores.utils.JsonNodeUtils;
import com.fasterxml.jackson.databind.JsonNode;
import org.apache.commons.lang3.StringUtils;
import org.apache.flink.api.common.eventtime.WatermarkStrategy;
import org.apache.flink.connector.jdbc.JdbcConnectionOptions;
import org.apache.flink.connector.jdbc.JdbcExecutionOptions;
import org.apache.flink.connector.jdbc.JdbcSink;
import org.apache.flink.connector.kafka.source.KafkaSource;
import org.apache.flink.connector.kafka.source.enumerator.initializer.OffsetsInitializer;
import org.apache.flink.connector.kafka.source.reader.deserializer.KafkaRecordDeserializationSchema;
import org.apache.flink.shaded.jackson2.com.fasterxml.jackson.databind.node.ObjectNode;
import org.apache.flink.streaming.api.datastream.DataStreamSource;
import org.apache.flink.streaming.api.datastream.KeyedStream;
import org.apache.flink.streaming.api.environment.StreamExecutionEnvironment;
import org.apache.flink.streaming.api.functions.sink.SinkFunction;
import org.apache.flink.table.data.TimestampData;
import org.apache.kafka.connect.json.JsonDeserializer;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.jdbc.core.PreparedStatementCreatorFactory;

import java.time.Duration;
import java.time.temporal.ChronoUnit;
import java.util.*;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.stream.Collectors;
import java.util.stream.Stream;

public class StreamingComputingJob  extends FlinkJobAbstract {
    private final static Logger logger = LoggerFactory.getLogger(StreamingComputingJob.class);
    private final static Collection<String> datetimeFieldNames = Stream.of("LASTMODIFIED","CREATED","TIME","DATE","FZRQ").collect(Collectors.toList());
    private final JobStreamingJdbcSinkData jdbcSinkData;

    public StreamingComputingJob(JobStreamingJdbcSinkData jdbcSinkData) {
        this.jdbcSinkData = jdbcSinkData;

        if (jdbcSinkData == null) {
            throw new BadEventException("jdbc sink data 配置不应该为空");
        }
    }

    @Override
    protected void coreExecute() throws Exception {
        StreamingJobContext jobContext = StreamingJobContext.getInstance();
        StreamExecutionEnvironment env = StreamingCoreEnvironments.getStreamExecutionEnvironment(jobContext,false);
        logger.warn("begin to execute the core function:{}", jobContext.getName());

        KafkaRecordDeserializationSchema kafkaRecordDeserializationSchema = KafkaRecordDeserializationSchema.valueOnly(JsonDeserializer.class);
        Collection<JobStreamingJdbcSinkData.Kafka2JdbcDefinitionWrapper> kafka2JdbcDefinitions = this.jdbcSinkData.getKafka2JdbcDefinitions();

        kafka2JdbcDefinitions.forEach(kafka2JdbcDefinitionWrapper -> {
            this.buildMultipleJobs(env, jobContext, kafkaRecordDeserializationSchema, kafka2JdbcDefinitionWrapper);
        });

        env.execute(String.format("flink-job:%s", jobContext.getName()));
    }

    private void buildMultipleJobs(
            StreamExecutionEnvironment env,
            StreamingJobContext jobContext,
            KafkaRecordDeserializationSchema kafkaRecordDeserializationSchema,
            JobStreamingJdbcSinkData.Kafka2JdbcDefinitionWrapper kafka2JdbcDefinitionWrapper) {
        KafkaTopicConnectorDefinition kafkaDefinition = kafka2JdbcDefinitionWrapper.getKafkaDefinition();
        KafkaSource<JsonNode> stringKafkaSource =
                KafkaSource.<ObjectNode>builder()
                        .setBootstrapServers(kafkaDefinition.getBootstrapServer())
                        .setTopics(kafkaDefinition.getTopic())
                        .setGroupId(kafkaDefinition.getConsumerGroupId())
                        .setStartingOffsets(OffsetsInitializer.latest())
                        .setDeserializer(kafkaRecordDeserializationSchema)
                        .build();

        DataStreamSource<JsonNode> dataStreamSource =
                env.fromSource(stringKafkaSource,
                        WatermarkStrategy.forBoundedOutOfOrderness(Duration.of(1, ChronoUnit.MINUTES)),
                        String.format("%s-%s", kafka2JdbcDefinitionWrapper.getName(), kafkaDefinition.getName())
                );

        KeyedStream<DebeziumJsonNodeDto, String> debeziumJsonNodeDtoKeyedStream =
                dataStreamSource.map(ix -> {
                    String op = ix.get("op").asText();
                    JsonNode after = ix.get("after");
                    JsonNode idNode = JsonNodeUtils.getSubJsonNodeByName(after, kafkaDefinition.getPrimaryKeyName());
                    String id = null;
                    if (idNode != null) {
                        id = idNode.isNumber() ? String.valueOf(idNode.asLong()) : idNode.asText();
                    }

                    JsonNode before = ix.get("before");
                    JsonNode source = ix.get("source");
                    Long ts_ms = ix.get("ts_ms").asLong();

                    if (StringUtils.isEmpty(id) && "d".equalsIgnoreCase(op)) {
                        idNode = JsonNodeUtils.getSubJsonNodeByName(before, kafkaDefinition.getPrimaryKeyName());
                        if (idNode != null) {
                            id = idNode.isNumber() ? String.valueOf(idNode.asLong()) : idNode.asText();
                        }
                    }

                    return DebeziumJsonNodeDto.create(id, op, ts_ms, before, after, source);
                }).keyBy(ix -> ix.getKey());

        JdbcExecutionOptions jdbcExecutionOptions =
                JdbcExecutionOptions.builder()
                        .withBatchSize(500)
                        .withBatchIntervalMs(800)
                        .withMaxRetries(10)
                        .build();

        Collection<JdbcDefinition> jdbcDefinitions = kafka2JdbcDefinitionWrapper.getJdbcDefinitions();
        jdbcDefinitions.forEach(ix -> buildJobJdbcTask(ix, jdbcExecutionOptions, debeziumJsonNodeDtoKeyedStream, kafkaDefinition));
    }

    private void buildJobJdbcTask(JdbcDefinition jdbcDefinition, JdbcExecutionOptions jdbcExecutionOptions,
                               KeyedStream<DebeziumJsonNodeDto, String> debeziumJsonNodeDtoKeyedStream,
                               KafkaTopicConnectorDefinition kafkaDefinition) {
        JdbcConnectionOptions jdbcConnectionOptions
                = new JdbcConnectionOptions.JdbcConnectionOptionsBuilder()
                .withUrl(jdbcDefinition.getUrl())
                .withDriverName(jdbcDefinition.getDriveClass())
                .withUsername(jdbcDefinition.getUsername())
                .withPassword(jdbcDefinition.getPassword())
                .build();

        String sqlTemplate = jdbcDefinition.getSqlTemplate();
        Collection<String> sqlTemplateSplits =
                Arrays.stream(sqlTemplate.split(";")).collect(Collectors.toList());
        AtomicInteger indexAtomic = new AtomicInteger(0);

        logger.info("构建待执行的SQL模板={}", sqlTemplate);
        sqlTemplateSplits.stream().filter(ix -> !StringUtils.isEmpty(ix))
                .forEach(sql -> {
                    JdbcNameParameterSqlParser nameParameterSqlParser = new JdbcNameParameterSqlParserImpl();
                    ParseSqlParameter parse2 = nameParameterSqlParser.parse(sql);
                    List<int[]> parameterIndexes = parse2.getParameterIndexes();
                    List<String> parameterNames = parse2.getParameterNames();
                    PreparedStatementCreatorFactory pscf = parse2.getPscf();

                    SinkFunction<DebeziumJsonNodeDto> sinkFunction =
                            JdbcSink.<DebeziumJsonNodeDto>sink(
                                    pscf.getSql(),
                                    (preparedStatement, o) -> {
                                        Map<String, Object> sourceParams = o.getParameters(kafkaDefinition.getPrimaryKeyName());
                                        for (int index = 0; index < parameterIndexes.size(); index++) {
                                            String fieldName = parameterNames.get(index);
                                            Object fieldValue = sourceParams.get(fieldName);
                                            if (fieldValue == null) {
                                                Optional<String> fieldNameOptional =
                                                        sourceParams.keySet().stream()
                                                                .filter(ix -> ix.equalsIgnoreCase(fieldName.trim()))
                                                                .findFirst();

                                                if (fieldNameOptional.isPresent()) {
                                                    fieldValue = sourceParams.get(fieldNameOptional.get());
                                                }
                                            }
                                            if (datetimeFieldNames.stream().anyMatch(ix -> fieldName.toUpperCase().contains(ix))) {
                                                if (fieldValue instanceof Double) {
                                                    fieldValue = TimestampData.fromEpochMillis(((Double) fieldValue).longValue()).toTimestamp();
                                                } else if (fieldValue instanceof Long) {
                                                    fieldValue = TimestampData.fromEpochMillis((Long) fieldValue).toTimestamp();
                                                }
                                            }

                                            preparedStatement.setObject(index + 1, fieldValue);
                                        }
                                    },
                                    jdbcExecutionOptions,
                                    jdbcConnectionOptions
                            );

                    debeziumJsonNodeDtoKeyedStream.addSink(sinkFunction)
                            .name(String.format("Jdbc-%s-%s-%s",
                                    System.getProperty(TableConstant.PARAM_ENV),
                                    jdbcDefinition.getName(), indexAtomic.incrementAndGet()));
                });
    }
}
