package com.bcxin.flink.streaming.computing.task;

import com.bcxin.event.core.*;
import com.bcxin.event.core.definitions.JdbcDefinition;
import com.bcxin.event.core.enums.JobType;
import com.bcxin.event.core.exceptions.BadEventException;
import com.bcxin.event.core.exceptions.IllegalArgumentEventException;
import com.bcxin.event.core.exceptions.NoFoundEventException;
import com.bcxin.flink.streaming.computing.task.dtos.JobStreamingJdbcSinkData;
import com.bcxin.flink.streaming.computing.task.jobs.StreamingComputingJob;
import com.bcxin.flink.streaming.cores.JdbcJobExecutorUtil;
import com.bcxin.flink.streaming.cores.SystemPropertyUtil;
import com.bcxin.flink.streaming.cores.properties.CheckpointConfigProperty;
import com.bcxin.flink.streaming.cores.properties.CheckpointConfigPropertyBuilder;
import com.bcxin.flink.streaming.cores.properties.StreamingConfigConstants;
import org.apache.commons.lang3.StringUtils;
import org.apache.flink.api.java.utils.ParameterTool;

import java.io.IOException;
import java.util.*;
import java.util.stream.Collectors;

public class StreamingJobApp {
    private static final JsonProvider jsonProvider = new JsonProviderImpl();

    public static void main(String[] args) throws Exception {
        System.setProperty("org.slf4j.simpleLogger.defaultLogLevel", "info");
        ParameterTool parameterTool = ParameterTool.fromArgs(args);
        String[] processedArgs = new String[]{
                parameterTool.get("env"),
                parameterTool.get("jobId"),
        };
        JobStreamingJdbcSinkData jdbcSinkData = buildJobEnvironment(processedArgs,parameterTool);

        StreamingComputingJob streamingComputingJob = new StreamingComputingJob(jdbcSinkData);

        streamingComputingJob.execute();
    }

    private static JobStreamingJdbcSinkData buildJobEnvironment(String[] args, ParameterTool parameterTool) throws IOException {
        String env = args[0];
        Properties properties = SystemPropertyUtil.loadEnvConf(env);
        JobParameterDTO jobParameterDTO = JdbcJobExecutorUtil.getJobParameter(Integer.parseInt(args[1]));
        if (jobParameterDTO == null) {
            throw new NoFoundEventException("参数无效; 无法加载对应的job参数");
        }

        if(jobParameterDTO.getJobType()!= JobType.STREAMING_COMPUTED) {
            throw new IllegalArgumentEventException(String.format("该jobType(%s)不支持流处理",jobParameterDTO.getJobType()));
        }

        String corePropertyConf = jobParameterDTO.getParam1(jsonProvider);
        String databasePropertyConf = jobParameterDTO.getParam2(jsonProvider);


        properties = loadProperties(corePropertyConf);

        CheckpointConfigProperty configProperty =
                CheckpointConfigPropertyBuilder.build(properties,parameterTool);

        StreamingJobContext.init(properties.getProperty(StreamingConfigConstants.JOB_NAME), configProperty);

        return buildJobData(databasePropertyConf);
    }

    private static Properties loadProperties(String param1) throws IOException {
        if (StringUtils.isEmpty(param1)) {
            throw new BadEventException("流任务配置信息不能为空");
        }

        String taskConf = param1;
        Map<String,String> inputMap = jsonProvider.toObject(Map.class,taskConf);

        Properties properties = System.getProperties();
        for(String key:inputMap.keySet()) {
            properties.setProperty(key, inputMap.get(key));
        }

        return properties;
    }

    private static JobStreamingJdbcSinkData buildJobData(String jdbcConfig) throws IOException {
        if (StringUtils.isEmpty(jdbcConfig)) {
            throw new BadEventException("流Job配置数据不能为空");
        }

        String jobMeta = jdbcConfig;
        Kafka2JdbcDefinitionWrapperDTO kafka2JdbcDefinitionWrapperDTO =
                jsonProvider.toObject(Kafka2JdbcDefinitionWrapperDTO.class, jobMeta);
        if (kafka2JdbcDefinitionWrapperDTO != null && kafka2JdbcDefinitionWrapperDTO.getKafka2JdbcDefinitions() != null) {
            return buildJobStreamingJdbcSinkData(kafka2JdbcDefinitionWrapperDTO,jobMeta);
        } else {
            JobStreamingJdbcSinkData jobData = jsonProvider.toObject(JobStreamingJdbcSinkData.class, jobMeta);

            return jobData;
        }
    }

    private static JobStreamingJdbcSinkData
    buildJobStreamingJdbcSinkData(Kafka2JdbcDefinitionWrapperDTO kafka2JdbcDefinitionWrapperDTO,
                                                                          String jobMeta) {
        Collection<JdbcSqlTemplateDefinition> jdbcSqlTemplateDefinitions =
                kafka2JdbcDefinitionWrapperDTO.getKafka2JdbcDefinitions().stream()
                        .filter(ii -> ii.getJdbcDefinitions() != null)
                        .flatMap(ix -> ix.getJdbcDefinitions().stream())
                        .collect(Collectors.toList());

        List<Long> jdbcIds =
                jdbcSqlTemplateDefinitions.stream().filter(ix -> ix.getJdbcId() != null).map(ix -> ix.getJdbcId())
                        .distinct()
                        .collect(Collectors.toList());
        Collection<JdbcParameterDTO> jdbcParameterDTOS = JdbcJobExecutorUtil.getJdbcParameterDTO(jdbcIds);
        JobStreamingJdbcSinkData jobData = jsonProvider.toObject(JobStreamingJdbcSinkData.class, jobMeta);

        /**
         * 生成对应的JdbcDefinition的定义
         */
        for (JobStreamingJdbcSinkData.Kafka2JdbcDefinitionWrapper kafka2JdbcDefinitionWrapper : jobData.getKafka2JdbcDefinitions()) {
            for (JdbcDefinition jdbcDefinition : kafka2JdbcDefinitionWrapper.getJdbcDefinitions()) {
                Optional<JdbcParameterDTO> selectedJdbcParameterOptional =
                        jdbcParameterDTOS.stream().filter(ix -> ix.getJdbcId() != null && ix.getJdbcId().equals(jdbcDefinition.getJdbcId())).findFirst();
                if (selectedJdbcParameterOptional.isPresent()) {
                    JdbcParameterDTO jdbcParameter = selectedJdbcParameterOptional.get();
                    jdbcDefinition.change(jdbcParameter.getName(),
                            jdbcParameter.getUrl(),
                            jdbcParameter.getUsername(),
                            jdbcParameter.getPassword(),
                            jdbcParameter.getConnectionTimeZone(),
                            jdbcParameter.getDriveClass());
                } else {
                    throw new NoFoundEventException("JdbcId参数无效; 无法找到对应的Jdbc配置");
                }
            }
        }

        return jobData;
    }
}
