package com.bcxin.flink.cdc.dispatch.task.jobs;

import com.bcxin.event.core.FlinkJobAbstract;
import com.bcxin.event.core.JsonProvider;
import com.bcxin.event.core.JsonProviderImpl;
import com.bcxin.event.core.exceptions.BadEventException;
import com.bcxin.event.core.exceptions.NoSupportEventException;
import com.bcxin.flink.cdc.dispatch.task.JobContext;
import com.bcxin.flink.cdc.dispatch.task.cdcs.CdcSourceMeta;
import com.bcxin.flink.cdc.dispatch.task.cdcs.ExecuteJobData;
import com.bcxin.flink.cdc.dispatch.task.components.CdcBinlog2DispatchComponent;
import com.bcxin.flink.cdc.dispatch.task.components.MultiThreadConsumerSink;
import com.bcxin.flink.cdc.dispatch.task.enums.DispatchType;
import com.bcxin.flink.cdc.dispatch.task.proerpties.CdcDatabaseSourceProperty;
import com.bcxin.flink.streaming.cores.JdbcJobExecutorUtil;
import com.ververica.cdc.connectors.mysql.source.MySqlSource;
import com.ververica.cdc.connectors.mysql.source.MySqlSourceBuilder;
import com.ververica.cdc.connectors.mysql.table.StartupOptions;
import com.ververica.cdc.debezium.JsonDebeziumDeserializationSchema;
import org.apache.commons.lang3.StringUtils;
import org.apache.flink.api.common.eventtime.WatermarkStrategy;
import org.apache.flink.api.common.functions.RichMapFunction;
import org.apache.flink.configuration.Configuration;
import org.apache.flink.runtime.state.hashmap.HashMapStateBackend;
import org.apache.flink.streaming.api.CheckpointingMode;
import org.apache.flink.streaming.api.datastream.DataStreamSource;
import org.apache.flink.streaming.api.datastream.KeyedStream;
import org.apache.flink.streaming.api.environment.StreamExecutionEnvironment;
import org.apache.flink.streaming.api.functions.sink.RichSinkFunction;
import org.apache.http.HttpStatus;
import org.apache.http.StatusLine;
import org.apache.http.client.methods.*;
import org.apache.http.entity.AbstractHttpEntity;
import org.apache.http.entity.StringEntity;
import org.apache.http.impl.client.CloseableHttpClient;
import org.apache.http.impl.client.HttpClients;
import org.apache.http.util.EntityUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.IOException;
import java.nio.charset.StandardCharsets;
import java.sql.*;
import java.time.Duration;
import java.time.temporal.ChronoUnit;
import java.util.*;

public class Binlog2DispatchCdcJob  extends FlinkJobAbstract {
    private final static Logger logger = LoggerFactory.getLogger(Binlog2DispatchCdcJob.class);
    @Override
    protected void coreExecute() throws Exception {
        JobContext jobContext = JobContext.getInstance();
        StreamExecutionEnvironment env = StreamExecutionEnvironment.getExecutionEnvironment();
        env.enableCheckpointing(3000, CheckpointingMode.EXACTLY_ONCE);
        env.setStateBackend(new HashMapStateBackend());
        env.getCheckpointConfig().setCheckpointStorage(jobContext.getConfigProperty().getCheckpointPath());

        DataStreamSource<String> dataStreamSource = initDatabaseBinlog(env);

        KeyedStream<CdcSourceMeta, String> cdcSourceMetaKeyedStream =
                dataStreamSource.map(new RichMapFunction<String, CdcSourceMeta>() {
                    @Override
                    public CdcSourceMeta map(String value) throws Exception {
                        JsonProvider jsonProvider = new JsonProviderImpl();
                        CdcSourceMeta sourceMeta = jsonProvider.toObject(CdcSourceMeta.class, value);
                        if (sourceMeta.getSource() == null) {
                            sourceMeta.setOriginalValue(value);
                        }
                        return sourceMeta;
                    }
                }).keyBy(ix -> ix.getPartitionKey());

        cdcSourceMetaKeyedStream.addSink(new MultiThreadConsumerSink());

        /*
        cdcSourceMetaKeyedStream.addSink(new RichSinkFunction<CdcSourceMeta>() {
            @Override
            public void invoke(CdcSourceMeta sourceMeta, Context context) throws Exception {
                //super.invoke(value, context);
                logger.error("v2.1.开始执行CDC数据Binlog跟踪:{},{}.{}",sourceMeta.getId(),sourceMeta.getDbName(),sourceMeta.getTableName());
                CdcBinlog2DispatchComponent.doExecute(Collections.singleton(sourceMeta));
            }
        });
         */

        env.execute(String.format("V2归集:%s-jobId=%s", jobContext.getName(), jobContext.getName()));
    }

    /**
     * 默认采用flink cdc就不会造成数据库锁的问题
     * @param env
     * @return
     */
    private DataStreamSource<String> initDatabaseBinlog(StreamExecutionEnvironment env) {
        JobContext jobContext = JobContext.getInstance();
        CdcDatabaseSourceProperty databaseProperty = jobContext.getDatabaseProperty();
        Properties debeziumProperties = new Properties();
        debeziumProperties.put("snapshot.mode","when_needed");
        debeziumProperties.put("snapshot.locking.mode", "none");// do not use lock
        /**
         * The connector is trying to read binlog starting at Struct{version=1.6.4.Final,connector=mysql,name=mysql_binlog_source,ts_ms=1682067191784,db=,server_id=0,file=mysql-bin.001505,pos=50666530,row=0}, but this is no longer available on the server
         *
         */
        String[] dbLists = Arrays.stream(databaseProperty.getDbList().split(";"))
                .filter(ii -> !StringUtils.isEmpty(ii)).toArray(String[]::new);

        String[] tableList = Arrays.stream(databaseProperty.getTableList().split(";"))
                .filter(ii -> !StringUtils.isEmpty(ii)).toArray(size->new String[size]);
        logger.info("当前设置的serverId为:{}",databaseProperty.getServerId());

        logger.error("捕获的DB binlog的配置信息为:{}-{}", jobContext.getDatabaseProperty().getHostName(),
                databaseProperty.getUserName()
        );

        Timestamp monitorLastTimestamp = Timestamp.valueOf("2023-04-27 09:00:00");
        MySqlSourceBuilder<String> mySqlSourceBuilder = MySqlSource.<String>builder()
                .hostname(jobContext.getDatabaseProperty().getHostName())
                .port(jobContext.getDatabaseProperty().getPort())
                .databaseList(dbLists)
                .tableList(tableList)
                .username(databaseProperty.getUserName())
                .password(databaseProperty.getPassword())
                .connectionTimeZone(databaseProperty.getConnectionTimeZone())
                /**
                 * 先从20230407开始最新开始
                 */
                .startupOptions(StartupOptions.timestamp(monitorLastTimestamp.getTime()))
                //.startupOptions(StartupOptions.latest())
                //.serverId(databaseProperty.getServerId())
                .serverId("5000-6800")
                .includeSchemaChanges(true)
                .connectTimeout(
                        Duration.of(databaseProperty.getConnectTimeout(),
                                ChronoUnit.MILLIS))
                .deserializer(new JsonDebeziumDeserializationSchema(false))
                .debeziumProperties(debeziumProperties);
        if (!StringUtils.isEmpty(databaseProperty.getServerId())) {
            mySqlSourceBuilder = mySqlSourceBuilder.serverId(databaseProperty.getServerId());
        }

        MySqlSource<String> mySqlSource = mySqlSourceBuilder.build();

        return env.fromSource(mySqlSource,
                WatermarkStrategy.forBoundedOutOfOrderness(Duration.of(200, ChronoUnit.MILLIS)),
                databaseProperty.getName());
    }


}
