/**
 * Copyright &copy; 2012-2014 <a href="https://github.com/thinkgem/jeesite">JeeSite</a> All rights reserved.
 */
package com.bcxin.platform.util;

import cn.hutool.core.date.DateUnit;
import cn.hutool.core.date.DateUtil;
import com.google.common.collect.ImmutableList;
import com.google.common.collect.Lists;
import org.apache.commons.lang3.time.DateFormatUtils;
import org.joda.time.DateTime;
import org.joda.time.LocalDate;

import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.*;

/**
 * 日期工具类, 继承org.apache.commons.lang.time.DateUtils类
 *
 * @author zhangye
 * @version 1.0
 */
public class DateUtils extends org.apache.commons.lang3.time.DateUtils {

	private static String[] parsePatterns = { "yyyy-MM-dd", "yyyy-MM-dd HH:mm:ss", "yyyy-MM-dd HH:mm", "yyyy-MM",
			"yyyy/MM/dd", "yyyy/MM/dd HH:mm:ss", "yyyy/MM/dd HH:mm", "yyyy/MM", "yyyy.MM.dd", "yyyy.MM.dd HH:mm:ss",
			"yyyy.MM.dd HH:mm", "yyyy.MM", "yyyyMMddHHmmss", "yyyyMMdd", "yyyyMM", "yyyy年MM月dd日", "HH:mm" };

	/**
	 * 得到当前日期字符串 格式（yyyy-MM-dd）
	 */
	public static String getDate() {
		return getDate("yyyy-MM-dd");
	}

	/**
	 * 得到当前日期字符串 格式（yyyy-MM-dd） pattern可以为："yyyy-MM-dd" "HH:mm:ss" "E"
	 */
	public static String getDate(String pattern) {
		return DateFormatUtils.format(new Date(), pattern);
	}

	/**
	 * 得到日期字符串 默认格式（yyyy-MM-dd） pattern可以为："yyyy-MM-dd" "HH:mm:ss" "E"
	 */
	public static String formatDate(Date date, Object... pattern) {
		String formatDate = null;
		if (pattern != null && pattern.length > 0) {
			formatDate = DateFormatUtils.format(date, pattern[0].toString());
		} else {
			formatDate = DateFormatUtils.format(date, "yyyy-MM-dd");
		}
		return formatDate;
	}

	/**
	 * 得到日期时间字符串，转换格式（yyyy-MM-dd HH:mm:ss）
	 */
	public static String formatDateTime(Date date) {
		return formatDate(date, "yyyy-MM-dd HH:mm:ss");
	}

	/**
	 * 得到日期时间字符串，转换格式（HH:mm:ss）
	 */
	public static String formatTime(Date date) {
		return formatDate(date, "HH:mm:ss");
	}

	/**
	 * 得到当前时间字符串 格式（HH:mm:ss）
	 */
	public static String getTime() {
		return formatDate(new Date(), "HH:mm:ss");
	}

	/**
	 * 得到当前日期和时间字符串 格式（yyyy-MM-dd HH:mm:ss）
	 */
	public static String getDateTime() {
		return formatDate(new Date(), "yyyy-MM-dd HH:mm:ss");
	}

	/**
	 * 得到当前年份字符串 格式（yyyy）
	 */
	public static String getYear() {
		return formatDate(new Date(), "yyyy");
	}

	/**
	 * 得到当前月份字符串 格式（MM）
	 */
	public static String getMonth() {
		return formatDate(new Date(), "MM");
	}

	/**
	 * 得到当天字符串 格式（dd）
	 */
	public static String getDay() {
		return formatDate(new Date(), "dd");
	}

	/**
	 * 得到当前星期字符串 格式（E）星期几
	 */
	public static String getWeek() {
		return formatDate(new Date(), "E");
	}

	/**
	 * 日期型字符串转化为日期 格式 { "yyyy-MM-dd", "yyyy-MM-dd HH:mm:ss", "yyyy-MM-dd HH:mm",
	 * "yyyy/MM/dd", "yyyy/MM/dd HH:mm:ss", "yyyy/MM/dd HH:mm", "yyyy.MM.dd",
	 * "yyyy.MM.dd HH:mm:ss", "yyyy.MM.dd HH:mm" }
	 */
	public static Date parseDate(Object str) {
		if (str == null) {
			return null;
		}
		try {
			return parseDate(str.toString(), parsePatterns);
		} catch (ParseException e) {
			return null;
		}
	}

	/**
	 * 获取明天日期
	 *
	 * @return yyyy-MM-dd
	 */
	public static String getTomorrow() {
		SimpleDateFormat df = new SimpleDateFormat("yyyy-MM-dd");
		Calendar c = Calendar.getInstance();
		c.add(Calendar.DATE, 1);
		return df.format(c.getTime());
	}

	/**
	 * 获取过去的天数
	 *
	 * @param date
	 * @return
	 */
	public static long pastDays(Date date) {
		long t = System.currentTimeMillis() - date.getTime();
		return t / (24 * 60 * 60 * 1000);
	}

	/**
	 * 获取过去的小时
	 *
	 * @param date
	 * @return
	 */
	public static long pastHour(Date date) {
		long t =System.currentTimeMillis() - date.getTime();
		return t / (60 * 60 * 1000);
	}

	/**
	 * 获取过去的分钟
	 *
	 * @param date
	 * @return
	 */
	public static long pastMinutes(Date date) {
		long t = System.currentTimeMillis() - date.getTime();
		return t / (60 * 1000);
	}

	/**
	 * 转换为时间（天,时:分:秒.毫秒）
	 *
	 * @param timeMillis
	 * @return
	 */
	public static String formatDateTime(long timeMillis) {
		long day = timeMillis / (24 * 60 * 60 * 1000);
		long hour = (timeMillis / (60 * 60 * 1000) - day * 24);
		long min = ((timeMillis / (60 * 1000)) - day * 24 * 60 - hour * 60);
		long s = (timeMillis / 1000 - day * 24 * 60 * 60 - hour * 60 * 60 - min * 60);
		long sss = (timeMillis - day * 24 * 60 * 60 * 1000 - hour * 60 * 60 * 1000 - min * 60 * 1000 - s * 1000);
		return (day > 0 ? day + "," : "") + hour + ":" + min + ":" + s + "." + sss;
	}

	/*
     * 将时间戳转换为时间
     */
	public static String stampToDate(long timeMillis){
		return formatDateTime(new Date(timeMillis));
	}

	/**
	 * 获取两个日期之间的天数
	 *
	 * @param before
	 * @param after
	 * @return
	 */
	public static double getDistanceOfTwoDate(Date before, Date after) {
		long beforeTime = before.getTime();
		long afterTime = after.getTime();
		return ((afterTime - beforeTime) / (1000 * 60 * 60 * 24) + 1L);
	}

	/**
	 * 获取两个日期之间的天数 年月日
	 *
	 * @param before
	 * @param after
	 * @return
	 */
	public static String getDayOfTwoDate(Date before, Date after) {
		long beforeTime = before.getTime();
		long afterTime = after.getTime();
		int day = (int) ((afterTime - beforeTime) / (1000 * 60 * 60 * 24) + 1L);
		int year = day / 365;
		int month = day % 365 / 30;
		int days = day % 365 % 30;
		String yearStr = year > 0 ? year + "年" : "";
		String monthStr = month > 0 ? month + "月" : "";
		String daysStr = days > 0 ? days + "天" : "";
		String limitTime = yearStr + monthStr + daysStr;
		return limitTime;
	}

	/**
	 * 根据月份获得当月的日期集合
	 *
	 * @param month
	 *            格式：2018-01
	 * @return 2018-01-01 ,2018-01-02 ,2018-01-03 ,2018-01-04 ...
	 */
	public static List<String> getDatesByMonth(String month) {
		DateTime date = new DateTime(month + "-01");
		final ImmutableList.Builder<Long> dayList = ImmutableList.builder();

		LocalDate firstDay = date.toLocalDate().withDayOfMonth(1);
		final LocalDate nextMonthFirstDay = firstDay.plusMonths(1);
		while (firstDay.isBefore(nextMonthFirstDay)) {
			dayList.add(firstDay.toDateTimeAtStartOfDay().getMillis());
			firstDay = firstDay.plusDays(1);
		}
		ImmutableList<Long> dates = dayList.build();
		List<String> dateList = Lists.newArrayList();
		dates.forEach(t -> {
			dateList.add(new SimpleDateFormat("yyyy-MM-dd").format(new Date(t)));
		});
		return dateList;
	}
public static String dateAdd(String type, String sdate, String format, int num) throws Exception{
        SimpleDateFormat df = new SimpleDateFormat(format);
        Date date = df.parse(sdate);
        Calendar cal = Calendar.getInstance();
        cal.setTime(date);
        if("y".equals(type)){
            cal.add(Calendar.YEAR, num);
        }else if("m".equals(type)){
            cal.add(Calendar.MONTH, num);
        }else if("h".equals(type)){
            cal.add(Calendar.HOUR, num);
        }else if("i".equals(type)){
            cal.add(Calendar.MINUTE, num);
        }else if("s".equals(type)){
            cal.add(Calendar.SECOND, num);
        }else {
            cal.add(Calendar.DATE, num);
        }
        return df.format(cal.getTime());
    }

	/**
	 * 根据开始时间和结束时间获得的日期集合
	 *
	 * @param startDate
	 * @param endDate
	 */
	public static List<String> getBetweenDatesByDate(Date startDate, Date endDate) {
		List<String> dateList = Lists.newArrayList();
		long days = DateUtil.betweenDay(startDate, endDate, false);
		dateList.add(DateUtil.formatDate(startDate));
		for (int i = 1; i <= days; i++) {
			Date nextDate = DateUtil.offsetDay(startDate, i);
			dateList.add(DateUtil.formatDate(nextDate));
		}
		return dateList;
	}

	/**
	 * 获取某天是星期几
	 *
	 * @param date
	 *            格式：2018-01
	 * @return 星期五
	 */
	public static String getWeek(Date date) {
		SimpleDateFormat sdf = new SimpleDateFormat("EEEE");
		String week = sdf.format(date);
		return week;
	}

	/**
	 * 获取某天是星期几(int)
	 *
	 * @param date
	 *            格式：2018-01
	 * @return 1 2 3 4 5 6 7
	 */
	public static int getWeekInt(Date date) {
		int weekInt = 0;
		SimpleDateFormat sdf = new SimpleDateFormat("EEEE");
		String week = sdf.format(date);
		if ("星期一".equals(week)) {
			weekInt = 1;
		}
		if ("星期二".equals(week)) {
			weekInt = 2;
		}
		if ("星期三".equals(week)) {
			weekInt = 3;
		}
		if ("星期四".equals(week)) {
			weekInt = 4;
		}
		if ("星期五".equals(week)) {
			weekInt = 5;
		}
		if ("星期六".equals(week)) {
			weekInt = 6;
		}
		if ("星期日".equals(week)) {
			weekInt = 7;
		}
		return weekInt;
	}

	/**
	 * 获取某天一个月后共几天
	 *
	 * @param date
	 *            格式：2018-01
	 * @return 30
	 */
	public static int getDayNumAfterOneMonth(Date date) {
		Calendar cal = Calendar.getInstance();
		cal.setTime(date);
		cal.add(Calendar.MONTH, 1);
		Date dayAfter = cal.getTime();
		// 获取天数
		double dayNum = DateUtils.getDistanceOfTwoDate(date, dayAfter);
		return (int) dayNum;
	}

	/**
	 * 获取某天一个月后是几月几号
	 *
	 * @param date
	 *            格式：2018-01
	 * @return 2018-01
	 */
	public static Date getDayAfterOneMonth(Date date) {
		Calendar cal = Calendar.getInstance();
		cal.setTime(date);
		cal.add(Calendar.MONTH, 1);
		Date dayAfter = cal.getTime();
		return dayAfter;
	}

	/**
	 * 获取某天过几天是几号
	 *
	 * @param date
	 *            格式：2018-01
	 * @param i
	 *            格式：1
	 * @return 2018-01
	 */
	public static String getDayAfterSomeDay(Date date, int i) {
		Calendar cal = Calendar.getInstance();
		cal.setTime(date);
		cal.add(Calendar.DAY_OF_MONTH, i);
		Date dayAfter = cal.getTime();
		return formatDate(dayAfter);
	}

	/**
	 * 获取几分钟后是几号
	 *
	 * @param date
	 * @param i
	 * @return
	 */
	public static Date getDayAfterSomeMinute(Date date, int i) {
		Calendar cal = Calendar.getInstance();
		cal.setTime(date);
		cal.add(Calendar.MINUTE, i);
		return cal.getTime();
	}

	/**
	 * 计算两个日期之间相差的小时
	 *
	 * @param smdate
	 *            较小的时间
	 * @param bdate
	 *            较大的时间
	 * @return 相差小时
	 * @throws ParseException
	 */
	public static String hourDiff(Date smdate, Date bdate) {
		String hour = "";
		String minute = "";
		if (smdate == null || bdate == null) {
			return "";
		}
		SimpleDateFormat sdf = new SimpleDateFormat("yyyy-MM-dd HH:mm:ss");
		try {
			smdate = sdf.parse(sdf.format(smdate));
			bdate = sdf.parse(sdf.format(bdate));
		} catch (ParseException e) {
			e.printStackTrace();
		}
		Calendar cal = Calendar.getInstance();
		cal.setTime(smdate);
		double time1 = cal.getTimeInMillis();
		cal.setTime(bdate);
		double time2 = cal.getTimeInMillis();
		double between_hour = (time2 - time1) / (1000 * 3600);
		double between_minu = (time2 - time1) / (1000 * 60);

		if (between_hour >= 1) {
			hour = (int) between_hour + "小时";
		} else {
			hour = "";
		}
		if (between_minu % 60 > 0.0) {
			minute = (int) between_minu % 60 + "分钟";
		}
		return hour + minute;
	}

	/**
	 * 计算两个日期之间相差的分钟数
	 *
	 * @param smdate
	 *            较小的时间
	 * @param bdate
	 *            较大的时间
	 * @return 相差分钟数
	 * @throws ParseException
	 */
	public static Double minDiff(Date smdate, Date bdate) {
		String hour = "";
		String minute = "";
		if (smdate == null || bdate == null) {
			return null;
		}
		SimpleDateFormat sdf = new SimpleDateFormat("yyyy-MM-dd HH:mm:ss");
		try {
			smdate = sdf.parse(sdf.format(smdate));
			bdate = sdf.parse(sdf.format(bdate));
		} catch (ParseException e) {
			e.printStackTrace();
		}
		Calendar cal = Calendar.getInstance();
		cal.setTime(smdate);
		double time1 = cal.getTimeInMillis();
		cal.setTime(bdate);
		double time2 = cal.getTimeInMillis();
		double between_minu = (time2 - time1) / (1000 * 60);
		return between_minu;
	}

	/**
	 * 计算两个日期之间相差的分钟数
	 *
	 * @param smallDate
	 *            较小的时间
	 * @param bigDate
	 *            较大的时间
	 * @return 相差分钟数
	 * @throws ParseException
	 */
	public static Double minDiff(String smallDate, String bigDate) {

		Date smdate = DateUtils.parseDate(smallDate);
		Date bdate = DateUtils.parseDate(bigDate);

		String hour = "";
		String minute = "";
		if (smdate == null || bdate == null) {
			return null;
		}
		SimpleDateFormat sdf = new SimpleDateFormat("yyyy-MM-dd HH:mm:ss");
		try {
			smdate = sdf.parse(sdf.format(smdate));
			bdate = sdf.parse(sdf.format(bdate));
		} catch (ParseException e) {
			e.printStackTrace();
		}
		Calendar cal = Calendar.getInstance();
		cal.setTime(smdate);
		double time1 = cal.getTimeInMillis();
		cal.setTime(bdate);
		double time2 = cal.getTimeInMillis();
		double between_minu = (time2 - time1) / (1000 * 60);
		return between_minu;
	}

	/*
	 * 传入两个日期，date1和date2<br> 返回date2减去date1所得的分钟数，long类型
	 */
	public static long getDateSubDateToMinutes(Date date1, Date date2) {
		long minute = 0;
		try {
			if (date1 != null && date2 != null) {
				long long1 = date1.getTime();
				long long2 = date2.getTime();
				minute = (long2 - long1) / 1000 / 60;
			} else {
				// System.out.println("date1或date2的对象为空!");
			}
		} catch (Exception e) {
			System.err.println("计算date2减去date1所得的秒数失败!");
		}
		return minute;
	}

	/*
	 * 传入两个日期，date1和date2<br> 返回date2减去date1所得的秒，long类型
	 */
	public static long getDateSubDateToSeconds(Date date1, Date date2) {
		long second = 0;
		try {
			if (date1 != null && date2 != null) {
				long long1 = date1.getTime();
				long long2 = date2.getTime();
				second = (long2 - long1) / 1000;
			} else {
				// System.out.println("date1或date2的对象为空!");
			}
		} catch (Exception e) {
			System.err.println("计算date2减去date1所得的秒数失败!");
		}
		return second;
	}

	/**
	 * 根据分钟数 计算 天时分
	 *
	 * @param totalMinutes
	 * @return
	 */
	public static String getDayHourMinute(Long totalMinutes) {
		// 计算天 1天 = 24*60 1小时=60
		Long day = totalMinutes / (24 * 60);
		Long hour = (totalMinutes % (24 * 60)) / 60;
		Long minute = (totalMinutes % (24 * 60)) % 60;
		String content = "";
		if (day > 0) {
			content = content + day + "天";
		}
		if (hour > 0) {
			content = content + hour + "小时";
		}
		if (minute > 0) {
			content = content + minute + "分钟";
		}
		return content;
	}

	/**
	 * 根据年 月 获取对应的月份 天数
	 */
	public static int getDaysByYearMonth(int year, int month) {

		Calendar a = Calendar.getInstance();
		a.set(Calendar.YEAR, year);
		a.set(Calendar.MONTH, month - 1);
		a.set(Calendar.DATE, 1);
		a.roll(Calendar.DATE, -1);
		int maxDate = a.get(Calendar.DATE);
		return maxDate;
	}

	/**
	 * 校验日期是否合法
	 * 
	 * @param sourceDate
	 * @return boolean by llc 2018-09-17
	 */
	public static boolean checkDate(String sourceDate) {
		if (sourceDate == null) {
			return false;
		}
		/*** 针对 201-9-08-26能判断日期正确的情况 ***/
		if(sourceDate.length()-(sourceDate.replaceAll("-","").length())!=2){
			return false;
		}
		try {
			SimpleDateFormat dateFormat = new SimpleDateFormat("yyyy-MM-dd");
			dateFormat.setLenient(false);
			dateFormat.parse(sourceDate);
			return true;
		} catch (Exception e) {
		}
		return false;
	}


	/**
	 * 校验时间是否合法
	 *
	 * @param sourceTime
	 * @date 2020-01-21
	 * @auth llc
	 * @return boolean
	 */
	public static boolean checkTime(String sourceTime ) {
		if (sourceTime == null) {
			return false;
		}
		/*** 针对 201-9-08-26能判断日期正确的情况 ***/
		if(sourceTime.length()-(sourceTime.replaceAll("-","").length())!=2){
			return false;
		}
		try {
			SimpleDateFormat dateFormat = new SimpleDateFormat("yyyy-MM-dd HH:mm:ss");
			dateFormat.setLenient(false);
			dateFormat.parse(sourceTime);
			return true;
		} catch (Exception e) {
		}
		return false;
	}


	/**
	 * 比较日期大小 说明：如果 strDate1 <= strDate2，返回true，否则返回false
	 * 
	 * @param strDate1
	 * @param strDate2
	 * @return boolean by llc 2018-09-17
	 */
	public static boolean dateCompare(String strDate1, String strDate2) {
		boolean flag = false;
		if (strDate1 == null || strDate2 == null) {
			return flag;
		}
		SimpleDateFormat dateFormat = new SimpleDateFormat("yyyy-MM-dd");
		try {
			Date date1 = dateFormat.parse(strDate1);
			Date date2 = dateFormat.parse(strDate2);
			if (date1.getTime() <= date2.getTime()) {
				flag = true;
			} else {
				flag = false;
			}

		} catch (Exception e) {
			flag = false;
		}
		return flag;
	}

	/**
	 * 计算两者时间的等待时长（用于审批）
	 * 
	 * @param startTime
	 * @param endTime
	 * @return String by llc 2018-11-13
	 */
	public static String calculationTimes(Date startTime, Date endTime) {
		long longWaitTime = DateUtil.between(startTime, endTime, DateUnit.SECOND);
		String timeDiff = DateUtil.secondToTime(Integer.parseInt(String.valueOf(longWaitTime)));
		String[] timeArr = timeDiff.split(":");
		List<Integer> timeList = new ArrayList<>();
		for (String time : timeArr) {
			if (time.equalsIgnoreCase("00")) {
				time = "0";
			}
			timeList.add(Integer.parseInt(time));
		}
		String waitTime = "已等待";
		if (timeList.get(0) == 0) {
			waitTime += timeList.get(1) + "分钟" + timeList.get(2) + "秒";
		} else {
			if (timeList.get(0) >= 24) {
				waitTime += (timeList.get(0) / 24) + "天" + (timeList.get(0) % 24) + "小时" + timeList.get(1) + "分钟"
						+ timeList.get(2) + "秒";
			} else {
				waitTime += timeList.get(0) + "小时" + timeList.get(1) + "分钟" + timeList.get(2) + "秒";
			}
		}
		return waitTime;
	}

	/**
	 * 获得指定日期的后一天
	 * 
	 * @param dates
	 * @return
	 */
	public static String getNextDay(String dates) {
		Calendar c = Calendar.getInstance();
		Date date = null;
		try {
			date = new SimpleDateFormat("yy-MM-dd").parse(dates);
		} catch (ParseException e) {
			e.printStackTrace();
		}
		c.setTime(date);
		int day = c.get(Calendar.DATE);
		c.set(Calendar.DATE, day + 1);

		String dayAfter = new SimpleDateFormat("yyyy-MM-dd").format(c.getTime());
		return dayAfter;
	}


	/**
	 * 获取某日是星期几(已按照中国情况做处理)
	 *
	 * @param date
	 * @return
	 * by llc 2019-01-23
	 */
	public static int dayOfWeek(String date) {
		int weekDaty = DateUtil.dayOfWeek(DateUtil.parseDate(date)) == 1? 7 : DateUtil.dayOfWeek(DateUtil.parseDate(date))-1;
		return weekDaty;
	}


	/**
	 * 获得指定日期的前n天数据
	 * @param specifiedDay
	 * @return
	 * @throws Exception
	 */
	public static String getSpecifiedDayBefore(String specifiedDay,int days){
		Calendar c = Calendar.getInstance();
		Date date=null;
		try {
			date = new SimpleDateFormat("yy-MM-dd").parse(specifiedDay);
		} catch (ParseException e) {
			e.printStackTrace();
		}
		c.setTime(date);
		int day=c.get(Calendar.DATE);
		c.set(Calendar.DATE,day - days);

		String dayBefore=new SimpleDateFormat("yyyy-MM-dd").format(c.getTime());
		return dayBefore;
	}

	public static List<Date> getBetweenDates(Date start, Date end) {
		List<Date> result = new ArrayList<Date>();
		Calendar tempStart = Calendar.getInstance();
		tempStart.setTime(start);
		Calendar tempEnd = Calendar.getInstance();
		tempEnd.setTime(end);
		while (tempStart.before(tempEnd)) {
			result.add(tempStart.getTime());
			tempStart.add(Calendar.DAY_OF_YEAR, 1);
		}
		return result;
	}

    /**
     * jianjun.wang
     * 获取当月所有天
     * @return
     */
	public static List<String> getDayListOfMonth(String paramMonth) {
		List<String> list = new ArrayList<String>();
		Calendar aCalendar = Calendar.getInstance(Locale.CHINA);
		aCalendar.setTime(DateUtil.parse(paramMonth, "yyyy-MM"));
		int year = aCalendar.get(Calendar.YEAR);//年份
		int month = aCalendar.get(Calendar.MONTH) + 1;//月份
		int day = aCalendar.getActualMaximum(Calendar.DATE);
		for (int i = 1; i <= day; i++) {
			String aDate = String.valueOf(year) + "/" + month + "/" + i;
			list.add(aDate);
		}
		return list;
	}

	/***
	 * 获取某个时间的前几分钟
	 * @param dateStr
	 * @param n
	 * @date 2020-01-21
	 * @auth llc
	 * @return
	 */
	public static String getBeforeTime( String dateStr, int n){
		Calendar c = Calendar.getInstance();
		Date beforeTime  =null;
		try {
			beforeTime  = new SimpleDateFormat("yyyy-MM-dd HH:mm:ss").parse(dateStr);
		} catch (ParseException e) {
			e.printStackTrace();
		}
		c.setTime(beforeTime);
		c.add(Calendar.MINUTE, n);
		String dayBefore=  new SimpleDateFormat("yyyy-MM-dd HH:mm:ss").format(c.getTime());
		return dayBefore;
	}


	/**
	 * @param args
	 * @throws ParseException
	 */
	public static void main(String[] args) throws ParseException {
		System.out.println(DateUtils.formatDate(new Date(), "HHmmss"));
		// System.out.println(formatDate(parseDate("2010/3/6")));
		// System.out.println(getDate("yyyy年MM月dd日 E"));
		// long time = new Date().getTime()-parseDate("2012-11-19").getTime();
		// System.out.println(getDayAfterSomeMinute(new Date(),-10));
		// System.out.println(getDayAfterSomeMinute(new Date(), -10));
		// System.out.println(350 / 60);
		// System.out.println((int) ((Math.random() * 9 + 1) * 100000));
		// System.out.println(DateUtil.between(parseDate("2018-12-06
		// 07:00:00","yyyy-MM-dd HH:mm:ss"),parseDate("2018-12-06
		// 07:30:00","yyyy-MM-dd HH:mm:ss"), DateUnit.SECOND));
		// System.out.println(DateUtil.between(DateUtils.parseDate("2018-12-13"
		// + " 23:59:59", "yyyy-MM-dd
		// HH:mm:ss"),DateUtils.parseDate("2018-12-13" + " 23:59:59",
		// "yyyy-MM-dd HH:mm:ss"), DateUnit.SECOND));
		// System.out.println(DateUtil.between(DateUtils.parseDate("2018-12-14"
		// + " 00:00:00", "yyyy-MM-dd
		// HH:mm:ss"),DateUtils.parseDate("2018-12-14" + " 02:00:00",
		// "yyyy-MM-dd HH:mm:ss"), DateUnit.SECOND));
		// System.out.println(DateUtil.between(new
		// Date(),DateUtils.parseDate("2018-12-14" + " 02:00:00", "yyyy-MM-dd
		// HH:mm:ss"), DateUnit.SECOND));
		// System.out.println(getNextDay("2018-10-31"));
		// System.out.println(DateUtils.formatDate(new Date()));
		// System.out.println(new BigDecimal(Math.round(232 * 100 / 60) /
		// 100.0));
//		System.out.println(new BigDecimal(10.01).divide(new BigDecimal(3), 2, BigDecimal.ROUND_DOWN));
		// BigDecimal a = new BigDecimal(2.3); //
		// BigDecimal b = new BigDecimal(4.3); //
		// System.out.println(b.compareTo(a) == 1);
//		System.out.println(DateUtils.parseDate(DateUtils.formatDate(new Date())));
		String monthfirstDay = "2020-04" + "-01";
		String monthLastDay = DateUtils.formatDate(DateUtil.endOfMonth(DateUtils.parseDate(monthfirstDay)));

		System.out.println(checkTime("2020-01-21 14:12:61"));
//		System.out.println(formatDate(new Date()));
//		// System.out.println(new Date());
//		// System.out.println("我们"+ "\n" + "好孩子");
//		// Date date = new Date();
//		// System.out.println(date.getTime());
//		List<String> x = new ArrayList<String>();
//		x.add("1");
//		x.add("2");
//		String a = Joiner.on(",").join(x);
//		System.out.println(a);

//		System.out.println(DateUtil.formatDate(DateUtil.beginOfMonth(new Date()))); // 本月第一天
//		System.out.println(DateUtil.formatDate(DateUtil.beginOfWeek(new Date()))); // 本周第一天
//		System.out.println(DateUtil.formatDate(new Date())); // 今天
//		System.out.println(DateUtil.formatDate(DateUtil.beginOfWeek(DateUtil.lastWeek()))); // 上周第一天
//		System.out.println(DateUtil.formatDate(DateUtil.endOfWeek(DateUtil.lastWeek())));// 上周最后一天
//		System.out.println(DateUtil.formatDate(DateUtil.beginOfMonth(DateUtil.lastMonth())));// 上月第一天
//		System.out.println(DateUtil.formatDate(DateUtil.endOfMonth(DateUtil.lastMonth())));// 上月最后一天
//		Map<String,Object> map = new HashMap<>();
//		BigDecimal big = new BigDecimal(2.31);
//		map.put("a",big);
//		BigDecimal c = new BigDecimal(String.valueOf(map.get("a").toString()));
//		String d = "0次/共0分钟";
//
//		Date time = DateUtils.parseDate("01:00:00","HH:mm:ss");
//
//		Date time2 = DateUtil.offsetMinute(time,-5);
//		System.out.println(DateUtils.formatDate(time2,"HH:mm"));
//
//		String date = "2019-01-26";
//		System.out.println(date+":星期"+DateUtil.dayOfWeek(DateUtil.parseDate(date)));
//
//		System.out.println(DateUtil.formatDate(DateUtil.endOfWeek(new Date())));
//
//		String sourceDate = "2019-01-31";
//		System.out.println(DateUtils.parseDate(sourceDate));
//		try {
//			System.out.println(DateUtils.parseDate(sourceDate,"yyyy-MM-dd"));
//		}catch (ParseException e) {
//		}
//
////		System.out.println(DateUtils.formatDate(DateUtil.date()));
//		if(DateUtils.dateCompare("2019-08-31",DateUtils.getDate())){
//			System.out.println("任务进行中");
//		}else{
//			System.out.println("任务未开始");
//		}
		System.out.println(DateUtils.dateCompare("2020-02-19",DateUtils.getDate()));

//		System.out.println(DateUtils.getSpecifiedDayBefore(DateUtils.getDate(),2));
//		List<Date> dateList = DateUtils.getBetweenDates(DateUtils.parseDate(DateUtils.getSpecifiedDayBefore(DateUtils.getDate(),2)),new Date());
//		for (Date date: dateList ) {
//			System.out.println(DateUtils.formatDate(date).replaceAll("-",""));
//		}
//		System.out.println("日期格式:"+ checkDate("201-9-08-31"));
	}
}
