package com.bcxin.platform.service.product.impl;

import com.bcxin.platform.common.constant.DictConst;
import com.bcxin.platform.common.core.text.Convert;
import com.bcxin.platform.common.exception.V5BusinessException;
import com.bcxin.platform.common.utils.*;
import com.bcxin.platform.common.utils.bean.BeanUtils;
import com.bcxin.platform.domain.product.ProductCoupon;
import com.bcxin.platform.dto.product.ProductCouponDownloadDto;
import com.bcxin.platform.dto.product.ProductCouponDto;
import com.bcxin.platform.mapper.product.ProductCouponMapper;
import com.bcxin.platform.service.product.ProductCouponService;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import javax.annotation.Resource;
import java.util.Date;
import java.util.List;

/**
 * 产品优惠券
 *
 * @author llc
 * @date 2020-12-16
 */
@Service
@Transactional("transactionManager")
public class ProductCouponServiceImpl implements ProductCouponService {

    @Resource
    private ProductCouponMapper productCouponMapper;

    @Resource
    private IdWorker idWorker;


    /**
     * 校验优惠券
     * @return
     * @author llc
     * @date 2020-01-15
     */
    @Override
    public Result checkProductCoupon(ProductCoupon productCoupon) throws V5BusinessException {
        if (productCoupon.getProductCouponId() == null) {
            return Result.fail("优惠券码不能为空");
        }
        if (StringUtils.isEmpty(productCoupon.getProductCategoryType())) {
            return Result.fail("产品范畴不能为空");
        }
        productCoupon =  productCouponMapper.getProductCouponById(productCoupon.getProductCouponId(),productCoupon.getProductCategoryType());
        if(productCoupon == null){
            return Result.fail("优惠券码无效");
        }else{
            Date now = new Date();
            if(productCoupon.getStartTime().after(now)){
                return Result.fail("优惠券未生效");
            }
            if(DictConst.COUPONSTATUS_YSY.equals(productCoupon.getCouponStatus())){
                return Result.fail("优惠券已使用");
            }
            if(productCoupon.getExpireTime().before(now)){
                return Result.fail("优惠券已过期");
            }
            return Result.success(Result.SUCCESS_QUERY_MSG,productCoupon);
        }
    }

    /**
     * 查询产品优惠券
     *
     * @param productCouponId 产品优惠券ID
     * @return 产品优惠券
     */
    @Override
    public ProductCoupon selectProductCouponById(Long productCouponId){
        return productCouponMapper.selectProductCouponById(productCouponId);
    }

    /**
     * 查询产品优惠券列表
     *
     * @param productCoupon 产品优惠券
     * @return 产品优惠券
     */
    @Override
    public List<ProductCouponDownloadDto> selectProductCouponList(ProductCouponDto productCoupon){
        return productCouponMapper.selectProductCouponList(productCoupon);
    }

    /**
     * 修改产品优惠券
     *
     * @param productCoupon 产品优惠券
     * @return 结果
     */
    @Override
    public int updateProductCoupon(ProductCoupon productCoupon){
        if(productCoupon.getProductCouponId() == null){
            productCoupon.setCreateTime(DateUtils.getNowDate());
            productCoupon.setCreateBy(ShiroUtils.getUserId());
            productCoupon.setExpireTime(DateUtils.addSeconds(DateUtils.addDays(productCoupon.getStartTime(),productCoupon.getExpireDays()),-1));
            productCoupon.setProductCategoryType("11");//云会议
            productCoupon.setCouponStatus(DictConst.COUPONSTATUS_WSY);
            productCoupon.setProductCouponId(idWorker.nextId());
        }else{
            ProductCoupon dbProductCoupon = productCouponMapper.selectProductCouponById(productCoupon.getProductCouponId());
            BeanUtils.copyPropertiesIgnore(productCoupon,dbProductCoupon,true);
            BeanUtils.copyPropertiesIgnore(dbProductCoupon,productCoupon,false);
        }
        productCoupon.setUpdateTime(DateUtils.getNowDate());
        return productCouponMapper.save(productCoupon);
    }

    /**
     * 删除产品优惠券对象
     *
     * @param ids 需要删除的数据ID
     * @return 结果
     */
    @Override
    public int deleteProductCouponByIds(String ids){
        return productCouponMapper.deleteProductCouponByIds(Convert.toStrArray(ids));
    }

    /**
     * 删除产品优惠券信息
     *
     * @param productCouponId 产品优惠券ID
     * @return 结果
     */
    @Override
    public int deleteProductCouponById(Long productCouponId){
        return productCouponMapper.deleteProductCouponById(productCouponId);
    }
}