package com.bcxin.platform.service.order;

import com.alibaba.fastjson.JSON;
import com.bcxin.platform.common.constant.DictConst;
import com.bcxin.platform.common.exception.PlatFormBusinessException;
import com.bcxin.platform.common.exception.V5BusinessException;
import com.bcxin.platform.common.utils.DateUtil;
import com.bcxin.platform.common.utils.IdWorker;
import com.bcxin.platform.common.utils.Result;
import com.bcxin.platform.common.utils.StringUtils;
import com.bcxin.platform.domain.order.ComOrder;
import com.bcxin.platform.domain.order.ComOrderConfirmPay;
import com.bcxin.platform.domain.order.ComOrderDetail;
import com.bcxin.platform.domain.product.ProductCoupon;
import com.bcxin.platform.dto.meeting.MeetingOrderRequestDTO;
import com.bcxin.platform.dto.meeting.OrderDTO;
import com.bcxin.platform.dto.meeting.ResourceItemDTO;
import com.bcxin.platform.dto.order.ComOrderDto;
import com.bcxin.platform.dto.order.ComOrderPayDTO;
import com.bcxin.platform.dto.order.ComOrderVo;
import com.bcxin.platform.mapper.order.ComOrderConfirmPayMapper;
import com.bcxin.platform.mapper.order.ComOrderDetailMapper;
import com.bcxin.platform.mapper.order.ComOrderMapper;
import com.bcxin.platform.mapper.product.ProductCouponMapper;
import com.bcxin.platform.service.common.CommonService;
import com.bcxin.platform.service.meeting.HuaweiMeetMatchService;
import com.bcxin.platform.service.system.ISysConfigService;
import com.bcxin.platform.service.wallet.ComWalletPaymentService;
import com.bcxin.platform.util.ObjectUtils;
import com.bcxin.platform.util.PageInfoUtils;
import com.bcxin.platform.util.constants.CommonConst;
import com.bcxin.platform.util.constants.DictMessageTypeConst;
import com.bcxin.platform.util.constants.MsgConst;
import com.github.pagehelper.PageHelper;
import com.google.common.collect.Lists;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import javax.annotation.Resource;
import java.math.BigDecimal;
import java.util.*;
import java.util.stream.Collectors;

/**
 * 配置产品
 *
 * @author llc
 * @date 2020-01-15
 */
@Service
@Transactional("transactionManager")
public class ComOrderServiceImpl implements ComOrderService {

    @Resource
    private ComOrderMapper comOrderMapper;

    @Resource
    private ComOrderDetailMapper comOrderDetailMapper;

    @Resource
    private IdWorker idWorker;

    @Resource
    private ComWalletPaymentService comWalletPaymentService;

    @Resource
    private ISysConfigService sysConfigService;

    @Resource
    private CommonService commonService;

    @Resource
    private ProductCouponMapper productCouponMapper;

    @Resource
    private ComOrderConfirmPayMapper comOrderConfirmPayMapper;

    @Autowired
    private HuaweiMeetMatchService comHuaweiMeetMatchService;

    /**
     * 校验订单
     *
     * @return
     * @throws Exception 作为服务消费者,需注解@TxTransaction(isStart=true)，开启分布式事务
     * @author llc
     * @date 2020-01-20
     */
    @Override
    public Result checkOrder(ComOrderDto dto) throws V5BusinessException {
        if (dto.getActualPayAmount() == null) {
            return Result.fail("支付金额不能为空");
        }
        //if (BigDecimal.ZERO.compareTo(dto.getActualPayAmount()) == 0) {
        //    return Result.fail("支付金额不能为0");
        //}
        /*** 调用支付接口 ***/
        ComOrderPayDTO orderPayDto = new ComOrderPayDTO();
        orderPayDto.setActualPayAmount(dto.getActualPayAmount());
        String meetIncomeComId = sysConfigService.selectConfigByKey(CommonConst.MEET_INCOME_COMID);
        orderPayDto.setInComId(Long.parseLong(meetIncomeComId));  // 会议收入企业ID
        orderPayDto.setComId(dto.getComId());

        Result result = comWalletPaymentService.payValidate(orderPayDto);
        return result ;
    }


    /**
     * 提交订单
     *
     * @return
     * @throws Exception
     * @author llc
     * @date 2020-01-15
     */
    @Override
    public Result saveOrder(ComOrderDto dto) throws V5BusinessException {
        if (StringUtils.isEmpty(dto.getActiveCode())) {
            return Result.fail("验证码不能为空");
        }
        if (StringUtils.isEmpty(dto.getOrderDetailListStr())) {
            return Result.fail("订单详情不能为空");
        }
        if (StringUtils.isEmpty(dto.getProductCategoryType())) {
            return Result.fail("产品范畴不能为空");
        }
        if (dto.getAmount() == null) {
            return Result.fail("订单金额不能为空");
        }
        if (BigDecimal.ZERO.compareTo(dto.getAmount()) == 0) {
            return Result.fail("订单金额不能为0");
        }
        if (dto.getActualPayAmount() == null) {
            return Result.fail("支付金额不能为空");
        }
        //if (BigDecimal.ZERO.compareTo(dto.getActualPayAmount()) == 0) {
        //    return Result.fail("支付金额不能为0");
        //}

        /*** 如果存在优惠券，需要再次校验优惠券 ***/
        ProductCoupon productCoupon =  new ProductCoupon();
        if(dto.getProductCouponId() != null){
            productCoupon = productCouponMapper.getProductCouponById(dto.getProductCouponId(),dto.getProductCategoryType());
            if(productCoupon ==  null){
                return Result.fail("优惠券码无效");
            }else{
                Date now = new Date();
                if(productCoupon.getStartTime().after(now)){
                    return Result.fail("优惠券未生效");
                }
                if(DictConst.COUPONSTATUS_YSY.equals(productCoupon.getCouponStatus())){
                    return Result.fail("优惠券已使用");
                }
                if(productCoupon.getExpireTime().before(now)){
                    return Result.fail("优惠券已过期");
                }
            }
        }

        Long comOrderId = idWorker.nextId();
        dto.setComOrderId(comOrderId);
        List<ComOrderDetail> orderDetailList = JSON.parseArray(dto.getOrderDetailListStr(), ComOrderDetail.class);
        BigDecimal sumAmount = BigDecimal.ZERO;
        for (ComOrderDetail comOrderDetail : orderDetailList) {
            comOrderDetail.setComOrderDetailId(idWorker.nextId());
            comOrderDetail.setComOrderId(comOrderId);
            comOrderDetail.setComId(dto.getComId());
            comOrderDetail.setCreateTime(dto.getCreateTime());
            comOrderDetail.setCreateBy(dto.getCreateBy());
            String type = comOrderDetail.getProductBuyModeId().substring(6,8);
            Integer dateType = "01".equals(type)? DateUtil.DATATYPE_MONTH:"02".equals(type)?DateUtil.DATATYPE_YEAR:0;
            Date expireTime = null;
            if(dateType!=0){
                try {
                    Date nowDate = DateUtil.convertStringToDate(DateUtil.getCurrentDate());
                    expireTime = DateUtil.dateAdd2(dateType,nowDate,comOrderDetail.getDuration());
                } catch (Exception e) {
                    throw new V5BusinessException(Result.BUSINESS_ERROR,"到期时间加载失败，ERROR:"+e.getMessage());
                }
            }
            comOrderDetail.setExpireTime(expireTime);
            sumAmount = sumAmount.add(comOrderDetail.getAmount());
        }
        if (!(sumAmount.compareTo(dto.getAmount()) == 0)) {
            return Result.fail("订单总额和订单明细金额之和不一致");
        }
        /*** 调用支付接口 ***/
        ComOrderPayDTO orderPayDto = new ComOrderPayDTO();
        orderPayDto.setActualPayAmount(dto.getActualPayAmount());
        String meetIncomeComId = sysConfigService.selectConfigByKey(CommonConst.MEET_INCOME_COMID);
        orderPayDto.setInComId(Long.parseLong(meetIncomeComId));  // 会议收入企业ID
        orderPayDto.setComId(dto.getComId());
        orderPayDto.setComment("");
        orderPayDto.setActiveCode(dto.getActiveCode());

        Result result = comWalletPaymentService.actualPay(orderPayDto);
        if (result != null && Result.SUCCESS.equals(result.getRetType())) {

            /*** 保存订单 ***/
            dto.setOrderStatus(CommonConst.Y); // 订单状态为已支付
            dto.setOrderDeployStatus(CommonConst.N); // 订单默认未分配
            dto.setPurchaseStatus(CommonConst.N);// 订单默认未采购
            comOrderMapper.insertComOrder(dto);
            int count = comOrderDetailMapper.insertComOrderDetailList(orderDetailList);
            if (count != orderDetailList.size()) {
                return Result.success("生成订单出错");
            }

            /*** 更新优惠券 ***/
            if(productCoupon.getProductCouponId() != null){
                productCoupon.setCouponStatus(DictConst.COUPONSTATUS_YSY); // 已使用
                productCoupon.setComId(dto.getComId());
                productCoupon.setUseTime(new Date());
                productCoupon.setComOrderId(comOrderId);
                productCouponMapper.updateProductCouponById(productCoupon);
            }

            /*** 发送邮件给采购人员 ***/
            String title = MsgConst.TYPE_010301_ORDER_ADD_TITLE;
            String content = MsgConst.TYPE_010301_ORDER_ADD_CONTENT;
            //给订单处理人发邮件
            String emails = sysConfigService.selectConfigByKey(CommonConst.ORDER_YW_EMAIL);
            commonService.sendEmailMessage(title,content, DictMessageTypeConst.MESSAGETYPE_010301,emails);
            return Result.success("支付成功");
        } else {
            return Result.fail(result.getMsg());
        }

    }

    @Override
    public Result getOrderList(ComOrderDto dto) throws V5BusinessException {

        if (dto.getPageNumber() == null) {
            return Result.fail("第几页不能为空");
        }
        if (dto.getPageSize() == null) {
            return Result.fail("每页多少条不能为空");
        }
        PageHelper.startPage(dto.getPageNumber(), dto.getPageSize());
        List<ComOrderDto> orderList = comOrderMapper.getComOrderList(dto);

        if (orderList != null && orderList.size() > 0) {
            List<Long> orderIdList = new ArrayList<>();
            for (ComOrderDto comOrderDto : orderList) {
                orderIdList.add(comOrderDto.getComOrderId());
            }

            List<Map<String, Object>> orderDetailList = comOrderDetailMapper.getOrderDetailListByOrderIdS(orderIdList);

            /*** 把订单详情放入订单列表 ***/
            for (ComOrderDto orderDto : orderList) {
                List<Map<String, Object>> list = new ArrayList<>();
                for(Map<String, Object> map : orderDetailList){
                    if(ObjectUtils.equals(orderDto.getComOrderId(),map.get("comOrderId"))){
                        list.add(map);
                    }
                }
                orderDto.setOrderDetailList(list);
            }
        }

        return Result.success(Result.SUCCESS_QUERY_MSG, new PageInfoUtils(orderList));
    }

    @Override
    public List<Map<String, String>> getOrderDetailListByAssignedOrgId(ComOrderDto dto) throws PlatFormBusinessException {
        return comOrderDetailMapper.getOrderDetailListByAssignedOrgId(dto.getAssignedOrgId());
    }


    /**
     * 获取订单列表
     *
     * @return
     * @author llc
     * @date 2020-01-17
     */
    @Override
    public List<ComOrderVo>  orderList(ComOrderVo dto) throws PlatFormBusinessException {
        List<ComOrderVo> orderList = comOrderMapper.getOrderList(dto);
        if (orderList != null && orderList.size() > 0) {
            List<Long> orderIdList = new ArrayList<>();
            for (ComOrderVo comOrderDto : orderList) {
                orderIdList.add(comOrderDto.getComOrderId());
            }

            List<Map<String, Object>> orderDetailList = comOrderDetailMapper.getOrderDetailListByOrderIdS(orderIdList);

            /*** 把订单详情放入订单列表 ***/
            for (ComOrderVo orderDto : orderList) {
                List<Map<String, Object>> list = new ArrayList<>();
                for (Map<String, Object> map : orderDetailList) {
                    if (ObjectUtils.equals(orderDto.getComOrderId(), map.get("comOrderId"))) {
                        list.add(map);
                    }
                }
                orderDto.setOrderDetailList(list);
            }
        }

        //给orderDetailListStr赋值
        orderList.stream().filter(s -> {
            StringBuilder a = new StringBuilder();
            for (int i = 0; i < s.getOrderDetailList().size(); i++) {
                Map<String, Object> map = s.getOrderDetailList().get(i);
                a.append(map.get("productTypeName").toString()).append(" (").append(map.get("productName")).append(") ").append(map.get("eachAmount")).append("").append(map.get("unit")).append(" ").append("数量").append(map.get("num")).append("个").append("<br>");
            }
            s.setOrderDetailListStr(a.toString());
            return true;
        }).collect(Collectors.toList());
        //返回给前端

        return orderList;
    }


    @Override
    public Result changeDeployStatus(ComOrderDto dto) throws Exception {
        comOrderMapper.changeDeployStatus(dto);
        return null;
    }

    /**
     * 定时任务发邮件(未完成订单)
     *
     * @return
     * @throws Exception
     */
    public Integer sendEmailJob() throws Exception {
        return comOrderMapper.selectCount("0");
    }

    /**
     * 获取统计信息
     *
     * @author wangjianjun
     * @date 2020/4/1
     */
    @Override
    public Result getOrderStatistic(ComOrderDto dto) {
        return Result.success("", comOrderMapper.getComOrderStatistic(dto));
    }

    /**
     * <b> 根据订单Id查询订单支付信息 </b>
     * @author ZXF
     * @create 2020/06/01 0001 15:47
     * @version
     * @注意事项 </b>
     */
    @Override
    public Object findOrderPayListByOrderId(ComOrderDto dto) {
        return comOrderConfirmPayMapper.selectByComOrderId(dto.getComOrderId());
    }

    /**
     * <b> 采购资源 </b>
     * @author ZXF
     * @create 2020/05/20 0020 14:46
     * @version
     * @注意事项 </b>
     */
    @Override
    public Result purchasingResource(ComOrderDto dto,Long perId) throws PlatFormBusinessException {
        if(dto.getComOrderId() == null){
            return Result.fail("comOrderId不能为空");
        }
        List<Map<String, String>> mapList = comOrderDetailMapper.getOrderDetailListByOrderId(dto.getComOrderId());
        if(mapList.size()==0){
            return Result.fail("未查询到订单详情");
        }
        MeetingOrderRequestDTO morDTO = new MeetingOrderRequestDTO();
        List<OrderDTO> order_list = Lists.newArrayList();
        OrderDTO oDTO;
        List<ResourceItemDTO> res_list;
        ResourceItemDTO riDTO;
        Long comId = null;
        for(Map<String, String> map : mapList){
            if(comId == null){
                comId = Long.parseLong(map.get("comId"));
            }
            oDTO = new OrderDTO();
            oDTO.setDuration(map.get("duration"));
            if(StringUtils.isNotEmpty(map.get("delivery_address_id"))){
                oDTO.setDelivery_address_id(map.get("delivery_address_id"));
            }
            oDTO.setDuration_type(map.get("duration_type"));
            res_list = Lists.newArrayList();
            riDTO = new ResourceItemDTO();
            riDTO.setService_type(map.get("service_type"));
            riDTO.setRes_spec_code(map.get("res_spec_code"));
            riDTO.setRes_size(map.get("res_size"));
            riDTO.setRes_type(map.get("res_type"));
            res_list.add(riDTO);
            oDTO.setRes_list(res_list);
            order_list.add(oDTO);
        }
        morDTO.setOrder_list(order_list);
        Result result = comHuaweiMeetMatchService.purchasingResource(morDTO);
        if(Result.SUCCESS.equals(result.getRetType())){
            String orderNos = String.valueOf(result.getData());
            if(StringUtils.isNotEmpty(orderNos)){
                comOrderConfirmPayMapper.batchInsert(dto.getComOrderId(), Arrays.asList(orderNos.split(",")));
                ComOrder order = new ComOrder();
                order.setComOrderId(dto.getComOrderId());
                order.setPurchaseStatus(DictConst.PURCHASE_STATUS_WZF);
                order.setUpdateBy(perId);
                order.setUpdateTime(new Date());
                comOrderMapper.update(order);
            }else{
                return Result.fail("华为下单未返回订单号！");
            }
        }
        return result;
    }

    /**
     * <b> 采购资源确认支付 </b>
     * @author ZXF
     * @create 2020/05/20 0020 14:46
     * @version
     * @注意事项 </b>
     */
    @Override
    public Result orderConfirmPay(ComOrderVo dto,Long perId) throws PlatFormBusinessException {
        if(dto.getComOrderId() == null){
            return Result.fail("comOrderId不能为空");
        }
        Result result = comHuaweiMeetMatchService.orderPayByOrderNo(dto.getOrderNo());
        if(Result.SUCCESS.equals(result.getRetType()) && result.getData()!=null){
            ComOrderConfirmPay pay = new ComOrderConfirmPay();
            pay.setComOrderId(dto.getComOrderId());
            pay.setOrderNo(dto.getOrderNo());
            pay.setTradeNo(String.valueOf(result.getData()));
            comOrderConfirmPayMapper.update(pay);
            List<String> list = comOrderConfirmPayMapper.findUnConfirmPayOrderNosByOrderId(dto.getComOrderId());
            if(list.size()==0){
                ComOrder order = new ComOrder();
                order.setComOrderId(dto.getComOrderId());
                order.setPurchaseStatus(DictConst.PURCHASE_STATUS_YZF);
                order.setUpdateBy(perId);
                order.setUpdateTime(new Date());
                comOrderMapper.update(order);
                return Result.success("订单支付成功","1");
            }
            return Result.success("订单支付成功");
        }else{
            return Result.fail(result.getMsg());
        }
    }

    /**
     * <b> 分配资源 </b>
     * @author ZXF
     * @create 2020/05/20 0020 14:47
     * @version
     * @注意事项 </b>
     */
    @Override
    public Result allocationResource(ComOrderDto dto,Long perId) throws PlatFormBusinessException {
        if(dto.getComOrderId() == null){
            return Result.fail("comOrderId不能为空");
        }

        if(dto.getAssignedOrgId() == null){
            return Result.fail("assignedOrgId不能为空");
        }
        /*List<Map<String, String>> mapList = comOrderDetailMapper.findOrderAndProductListByOrderId(dto.getComOrderId());
        if(mapList.size()==0){
            return Result.fail("未查询到订单详情");
        }
        Map<String, String> maps;
        List<Map<String, String>> mapsList = Lists.newArrayList();
        for (Map<String, String> map:mapList) {
            //是录播空间的
            if(StringUtils.isNotEmpty(map.get("num_other"))){
                int count = Integer.parseInt(map.get("count"));
                if(count>1){
                    //如果是录播，详情中的数量当做产品数，拆成一条条的记录做报文参数，空间num_other字段的值来填充数量count字段
                    for(int i=0;i<count-1;i++){
                        maps = Maps.newHashMap();
                        maps.put("id",map.get("id")+String.valueOf(i));
                        maps.put("type",map.get("type"));
                        maps.put("editable",map.get("editable"));
                        maps.put("expireDate",map.get("expireDate"));
                        maps.put("count",map.get("num_other"));
                        mapsList.add(maps);
                    }
                }
                map.put("count",map.get("num_other"));
            }
            map.remove("num_other");
        }
        mapList.addAll(mapsList);
        Result result = comHuaweiMeetMatchService.addResourcesByHuaweiComId(mapList,dto.getAssignedOrgId());
        if(Result.SUCCESS.equals(result.getRetType())){*/
        ComOrder order = new ComOrder();
        order.setComOrderId(dto.getComOrderId());
        order.setAssignedOrgId(dto.getAssignedOrgId());
        order.setAssignedOrgType(dto.getAssignedOrgType());
        order.setOrderDeployStatus(DictConst.ORDER_DEPLOY_STATUS_Y);
        order.setUpdateBy(perId);
        order.setUpdateTime(new Date());
        comOrderMapper.update(order);
        /*}
        return result;*/
        return Result.success(Result.SUCCESS_MSG);
    }

    /**
     * <b> 定时查询隔天到期的企业资源，请求接口做删除 </b>
     * @author ZXF
     * @create 2020/05/28 0028 10:51
     * @version
     * @注意事项 </b>
     */
    @Override
    public Result delComResource() throws PlatFormBusinessException {

        List<Map<String, String>> mapList = comOrderDetailMapper.findDueOrderList();
        if(mapList.size()==0){
            return Result.fail("未查询到订单详情");
        }
        List<String> list = Lists.newArrayList();
        List<String> failList = Lists.newArrayList();
        String huaweiComId = "";
        int num = 0;
        for (Map<String, String> map:mapList) {
            num ++;
            if(StringUtils.isNotEmpty(huaweiComId) && !huaweiComId.equals(map.get("huaweiComId"))){
                getDelResource(list,failList,huaweiComId);
                huaweiComId = map.get("huaweiComId");
                list = Lists.newArrayList();
            }
            if(StringUtils.isEmpty(huaweiComId)){
                huaweiComId = map.get("huaweiComId");
            }
            if("1".equals(map.get("isRecord"))){
                int count = Integer.parseInt(map.get("count"));
                if(count>1){
                    for(int i=0;i<count-1;i++){
                        String id = map.get("id")+String.valueOf(i);
                        list.add(id);
                    }
                }
            }
            list.add(map.get("id"));
            if(num == mapList.size()){
                getDelResource(list,failList,huaweiComId);
            }
        }
        //如果存在失败的记录就查询订单号，发送到管理员邮件，让其手动删除企业资源
        if(failList.size()>0){
            List<Map<String,String>> details = comOrderDetailMapper.getAssignedOrgNameByComOrderDetailIds(failList);
            //发邮件
            String emails=commonService.getSystemConfig("ORDER_YW_EMAIL");
            String title = "企业资源即将到期未能完成释放，请及时处理！";
            String content = "定时企业资源回收过程中，部分资源未能完成释放，请及时登录华为云平台对相关已分配资源进行删除！\n涉及资源信息：\n";
            StringBuffer tal = new StringBuffer();
            for(Map map :details){
                tal.append("        "+map.get("assignedOrgName")+"："+map.get("productName")+"（"+map.get("num")+"）");
            }
            content = content + tal;
            commonService.sendEmailMessage(title, content, "010301", emails);
        }
        return Result.success(Result.SUCCESS_MSG);
    }

    private void getDelResource(List<String> list,List<String> failList,String huaweiComId) throws PlatFormBusinessException {
        String[] arr = new String[list.size()];
        list.toArray(arr);
        Result result = comHuaweiMeetMatchService.delResource(arr,Long.parseLong(huaweiComId));
        //处理删除资源接口返回结果，如果返回失败记录失败的ID
        if(!Result.SUCCESS.equals(result.getRetType())){
            failList.addAll(list);
        }
    }
}