package com.bcxin.platform.service.meeting;

import cn.hutool.http.HttpUtil;
import com.alibaba.fastjson.JSON;
import com.bcxin.platform.common.exception.V5BusinessException;
import com.bcxin.platform.common.utils.Result;
import com.bcxin.platform.domain.company.ComBaseInfo;
import com.bcxin.platform.domain.meeting.ComHuaweiMeetMatch;
import com.bcxin.platform.domain.product.PoliceHuaweiMeetMatch;
import com.bcxin.platform.domain.system.SysPolice;
import com.bcxin.platform.dto.meeting.*;
import com.bcxin.platform.mapper.company.ComBaseInfoMapper;
import com.bcxin.platform.mapper.meeting.ComHuaweiMeetMatchMapper;
import com.bcxin.platform.mapper.meeting.PoliceHuaweiMeetMatchMapper;
import com.bcxin.platform.mapper.order.SysPoliceMapper;
import com.bcxin.platform.service.common.CommonService;
import com.bcxin.platform.util.constants.HuaweiMeetConst;
import com.bcxin.platform.util.http.MD5Util;
import com.bcxin.platform.util.huawei.HttpBuildUtil;
import com.bcxin.platform.util.huawei.RestRequest;
import com.bcxin.platform.util.huawei.RestResponse;
import com.fasterxml.jackson.databind.ObjectMapper;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import javax.annotation.Resource;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.UUID;

/**
 * 企业华为云会议映射
 *
 * @author wangjianjun
 * @date 2020-02-03
 */
@Service
@Transactional("transactionManager")
public class HuaweiMeetMatchServiceImpl implements HuaweiMeetMatchService {

    /**
     * 日志对象
     */
    protected Logger logger = LoggerFactory.getLogger(getClass());

    @Resource
    private ComHuaweiMeetMatchMapper comHuaweiMeetMatchMapper;
    @Resource
    private PoliceHuaweiMeetMatchMapper policeHuaweiMeetMatchMapper;
    @Resource
    private SysPoliceMapper sysPoliceMapper;
    @Resource
    private ComBaseInfoMapper comBaseInfoMapper;
    @Resource
    private CommonService commonService;

    /**
     * 获取企业华为云会议管理资源 // PageInfoUtils
     *
     * @param comHuaweiMeetMatch
     * @return
     * @auth wangjianjun
     * @date 2020-02-03
     */
    @Override
    public List<Map> getComHuaweiMeet(ComHuaweiMeetMatchDto comHuaweiMeetMatch) throws Exception {
        return comHuaweiMeetMatchMapper.findMapListByName(comHuaweiMeetMatch);
    }

    /**
     * 删除企业华为云会议管理资源
     *
     * @param comHuaweiMeetMatch
     * @return
     * @auth wangjianjun
     * @date 2020-02-03
     */
    @Override
    public Result delComHuaweiMeet(ComHuaweiMeetMatch comHuaweiMeetMatch) throws V5BusinessException {
        Result result = delHuaweiManage(comHuaweiMeetMatch.getHuaweiComId());
        if(Result.SUCCESS.equals(result.getRetType())){
            comHuaweiMeetMatchMapper.delete(comHuaweiMeetMatch.getComId().longValue());
        }
        return result;
    }

    /**
     * <b> 删除公安华为云会议管理资源 </b>
     * @author ZXF
     * @create 2020/05/25 0025 10:56
     * @version
     * @注意事项 </b>
     */
    @Override
    public Result delPoliceHuaweiMeet(PoliceHuaweiMeetMatch policeHuaweiMeetMatch) throws V5BusinessException {
        Result result = delHuaweiManage(policeHuaweiMeetMatch.getHuaweiComId());
        if(Result.SUCCESS.equals(result.getRetType())){
            policeHuaweiMeetMatchMapper.delete(policeHuaweiMeetMatch.getPoliceId().longValue());
        }
        return result;
    }

    /**
     * 新增华为云企业账号
     *
     * @param comHuaweiMeetMatch
     * @return
     * @auth wangjianjun
     * @date 2020-02-03
     */
    @Override
    public Result saveComHuaweiMeet(ComHuaweiMeetMatch comHuaweiMeetMatch) throws V5BusinessException {
        ComBaseInfo comBaseInfo = comBaseInfoMapper.selectComBaseInfoById(comHuaweiMeetMatch.getComId());
        if(comBaseInfo == null){
            return Result.fail("comId无效");
        }
        if (comHuaweiMeetMatchMapper.selectCount(comHuaweiMeetMatch.getComId().longValue()) > 0) {
            return Result.fail("华为云企业账号已存在");
        }
        //自定义的账号密码
        String comCerNo = comBaseInfo.getComCerNo();
        String account = "HW_"+comCerNo;
        String pwd = MD5Util.string2MD5("Saas@"+comCerNo.substring(comCerNo.length()-6,comCerNo.length())).substring(4,16);
        Result result = buildCompanyByAdmin(comBaseInfo.getComName(),account,pwd);
        if(Result.SUCCESS.equals(result.getRetType())){
            String Id = String.valueOf(result.getData());
            comHuaweiMeetMatch.setHuaweiManageUserName(account);
            comHuaweiMeetMatch.setHuaweiComId(Long.parseLong(Id));
            comHuaweiMeetMatch.setHuaweiManagePassWord(pwd);
            comHuaweiMeetMatchMapper.insert(comHuaweiMeetMatch);
            return Result.success(Result.SUCCESS_QUERY_MSG);
        }
        return result;
    }

    /**
     * <b> 新增（修改）公安华为云会议 </b>
     * @author ZXF
     * @create 2020/05/22 0022 15:49
     * @version
     * @注意事项 </b>
     */
    @Override
    public Result savePoliceHuaweiMeet(PoliceHuaweiMeetMatch policeHuaweiMeetMatch) throws V5BusinessException {
        SysPolice sysPolice = sysPoliceMapper.getByPrimaryKey(policeHuaweiMeetMatch.getPoliceId());
        if(sysPolice == null){
            return Result.fail("policeId无效");
        }
        if (policeHuaweiMeetMatchMapper.selectCount(policeHuaweiMeetMatch.getPoliceId().longValue()) > 0) {
            return Result.fail("华为云公安账号已存在");
        }
        //自定义的账号密码
        String policeCerNo = UUID.randomUUID().toString().replaceAll("-","");
        String account = "HW_"+policeCerNo.substring(3,18);
        String pwd = MD5Util.string2MD5("Saas@"+policeCerNo.substring(3,18)).substring(4,16);
        Result result = buildCompanyByAdmin(sysPolice.getPoliceName(),account,pwd);
        if(Result.SUCCESS.equals(result.getRetType())){
            String Id = String.valueOf(result.getData());
            policeHuaweiMeetMatch.setHuaweiManageUserName(account);
            policeHuaweiMeetMatch.setHuaweiComId(Long.parseLong(Id));
            policeHuaweiMeetMatch.setHuaweiManagePassWord(pwd);
            policeHuaweiMeetMatchMapper.insert(policeHuaweiMeetMatch);
            return Result.success(Result.SUCCESS_QUERY_MSG);
        }
        return result;
    }

    /**
     * <b> 查询华为账号的企业信息 </b>
     * @author ZXF
     * @create 2020/05/21 0021 10:33
     * @version
     * @注意事项 </b>
     */
    @Override
    public List<Map> findComHuaweiMeetMatchList() {
        return comHuaweiMeetMatchMapper.findComHuaweiMeetMatchList();
    }

    /**
     * 刷新登陆
     *
     * @param comHuaweiMeetMatch
     * @return
     * @auth wangjianjun
     * @date 2020-05-11
     */
    @Override
    public Result refreshComHuaweiMeet(ComHuaweiMeetMatch comHuaweiMeetMatch) throws Exception {
        Long comId = comHuaweiMeetMatch.getComId();
        HashMap<String, Object> paramMap = new HashMap<>();
        paramMap.put("key", "huaweiMeetCom"+comId);
        String result = HttpUtil.get(
                commonService.getSystemConfig("SYS_PIC_HTTP_ADDR") + "/public/common/delete-redis-by-key", paramMap);
        if (StringUtils.isEmpty(result) || !result.contains("msg")) {
            throw new Exception("调用saas接口返回出错");
        }
        HashMap<String, Object> resultMap = new ObjectMapper().readValue(result, HashMap.class);
        if (StringUtils.isNotBlank(resultMap.get("msg").toString())) {
            throw new Exception("调用saas接口返回出错");
        }
        return Result.success(Result.SUCCESS_QUERY_MSG);
    }

    /**
     * <b> 会议资源订购 </b>
     * @author ZXF
     * @create 2020/05/20 0020 14:57
     * @version
     * @注意事项 </b>
     */
    @Override
    public Result purchasingResource(MeetingOrderRequestDTO dto) throws V5BusinessException {
        /*** 获取华为云会议的accessToken ***/
        String accessToken = commonService.getHuaweiMeetIAMTokens(28814466L);
        MeetingExtInfoDTO meiDTO = new MeetingExtInfoDTO();
        meiDTO.setCustomer_id(commonService.getSystemConfig("MEET_LOGIN_USER_ID"));
        dto.setExtend_info(meiDTO);

        /*** 调用华为云会议接口 ***/
        String url = HuaweiMeetConst.GATEWAY_URL + HuaweiMeetConst.PURCHASING_RESOURCE;
        Map<String, String> headMap = new HashMap<>();
        headMap.put("X-Auth-Token", accessToken);
        String param = JSON.toJSONString(dto);
        logger.info(" ========== 会议资源订购接口");
        logger.info(" ========== 请求地址："+url);
        logger.info(" ========== 请求报文："+param);
        RestRequest restRequest = HttpBuildUtil.buildRestRequest("POST", headMap, param, null);
        RestResponse response = HttpBuildUtil.sendMsg(url, restRequest);
        logger.info(" ========== 返回报文："+response.getEntity());
        //如果返回报文处理resultMsg有值，说明有失败信息
        String resultMsg = HttpBuildUtil.failMsg(response);
        if(StringUtils.isNotEmpty(resultMsg)){
            return Result.fail(resultMsg);
        }
        Map map = HttpBuildUtil.getMapData(response);
        if(map == null){
            return Result.fail("未获取到订单编码!");
        }
        return Result.success(Result.SUCCESS_MSG,map.get("order_id"));
    }

    /**
     * <b> 支付包周期产品订单 </b>
     * @author ZXF
     * @create 2020/05/29 0029 14:10
     * @version
     * @注意事项 </b>
     */
    @Override
    public Result orderPayByOrderNo(String orderNo) throws V5BusinessException {
        /*** 获取华为云会议的accessToken ***/
        String accessToken = commonService.getHuaweiMeetIAMTokens(28814466L);
        /*** 调用华为云会议接口 ***/
        String url = HuaweiMeetConst.GATEWAY_BSS_URL + HuaweiMeetConst.ORDER_PAY.replace("{domain_id}",commonService.getSystemConfig("MEET_LOGIN_USER_ID"));
        Map<String, String> data = new HashMap<>();
        data.put("orderId", orderNo);
        Map<String, String> headMap = new HashMap<>();
        headMap.put("X-Auth-Token", accessToken);
        String param = JSON.toJSONString(data);
        logger.info(" ========== 支付包周期产品订单接口");
        logger.info(" ========== 请求地址："+url);
        logger.info(" ========== 请求报文："+param);
        RestResponse response = com.bcxin.platform.util.huawei.HttpUtil.post(url,param,accessToken);
        logger.info(" ========== 返回报文："+response.getEntity());
        //如果返回报文处理resultMsg有值，说明有失败信息
        String resultMsg = HttpBuildUtil.failMsg2(response);
        if(StringUtils.isNotEmpty(resultMsg)){
            return Result.fail(resultMsg);
        }
        Map map = HttpBuildUtil.getMapData(response);
        if(map == null){
            return Result.fail("未获取到订单编码!");
        }
        return Result.success(Result.SUCCESS_MSG,map.get("tradeNo"));
    }

    /**
     * <b> 分页查询企业资源 </b>
     * @author ZXF
     * @create 2020/05/21 0021 11:09
     * @version
     * @注意事项 </b>
     */
    @Override
    public Result searchResourcesByComId(Long huaweiComId) throws V5BusinessException {
        /*** 获取华为云会议的accessToken ***/
        String accessToken = commonService.getComManageHuaweiMeetAccessToken(
                commonService.getSystemConfig("MEET_MANAGE_SP_USER")
                , commonService.getSystemConfig("MEET_MANAGE_SP_PWD")
                , "28814466");

        /*** 调用华为云会议接口 ***/
        String url = HuaweiMeetConst.GATEWAY_URL + HuaweiMeetConst.GET_RESOURCE_LIST.replace("{corpId}",String.valueOf(huaweiComId));
        Map<String, String> map = new HashMap<>();
        map.put("offset", "0");
        map.put("limit", "100");
        //订单状态，0：正常 1：到期，仅查询时返回 2：停用
        map.put("status", "0");
        Map<String, String> headMap = new HashMap<>();
        headMap.put("X-Access-Token", accessToken);
        logger.info(" ========== 会议资源订购接口");
        logger.info(" ========== 请求地址："+url);
        RestRequest restRequest = HttpBuildUtil.buildRestRequest("GET", headMap, null, map);
        RestResponse response = HttpBuildUtil.sendMsg(url, restRequest);
        logger.info(" ========== 返回报文："+response.getEntity());
        //如果返回报文处理resultMsg有值，说明有失败信息
        String resultMsg = HttpBuildUtil.failMsg(response);
        if(StringUtils.isNotEmpty(resultMsg)){
            return Result.fail(resultMsg);
        }
        return Result.success(Result.SUCCESS_MSG,response.getEntity());
    }

    /**
     * <b> 管理员创建企业 </b>
     * @author ZXF
     * @create 2020/05/21 0021 15:21
     * @version
     * @注意事项 </b>
     */
    @Override
    public Result buildCompanyByAdmin(String companyName, String account, String pwd) throws V5BusinessException {
        AddCorpDTO acDTO = new AddCorpDTO();
        AdminDTO adminInfo = new AdminDTO();
        adminInfo.setAccount(account);
        adminInfo.setPwd(pwd);
        adminInfo.setName(companyName);
        adminInfo.setEmail(commonService.getSystemConfig("MEET_BUILD_SP_EMAIL"));
        acDTO.setAdminInfo(adminInfo);
        CorpBasicDTO basicInfo = new CorpBasicDTO();
        basicInfo.setName(companyName);
        acDTO.setBasicInfo(basicInfo);
        /*** 获取华为云会议的accessToken ***/
        String accessToken = commonService.getComManageHuaweiMeetAccessToken(
                commonService.getSystemConfig("MEET_MANAGE_SP_USER")
                , commonService.getSystemConfig("MEET_MANAGE_SP_PWD")
                , "28814466");

        /*** 调用华为云会议接口 ***/
        String url = HuaweiMeetConst.GATEWAY_URL + HuaweiMeetConst.BUILD_CORP;
        String param = JSON.toJSONString(acDTO);
        Map<String, String> headMap = new HashMap<>();
        headMap.put("X-Access-Token", accessToken);
        logger.info(" ========== 管理员创建企业接口");
        logger.info(" ========== 请求地址："+url);
        logger.info(" ========== 请求报文："+param);
        RestRequest restRequest = HttpBuildUtil.buildRestRequest("POST", headMap, param, null);
        RestResponse response = HttpBuildUtil.sendMsg(url, restRequest);
        logger.info(" ========== 返回报文："+response.getEntity());
        //如果返回报文处理resultMsg有值，说明有失败信息
        String resultMsg = HttpBuildUtil.failMsg(response);
        if(StringUtils.isNotEmpty(resultMsg)){
            return Result.fail(resultMsg);
        }
        Map map = HttpBuildUtil.getMapData(response);
        if(map == null){
            return Result.fail("未获取到企业ID!");
        }
        return Result.success(Result.SUCCESS_MSG,map.get("value"));
    }

    /**
     * <b> 管理删除企业 </b>
     * @author ZXF
     * @create 2020/05/21 0021 15:21
     * @version
     * @注意事项 </b>
     */
    @Override
    public Result delHuaweiManage(Long huaweiComId) throws V5BusinessException {
        /*** 获取华为云会议的accessToken ***/
        String accessToken = commonService.getComManageHuaweiMeetAccessToken(
                commonService.getSystemConfig("MEET_MANAGE_SP_USER")
                , commonService.getSystemConfig("MEET_MANAGE_SP_PWD")
                , "28814466");

        /*** 调用华为云会议接口 ***/
        String url = HuaweiMeetConst.GATEWAY_URL + HuaweiMeetConst.DEL_CORP.replace("{corpId}",String.valueOf(huaweiComId));
        Map<String, String> headMap = new HashMap<>();
        headMap.put("X-Access-Token", accessToken);
        logger.info(" ========== 管理删除企业接口");
        logger.info(" ========== 请求地址："+url);
        RestRequest restRequest = HttpBuildUtil.buildRestRequest("DELETE", headMap, null, null);
        RestResponse response = HttpBuildUtil.sendMsg(url, restRequest);
        logger.info(" ========== 返回报文："+response.getEntity());
        //如果返回报文处理resultMsg有值，说明有失败信息
        String resultMsg = HttpBuildUtil.failMsg(response);
        if(StringUtils.isNotEmpty(resultMsg)){
            return Result.fail(resultMsg);
        }
        return Result.success(Result.SUCCESS_MSG);
    }

    /**
     * <b> 分配企业资源 </b>
     * @author ZXF
     * @create 2020/05/21 0021 11:09
     * @version
     * @注意事项 </b>
     */
    @Override
    public Result addResourcesByHuaweiComId(List<Map<String, String>> list,Long huaweiComId) throws V5BusinessException {
        /*** 获取华为云会议的accessToken ***/
        String accessToken = commonService.getComManageHuaweiMeetAccessToken(
                commonService.getSystemConfig("MEET_MANAGE_SP_USER")
                , commonService.getSystemConfig("MEET_MANAGE_SP_PWD")
                , "28814466");
        /*** 调用华为云会议接口 ***/
        String url = HuaweiMeetConst.GATEWAY_URL + HuaweiMeetConst.GET_RESOURCE_LIST.replace("{corpId}",String.valueOf(huaweiComId));
        String param = JSON.toJSONString(list);
        Map<String, String> headMap = new HashMap<>();
        headMap.put("X-Access-Token", accessToken);
        logger.info(" ========== 分配企业资源接口");
        logger.info(" ========== 请求地址："+url);
        logger.info(" ========== 请求报文："+param);
        RestRequest restRequest = HttpBuildUtil.buildRestRequest("POST", headMap, param, null);
        RestResponse response = HttpBuildUtil.sendMsg(url, restRequest);
        logger.info(" ========== 返回报文："+response.getEntity());
        //如果返回报文处理resultMsg有值，说明有失败信息
        String resultMsg = HttpBuildUtil.failMsg(response);
        if(StringUtils.isNotEmpty(resultMsg)){
            return Result.fail(resultMsg);
        }
        return Result.success(Result.SUCCESS_MSG);
    }

    /**
     * <b> 删除企业资源 </b>
     * @author ZXF
     * @create 2020/05/28 0028 10:01
     * @version
     * @注意事项 </b>
     */
    @Override
    public Result delResource(String[] ids, Long huaweiComId) throws V5BusinessException {
        /*** 获取华为云会议的accessToken ***/
        String accessToken = commonService.getComManageHuaweiMeetAccessToken(
                commonService.getSystemConfig("MEET_MANAGE_SP_USER")
                , commonService.getSystemConfig("MEET_MANAGE_SP_PWD")
                , "28814466");

        /*** 调用华为云会议接口 ***/
        String url = HuaweiMeetConst.GATEWAY_URL + HuaweiMeetConst.DELETE_RESOURCE_LIST.replace("{corpId}",String.valueOf(huaweiComId));
        Map<String, String> headMap = new HashMap<>();
        headMap.put("X-Access-Token", accessToken);
        String param = JSON.toJSONString(ids);
        logger.info(" ========== 管理删除企业接口");
        logger.info(" ========== 请求地址："+url);
        logger.info(" ========== 请求报文："+param);
        RestRequest restRequest = HttpBuildUtil.buildRestRequest("POST", headMap, param, null);
        RestResponse response = HttpBuildUtil.sendMsg(url, restRequest);
        logger.info(" ========== 返回报文："+response.getEntity());
        //如果返回报文处理resultMsg有值，说明有失败信息
        String resultMsg = HttpBuildUtil.failMsg(response);
        if(StringUtils.isNotEmpty(resultMsg)){
            return Result.fail(resultMsg);
        }
        return Result.success(Result.SUCCESS_MSG);
    }

    /**
     * <b> 公安所属省份下拉搜索 </b>
     * @author ZXF
     * @create 2020/05/22 0022 16:55
     * @version
     * @注意事项 </b>
     */
    @Override
    public List<Map> queryPoliceRegoinCombobox() {
        return sysPoliceMapper.queryPoliceRegoinCombobox();
    }

    /**
     * <b> 根据省编码查询下属公安机构编码列 </b>
     * @author ZXF
     * @create 2020/05/22 0022 17:11
     * @version
     * @注意事项 </b>
     */
    @Override
    public List<Map> queryPoliceByProvinceIdCombobox(String provinceId) {
        return sysPoliceMapper.queryPoliceByProvinceIdCombobox(provinceId);
    }
}