package com.bcxin.runtime.domain.metas.services.impls;

import com.bcxin.runtime.domain.constants.SyncMapConstant;
import com.bcxin.runtime.domain.metas.commands.BatchCreateFormSyncMetaCommand;
import com.bcxin.runtime.domain.metas.commands.CreateFormSyncMetaCommand;
import com.bcxin.runtime.domain.metas.entities.ApplicationMetaEntity;
import com.bcxin.runtime.domain.metas.entities.FormMetaEntity;
import com.bcxin.runtime.domain.metas.entities.FormSyncMetaEntity;
import com.bcxin.runtime.domain.metas.entities.FormSyncTargetMetaEntity;
import com.bcxin.runtime.domain.metas.entities.enums.FormType;
import com.bcxin.runtime.domain.metas.repositories.ApplicationMetaRepository;
import com.bcxin.runtime.domain.metas.repositories.FormSyncMetaRepository;
import com.bcxin.runtime.domain.metas.repositories.FormSyncTargetMetaRepository;
import com.bcxin.runtime.domain.metas.services.FormSyncMetaService;
import com.bcxin.runtime.domain.snapshoots.FormSyncConfigSnapshot;
import com.bcxin.saas.core.components.JsonProvider;
import com.bcxin.saas.core.exceptions.SaasNofoundException;

import java.util.Collection;
import java.util.Optional;
import java.util.stream.Collectors;

public class FormSyncMetaServiceImpl implements FormSyncMetaService {
    private final FormSyncMetaRepository formSyncMetaRepository;
    private final ApplicationMetaRepository applicationMetaRepository;
    private final JsonProvider jsonProvider;
    private final FormSyncTargetMetaRepository formSyncTargetMetaRepository;

    public FormSyncMetaServiceImpl(FormSyncMetaRepository formSyncMetaRepository,
                                   ApplicationMetaRepository applicationMetaRepository, JsonProvider jsonProvider,
                                   FormSyncTargetMetaRepository formSyncTargetMetaRepository) {
        this.formSyncMetaRepository = formSyncMetaRepository;
        this.applicationMetaRepository = applicationMetaRepository;
        this.jsonProvider = jsonProvider;
        this.formSyncTargetMetaRepository = formSyncTargetMetaRepository;
    }

    @Override
    public void saveAll(CreateFormSyncMetaCommand command) {
        Collection<String> formIds =
                command.getFormSyncs().stream()
                        .map(ii -> ii.getFormId()).collect(Collectors.toList());

        Collection<FormMetaEntity> formMetaEntities =
                this.applicationMetaRepository.getFormMetasByFormIds(formIds);

        Collection<FormSyncMetaEntity> formSyncMetaEntities =
                command.getFormSyncs().stream()
                        .map(fs -> {
                    Optional<FormMetaEntity> formMetaEntityOptional =
                            formMetaEntities.stream().filter(ix ->
                                    ix.getFormId().equals(fs.getFormId())).findFirst();
                    if (!formMetaEntityOptional.isPresent()) {
                        throw new SaasNofoundException(String.format("FormId(%s)不存在", fs.getFormId()));
                    }

                    FormSyncTargetMetaEntity targetMetaEntity = this.formSyncTargetMetaRepository.getById(fs.getTargetId());
                    if (targetMetaEntity == null) {
                        throw new SaasNofoundException(String.format("TargetId(%s)不存在", fs.getFormId()));
                    }

                    String config = this.jsonProvider.getJson(fs.getConfig());
                    FormSyncMetaEntity formSyncMetaEntity =
                            FormSyncMetaEntity.create(formMetaEntityOptional.get(),
                                    fs.isOnline(), config,
                                    targetMetaEntity, fs.getNote());

                    return formSyncMetaEntity;
                }).filter(ii -> ii != null).collect(Collectors.toList());

        this.formSyncMetaRepository.saveAll(formSyncMetaEntities);
    }

    @Override
    public void delete(String id) {
        FormSyncMetaEntity formSyncMetaEntity = this.formSyncMetaRepository.getById(id);
        if (formSyncMetaEntity == null) {
            throw new SaasNofoundException(String.format("找不同步表单配置信息(%s)", id));
        }

        this.formSyncMetaRepository.delete(formSyncMetaEntity);
    }

    @Override
    public void batch(BatchCreateFormSyncMetaCommand command) {
        ApplicationMetaEntity applicationMetaEntity = this.applicationMetaRepository.getById(command.getAppId());
        if (applicationMetaEntity == null) {
            throw new SaasNofoundException(String.format("App(%s)找不到", command.getAppId()));
        }

        FormSyncTargetMetaEntity targetMetaEntity = this.formSyncTargetMetaRepository.getById(command.getTargetId());
        if (targetMetaEntity == null) {
            throw new SaasNofoundException(String.format("Target目标(%s)找不到", command.getTargetId()));
        }

        Collection<FormMetaEntity> formMetaEntities = applicationMetaEntity.getFormMetas();

        Collection<FormSyncMetaEntity> formSyncMetaEntities =
                formMetaEntities.stream()
                        .filter(ii->ii.getFormType()== FormType.NORMAL)
                        .map(form -> {
                    FormSyncConfigSnapshot configSnapshot = FormSyncConfigSnapshot
                            .create(SyncMapConstant.getMetaMapKey(
                                    command.getTargetId(),
                                    form.getTableName()),null);

                    String config = this.jsonProvider.getJson(configSnapshot);
                    FormSyncMetaEntity formSyncMetaEntity =
                            FormSyncMetaEntity.create(form,
                                    true, config,
                                    targetMetaEntity, "批量添加表单信息");

                    return formSyncMetaEntity;
                }).collect(Collectors.toList());

        this.formSyncMetaRepository.saveAll(formSyncMetaEntities);
    }
}
