package com.bcxin.runtime.domain.metas.entities;

import com.bcxin.runtime.domain.metas.entities.enums.FormSyncType;
import com.bcxin.saas.core.IAggregate;
import com.bcxin.saas.core.models.EntityBase;
import lombok.AccessLevel;
import lombok.Getter;
import lombok.Setter;

import javax.persistence.*;
import java.util.Date;
import java.util.UUID;

import com.bcxin.runtime.domain.enums.BooleanStatus;

@Getter
@Setter(AccessLevel.PROTECTED)
@Entity
@Table(name = "sync_meta_forms")
public class FormSyncMetaEntity extends EntityBase<String> implements IAggregate {
    @Id
    @Column(length = 50)
    private String id;

    @Column(length = 255, nullable = true)
    private String note;

    //@Lob
    @Column(length = 2000)
    private String config;

    @Temporal(TemporalType.TIMESTAMP)
    @Column(name = "last_updated_time")
    private Date lastUpdatedTime;

    @ManyToOne(fetch = FetchType.LAZY, optional = false)
    @JoinColumn(name = "form_meta_id")
    private FormMetaEntity formMeta;

    @Temporal(TemporalType.TIMESTAMP)
    @Column(name = "last_synced_time")
    private Date lastSyncedTime;

    @Column(name = "synced_record_count")
    private int syncedRecordCount;

    @Column(name = "is_online")
    private BooleanStatus isOnline;

    @Column(length = 500)
    private String filter;

    @ManyToOne(fetch = FetchType.LAZY)
    @JoinColumn(name = "target_meta_id", nullable = false)
    private FormSyncTargetMetaEntity targetMetaEntity;

    @Temporal(TemporalType.TIMESTAMP)
    @Column(name = "created_time", nullable = false, columnDefinition = "DATETIME default current_timestamp()")
    private Date createdTime;

    protected FormSyncMetaEntity() {
        this.createdTime = new Date();
        this.lastUpdatedTime = this.createdTime;
        this.setIsOnline(BooleanStatus.FALSE);
    }

    protected FormSyncMetaEntity(FormMetaEntity formMetaEntity,
                                 boolean isOnline,
                                 String config,
                                 FormSyncTargetMetaEntity formSyncTargetMetaEntity, String note) {
        this();
        assert formMetaEntity != null;
        assert note != null;

        this.setId(UUID.randomUUID().toString());
        this.setFormMeta(formMetaEntity);
        this.setTargetMetaEntity(formSyncTargetMetaEntity);
        this.setNote(note);
        this.setIsOnline(isOnline ? BooleanStatus.TRUE : BooleanStatus.FALSE);
        this.setConfig(config);
    }

    public void change(FormSyncType syncType, String note, String config) {
        this.setNote(note);
        this.setConfig(config);
    }

    public void markSyncQueue(Date lastSyncedTime, int syncedRecordCount, String note) {
        this.setLastSyncedTime(lastSyncedTime);
        this.setSyncedRecordCount(this.getSyncedRecordCount() + syncedRecordCount);
        this.setNote(note);
        this.setIsOnline(BooleanStatus.TRUE);
    }

    public static FormSyncMetaEntity create(FormMetaEntity formMetaEntity,
                                            boolean isOnline,
                                            String config,
                                            FormSyncTargetMetaEntity formSyncTargetMetaEntity, String note) {
        FormSyncMetaEntity formSyncMetaEntity = new FormSyncMetaEntity(
                formMetaEntity, isOnline, config,
                formSyncTargetMetaEntity, note);

        return formSyncMetaEntity;
    }
}
