package cn.myapps.message.notification.controller;

import java.util.Map;

import cn.myapps.common.util.Security;
import com.bcxin.saas.core.components.DistributedCacheProvider;
import com.bcxin.saas.core.components.JsonProvider;
import com.bcxin.saas.core.utils.ExceptionUtils;
import org.codehaus.jackson.map.annotate.JsonSerialize;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.config.ConfigurableBeanFactory;
import org.springframework.context.annotation.Scope;
import org.springframework.stereotype.Controller;
import org.springframework.web.bind.annotation.*;

import cn.myapps.message.base.controller.BaseController;
import cn.myapps.message.notification.model.Notification;
import cn.myapps.message.notification.service.NotificationProcess;
import cn.myapps.message.notification.service.NotificationProcessBean;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import net.sf.json.JSONObject;

@Controller(value="NotificationAction")
@RequestMapping(value="/api/message/notification")
@Scope(value=ConfigurableBeanFactory.SCOPE_PROTOTYPE)
@JsonSerialize(include=JsonSerialize.Inclusion.NON_NULL)
@Api(tags = "消息通知模块")
public class NotificationController extends BaseController<Notification>{


	@Autowired
	private NotificationProcess notificationProcess;

	private final DistributedCacheProvider distributedCacheProvider;
	private final JsonProvider jsonProvider;

	public NotificationController(DistributedCacheProvider distributedCacheProvider, JsonProvider jsonProvider) {
		super();
		this.distributedCacheProvider = distributedCacheProvider;
		this.jsonProvider = jsonProvider;
		content = new Notification();
		process = new NotificationProcessBean();
	}
	
	/**
	 * 用户登录时，获取消息通知
	 * @return
	 */
	@GetMapping(value="/login")
	@ApiOperation(value = "用户登录时，获取消息通知", notes = "用户登录时，获取消息通知")
	public Map<String, Object> sendMessageNotificationWhenLogin() {
		try {
			String userId = Security.getUserIdFromToken(request);
			String sessionKey = String.format("sendMessageNotificationWhenLogin:%s", userId);
			JSONObject datas = (JSONObject) request.getSession().getAttribute(sessionKey);
			if (datas == null) {
				datas = notificationProcess.sendMessageNotificationWhenLogin(userId);
				request.getSession().setAttribute(sessionKey, datas);
			}

			return addActionResult(true, "", datas);
		} catch (Exception e) {
			e.printStackTrace();
			return addActionResult(false, "获取数据异常，请联系系统管理员", null);
		}
	}

	/**
	 * 获取消息中心通知
	 * @return
	 */
	@GetMapping
	@ApiOperation(value = "获取消息中心通知", notes = "获取消息中心通知")
	public Map<String, Object> sendMessageNotification2User() {
		try {
			String userId = Security.getUserIdFromToken(request);
			String sessionKey = String.format("sendMessageNotification2User:%s", userId);
			String result =
					this.distributedCacheProvider.get(sessionKey, () -> {
						String datas = null;
						try {
							JSONObject content = notificationProcess.sendMessageNotification2User(userId);
							datas = this.jsonProvider.getJson(content);
						} catch (Exception e) {
							throw new RuntimeException(e);
						}

						return datas;
					}, 2 * 60);

			JSONObject data = this.jsonProvider.getData(result, JSONObject.class);

			return addActionResult(true, "", data);
		} catch (Exception e) {
			return addActionResult(false, "获取数据异常，请联系系统管理员", null);
		}
	}
	
	/**
	 * 清空消息中心通知
	 * @return
	 */
	@DeleteMapping(value="/clear")
	@ApiOperation(value = "清空消息中心通知", notes = "清空消息中心通知")
	public Map<String, Object> clearNotification(){
		try {
			//IUser user = getUser();
			String userId = Security.getUserIdFromToken(request);
			notificationProcess.clearMessageNotification(null, null, null, userId);
			return addActionResult(true, "", datas);
		} catch (Exception e) {
			e.printStackTrace();
			return addActionResult(false, e.getMessage(), null);
		}
	}
	
	/**
	 * 获得消息总数
	 * @return
	 */
	@GetMapping(value="/count")
	@ApiOperation(value = "获得消息总数", notes = "获得消息总数")
	public Map<String, Object> getNotificationCount() {
		try {
			String userId = Security.getUserIdFromToken(request);
			String notifyCacheKey = String.format("getNotificationCount:%s", userId);
			String cacheData =
					distributedCacheProvider.get(notifyCacheKey, () -> {
						try {
							JSONObject result = notificationProcess.getNotificationCount(userId);

							return this.jsonProvider.getJson(result);
						} catch (Exception ex) {
							throw new RuntimeException(ex);
						}
					}, 10);


			return addActionResult(true, "", cacheData == null ? null : (this.jsonProvider.getData(cacheData, JSONObject.class)));
		} catch (Exception e) {
			return addActionResult(false, ExceptionUtils.getStackMessage(e), null);
		}
	}

	@PostMapping
	public void doCreate(@RequestBody Notification notification) {
		try {
			notificationProcess.doCreate(notification);
		} catch (Exception e) {
			e.printStackTrace();
		}
		System.out.println("Notice-->" + notification);
	}
}
