package cn.myapps.message.notice.controller;

import cn.hutool.core.collection.CollectionUtil;
import cn.hutool.core.lang.Snowflake;
import cn.myapps.common.auth.IUser;
import cn.myapps.common.data.DataPackage;
import cn.myapps.common.data.ParamsTable;
import cn.myapps.common.exception.OBPMValidateException;
import cn.myapps.common.util.StringUtil;
import cn.myapps.message.base.controller.BaseController;
import cn.myapps.message.notice.model.Notice;
import cn.myapps.message.notice.service.NoticeProcess;
import com.alibaba.fastjson.JSONObject;
import com.bcxin.saas.core.exceptions.SaasBadException;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiImplicitParam;
import io.swagger.annotations.ApiImplicitParams;
import io.swagger.annotations.ApiOperation;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.config.ConfigurableBeanFactory;
import org.springframework.context.annotation.Scope;
import org.springframework.stereotype.Controller;
import org.springframework.web.bind.annotation.*;

import java.util.*;
import java.util.stream.Collectors;

@Slf4j
@Controller(value="NoticeAction")
@RequestMapping(value="/api/message/notice")
@Scope(value=ConfigurableBeanFactory.SCOPE_PROTOTYPE)
@Api(tags = "通知模块")
public class NoticeController extends BaseController<Notice>{

	@Autowired
	private NoticeProcess noticeProcess;
	@Autowired
	private Snowflake snowflake;

	public NoticeController() {
		super();
		content = new Notice();
	}
	
	/**
	 * 获取事项提醒通知
	 * @return
	 */
	@GetMapping
	@ApiOperation(value = "获取事项提醒通知", notes = "获取事项提醒通知")
	@ApiImplicitParams({
			@ApiImplicitParam(name = "readStatus",value = "阅读状态",required = true,paramType = "query",dataType = "int"),
			@ApiImplicitParam(name = "_currpage", value = "页长度", required = false, paramType = "query", dataType = "string",defaultValue="1"),
			@ApiImplicitParam(name = "_rowcount", value = "当前页码", required = false, paramType = "query", dataType = "string",defaultValue="10"),
			@ApiImplicitParam(name = "type",value = "后续补充",required = true,paramType = "query",dataType = "string")
	})
	public Map<String, Object> doQuery() {
		try {
			ParamsTable params = getParams();
			String summary = params.getParameterAsString("summary");

			Boolean isRead = null;
			String readStatus = params.getParameterAsString("readStatus");
			if (!StringUtil.isBlank(readStatus)) {
				if ("0".equals(readStatus)) {
					isRead = false;
				}
			}

			String subjectType = params.getParameterAsString("type");
			Collection<Integer> selectedSubjectTypes = new ArrayList<>();
			if (!StringUtil.isBlank(subjectType)) {
				selectedSubjectTypes = Arrays.stream(subjectType.split("_")).map(ii -> {
					try {
						return Integer.parseInt(ii);
					} catch (Exception ex) {
						return null;
					}
				}).filter(ii -> ii != null).collect(Collectors.toList());
			}
			Integer page = params.getParameterAsInteger("_currpage");
			Integer lines = params.getParameterAsInteger("_rowcount");
			if (page == null || page <= 0)
				page = 1;
			if (lines == null || lines <= 0)
				lines = 30;
			IUser user = getUser();
			DataPackage<Notice> data = noticeProcess.queryNoticeByCondition(summary, isRead, selectedSubjectTypes, page, lines, user.getId());
			return addActionResult(true, "", data);
		} catch (Exception e) {
			if (!(e instanceof OBPMValidateException)) {
				e.printStackTrace();
			}
			return addActionResult(false, e.getMessage(), null);
		}
	}


	@GetMapping("/modules")
	@ApiOperation(value = "获取消息的模块内容", notes = "获取消息的模块内容")
	@ApiImplicitParams({
			@ApiImplicitParam(name = "readStatus",value = "阅读状态",required = true,paramType = "query",dataType = "int"),
			@ApiImplicitParam(name = "_currpage", value = "页长度", required = false, paramType = "query", dataType = "string",defaultValue="1"),
			@ApiImplicitParam(name = "_rowcount", value = "当前页码", required = false, paramType = "query", dataType = "string",defaultValue="10"),
			@ApiImplicitParam(name = "type",value = "后续补充",required = true,paramType = "query",dataType = "string")
	})
	public Map<String, Object> doModuleQuery() {
		try {
			ParamsTable params = getParams();
			String summary = params.getParameterAsString("summary");

			Boolean isRead = null;
			String readStatus = params.getParameterAsString("readStatus");
			if (!StringUtil.isBlank(readStatus)) {
				if ("0".equals(readStatus)) {
					isRead = false;
				} else {
					isRead = true;
				}
			}

			IUser user = getUser();
			List<Map<String, Object>> data = noticeProcess.queryModuleNoticeByCondition(summary, isRead, user.getId());
			return addActionResult(true, "", data);
		} catch (Exception e) {
			if (!(e instanceof OBPMValidateException)) {
				e.printStackTrace();
			}
			return addActionResult(false, e.getMessage(), null);
		}
	}

    /**
     * 删除事项提醒通知
     * @return
     */
	@DeleteMapping
	@ApiOperation(value = "删除事项提醒通知", notes = "删除事项提醒通知")
	@ApiImplicitParams({
			@ApiImplicitParam(name = "noticeId",value = "通知事项id",required = true,paramType = "query",dataType = "string")
	})
	public Map<String, Object> doDelete(@RequestParam String noticeId){
		 try {
			 noticeProcess.doRemove(noticeId);
			    return  addActionResult(true, "", null);
			} catch (Exception e) {
				if (!(e instanceof OBPMValidateException)) {
	                e.printStackTrace();
	            }
	        	return addActionResult(false, e.getMessage(), null);
			}
	}

	@PostMapping
	public void doCreate(@RequestBody Notice notice) {
		try {
			notice.setId(snowflake.nextIdStr());
			noticeProcess.doCreate(notice);
		} catch (Exception e) {
			e.printStackTrace();
			log.info("创建消息异常，参数：{}，错误：{}", JSONObject.toJSONString(notice), e.getMessage(), e);
			throw new SaasBadException("创建消息异常", e);
		}
	}

	/**
	 * description：批量创建消息
	 * author：linchunpeng
	 * date：2025/6/30
	 */
	@PostMapping("/create/list")
	public void doCreateList(@RequestBody List<Notice> voList) {
		try {
			if (CollectionUtil.isNotEmpty(voList)) {
				voList.forEach(notice -> notice.setId(snowflake.nextIdStr()));
				noticeProcess.doCreateList(voList);
			}
		} catch (Exception e) {
			e.printStackTrace();
			log.info("批量创建消息，参数：{}，错误：{}", JSONObject.toJSONString(voList), e.getMessage(), e);
			throw new SaasBadException("批量创建消息异常", e);
		}
	}
	
     /**
     * 设置事项提醒通知为已读
     * 
     * @deprecated 改为所用标记所有未读通知为已读 {@link #markAllAsRead()}
     * @return
     */
    @Deprecated
    @PutMapping(value="/read")
	@ApiOperation(value = "设置事项提醒通知为已读", notes = "设置事项提醒通知为已读")
	@ApiImplicitParams({
			@ApiImplicitParam(name = "noticeId",value = "通知事项id",required = true,paramType = "query",dataType = "string")
	})
	public Map<String, Object> set2Read(@RequestParam String noticeId){
		try {
			noticeProcess.setNotice2Read(noticeId);
		    return addActionResult(true, "", null);
		} catch (Exception e) {
			if (!(e instanceof OBPMValidateException)) {
                e.printStackTrace();
            }
        	return addActionResult(false, e.getMessage(), null);
		    }
		}
	
    /**
     * 标记所有未读通知为已读
     * 
     * @return
     */
    @PutMapping(value="/allRead")
	@ApiOperation(value = "标记所有未读通知为已读", notes = "标记所有未读通知为已读")
    public Map<String, Object> markAllAsRead() {
        try {
			noticeProcess.markAllAsRead(getUser().getId());
            return addActionResult(true, "", null);
        } catch (Exception e) {
        	if (!(e instanceof OBPMValidateException)) {
                e.printStackTrace();
            }
        	return addActionResult(false, e.getMessage(), null);
        }
    }
}
