package cn.myapps.caches;

import cn.myapps.caches.requests.CacheRequest;
import cn.myapps.caches.responses.CacheResponse;
import com.bcxin.saas.core.components.JsonProvider;
import com.bcxin.saas.core.exceptions.SaasBadException;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import lombok.Data;
import org.springframework.data.redis.core.StringRedisTemplate;
import org.springframework.http.ResponseEntity;
import org.springframework.util.StringUtils;
import org.springframework.web.bind.annotation.*;

import java.io.Serializable;
import java.time.Duration;
import java.time.temporal.ChronoUnit;

@RestController
@RequestMapping(value="/caches")
@Api(tags = "缓存信息")
public class CacheController {

    private final StringRedisTemplate redisTemplate;
    private final JsonProvider jsonProvider;

    public CacheController(StringRedisTemplate redisTemplate, JsonProvider jsonProvider) {
        this.redisTemplate = redisTemplate;
        this.jsonProvider = jsonProvider;
    }

    @GetMapping("/{key}")
    @ApiOperation(value = "获取缓存信息", notes = "获取缓存信息")
    public ResponseEntity<CacheResponse> get(@PathVariable String key) {
        if (!StringUtils.hasLength(key)) {
            throw new SaasBadException("缓存key不能为空.");
        }

        String content = this.redisTemplate.opsForValue().get(getCacheKey(key));

        CacheItem cacheItem = this.jsonProvider.getData(content, CacheItem.class);
        if (cacheItem == null) {
            return ResponseEntity.ok(null);
        }

        return ResponseEntity.ok(CacheResponse.create(cacheItem.getCode(), cacheItem.getData()));
    }


    @PostMapping
    @ApiOperation(value = "新增缓存信息", notes = "新增缓存信息")
    public ResponseEntity post(@RequestBody CacheRequest request) {
        if (!StringUtils.hasLength(request.getKey())) {
            throw new SaasBadException("缓存key不能为空.");
        }

        String cacheKey = getCacheKey(request.getKey());
        String content = this.jsonProvider.getJson(CacheItem.create(
                request.getCode(),
                request.getData()));
        if (request.getExpiredInSeconds() == null || request.getExpiredInSeconds() < 0) {
            this.redisTemplate.opsForValue().set(cacheKey, content);
        } else {
            this.redisTemplate.opsForValue().set(cacheKey, content,
                    Duration.of(request.getExpiredInSeconds(), ChronoUnit.SECONDS));
        }

        return ResponseEntity.ok().build();
    }

    @DeleteMapping("/{key}")
    @ApiOperation(value = "删除指定缓存", notes = "删除指定缓存")
    public ResponseEntity delete(@PathVariable String key) {
        this.redisTemplate.delete(getCacheKey(key));

        return ResponseEntity.ok().build();
    }

    private static String getCacheKey(String key) {
        return String.format("msg:cache:data:%s", key);
    }

    @Data
    public static class CacheItem implements Serializable {
        private String code;
        private Object data;

        public static CacheItem create(String code,Object data) {
            CacheItem item = new CacheItem();
            item.setData(data);
            item.setCode(code);

            return item;
        }
    }
}
