package cn.myapps.designtime.view.controller;

import cn.myapps.common.data.DataPackage;
import cn.myapps.common.exception.OBPMValidateException;
import cn.myapps.common.model.activity.Activity;
import cn.myapps.common.model.link.LinkVO;
import cn.myapps.common.model.view.*;
import cn.myapps.common.model.view.constant.ViewConstant;
import cn.myapps.common.util.StringUtil;
import cn.myapps.designtime.activity.controller.ActivityUtil;
import cn.myapps.designtime.activity.service.ActivityDesignTimeService;
import cn.myapps.designtime.common.controller.AbstractDesignTimeController;
import cn.myapps.designtime.common.controller.Resource;
import cn.myapps.designtime.common.service.DesignTimeServiceManager;
import cn.myapps.designtime.form.service.FormDesignTimeService;
import cn.myapps.designtime.link.service.LinkDesignTimeService;
import cn.myapps.designtime.view.column.service.ColumnDesignTimeService;
import cn.myapps.designtime.view.service.ViewDesignTimeService;
import cn.myapps.runtime.dynaform.form.helper.FormHelper;
import cn.myapps.util.sequence.Sequence;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiImplicitParam;
import io.swagger.annotations.ApiImplicitParams;
import io.swagger.annotations.ApiOperation;
import net.sf.json.JSONArray;
import net.sf.json.JSONObject;
import org.springframework.http.HttpStatus;
import org.springframework.http.MediaType;
import org.springframework.stereotype.Component;
import org.springframework.web.bind.annotation.*;

import java.util.*;

import static com.jayway.jsonpath.JsonPath.parse;

@Api(tags = "视图设计模块")
@Component
@RequestMapping(path = "/api/designtime/applications", produces = MediaType.APPLICATION_JSON_UTF8_VALUE)
public class ViewController extends AbstractDesignTimeController {

    private ViewDesignTimeService viewService;
    private FormDesignTimeService formService;
    private ColumnDesignTimeService columnsService;
    private ActivityDesignTimeService activityService;
    public ViewController() throws Exception {
        viewService = DesignTimeServiceManager.viewDesignTimeService();
        formService = DesignTimeServiceManager.formDesignTimeService();
        columnsService = DesignTimeServiceManager.columnDesignTimeService();
        activityService = DesignTimeServiceManager.activityDesignTimeService();
    }

    /**
     * 获取视图列表
     *
     * @param moduleId 模块id
     * @return
     * @throws Exception
     */
    @GetMapping(path = "/modules/{moduleId}/views")
    @ResponseStatus(HttpStatus.OK)
    @ApiOperation(value = "获取视图列表", notes = "获取视图列表（可根据名字或者描述查询）")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "moduleId", value = "模块id", required = true, paramType = "path", dataType = "string"),
            @ApiImplicitParam(name = "searchword", value = "查询关键字", required = false, paramType = "query", dataType = "string"),
            @ApiImplicitParam(name = "filterType", value = "过滤视图类型", required = false, paramType = "query", dataType = "string"),
            @ApiImplicitParam(name = "pageNo", value = "当前页数", required = true, paramType = "query", dataType = "int"),
            @ApiImplicitParam(name = "linesPerPage", value = "每页行数", required = true, paramType = "query", dataType = "int"),
    })
    public Resource doGetViewList(@PathVariable String moduleId, @RequestParam(required = false) String searchword,String filterType,String pageNo,String linesPerPage) throws Exception {
        int page = (pageNo != null && pageNo.length() > 0) ? Integer.parseInt(pageNo) : 1;
        int line = (linesPerPage != null && linesPerPage.length() > 0) ? Integer.parseInt(linesPerPage) : 10;
        String[] filterTypes = null;
        if(!StringUtil.isBlank(filterType)){
            filterTypes = filterType.split(",");
        }
        try {
            DataPackage<AbstractView> views = viewService.queryByNameOrDescript(moduleId, searchword, page, line);
            Collection<AbstractView> viewsList = views.getDatas();
            JSONArray viewArray = new JSONArray();
            outer:
            for (AbstractView view : viewsList) {
                if(filterTypes != null){
                    for(int i=0;i<filterTypes.length;i++){
                        //过滤对应视图类型
                        if(view.getViewTypeImpl().intValue() == Integer.valueOf(filterTypes[i])){
                            continue outer;
                        }
                    }
                }
                String viewId = view.getId();
                String viewName = view.getName();
                String viewDescription = view.getDescription();
                JSONObject viewJson = new JSONObject();
                viewJson.put("id", viewId);
                viewJson.put("name", viewName);
                viewJson.put("description", viewDescription);
                viewJson.put("type", view.getViewTypeImpl().intValue());
                viewArray.add(viewJson);
            }
            JSONObject viewsJson = new JSONObject();
            viewsJson.put("linesPerPage", views.getLinesPerPage());
            viewsJson.put("rowCount", views.getPageCount());
            viewsJson.put("pageNo", views.getPageNo());
            viewsJson.put("pageCount", views.getRowCount());
            viewsJson.put("datas", viewArray);
            return success("ok", viewsJson);
        } catch (Exception e) {
            e.printStackTrace();
            return error(500, e.getMessage(), null);
        }
    }

    /**
     * 获取视图详情
     *
     * @param viewId 视图id
     * @return
     * @throws Exception
     */
    @GetMapping(path = "/modules/views/{viewId}")
    @ResponseStatus(HttpStatus.OK)
    @ApiOperation(value = "获取视图详情", notes = "获取视图详情")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "viewId", value = "视图id", required = true, paramType = "path", dataType = "string")
    })
    public Resource doGetViewDetailed(@PathVariable String viewId) throws Exception {
        try {
            AbstractView view = viewService.findById(viewId);

            return success("ok", view);
        } catch (Exception e) {
            e.printStackTrace();
            return error(500, e.getMessage(), null);
        }
    }

    /**
     * 复制视图
     *
     * @param content  请求包体
     * @return
     * @throws Exception
     */
    @PostMapping(path = "/modules/views/copy")
    @ResponseStatus(HttpStatus.CREATED)
    @ApiOperation(value = "复制视图", notes = "复制视图")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "content", value = "请求包体", required = true, paramType = "body", dataType = "string")})
    public Resource doCopyViews(@RequestBody String content) throws Exception {
        try {
            List<String> ids = parse(content).json();
            String[] idArray = {};
            idArray = ids.toArray(idArray);
            viewService.doCopyView(idArray);

            return success("ok", null);
        } catch (Exception e) {
            e.printStackTrace();
            return error(500, e.getMessage(), null);
        }
    }

    /**
     * 新建视图
     *
     * @param applicationId 软件id
     * @param moduleId 模块id
     * @param viewType 视图类型
     * @param content  请求包体
     * @return
     * @throws Exception
     */
    @PostMapping(path = "/{applicationId}/modules/{moduleId}/views")
    @ResponseStatus(HttpStatus.CREATED)
    @ApiOperation(value = "新建视图", notes = "新建视图")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "applicationId", value = "软件id", required = true, paramType = "path", dataType = "string"),
            @ApiImplicitParam(name = "moduleId", value = "模块id", required = true, paramType = "path", dataType = "string"),
            @ApiImplicitParam(name = "viewType", value = "视图类型", required = true, paramType = "query", dataType = "int"),
            @ApiImplicitParam(name = "content", value = "请求包体", required = true, paramType = "body", dataType = "string")})
    public Resource doCreateView(@PathVariable String applicationId, @PathVariable String moduleId,  @RequestParam int viewType, @RequestBody String content) throws Exception {
        try {
            AbstractView view = null;
            JSONObject json = JSONObject.fromObject(content);
            switch(viewType){
                case ViewConstant.VIEW_TYPE_NORMAL:
                    view = (ListView)json2obj(json,ListView.class);
                    break;
                case ViewConstant.VIEW_TYPE_MAP:
                    view = (MapView)json2obj(json,MapView.class);
                    break;
                case ViewConstant.VIEW_TYPE_TREE:
                    view = (TreeView)json2obj(json,TreeView.class);
                    prepareTreeLinkJson(view);
                    break;
                case ViewConstant.VIEW_TYPE_CALENDAR:
                    view = (CalendarView)json2obj(json,CalendarView.class);
                    break;
                case ViewConstant.VIEW_TYPE_COLLAPSIBLE:
                    view = (CollapsibleView)json2obj(json,CollapsibleView.class);
                    break;
                case ViewConstant.VIEW_TYPE_GANTT:
                    view = (GanttView)json2obj(json,GanttView.class);
                    break;
            }

            boolean validate = validate(view);
            if(!validate){
                return error(500, "保存失败，列字段映射配置错误", null);
            }
            if(view != null) {
                view.setApplicationid(applicationId);
                view.setParentId(moduleId);
                String viewId = Sequence.getDesignTimeSequence();
                view.setId(viewId);
                //重名校验
                doSaveValidate(view);
                viewService.doCreate(view);

                JSONObject jsonObject = new JSONObject();
                jsonObject.put("id", viewId);

                return success("ok", jsonObject);
            }
            return error(500, "保存失败，没找到相关视图类型",null);
        } catch (Exception e) {
            e.printStackTrace();
            return error(500, e.getMessage(), null);
        }
    }

    /**
     * 更新视图
     *
     * @param applicationId  视图id
     * @param moduleId 模块id
     * @param viewId  视图id
     * @param viewType  视图类型
     * @param content 请求包体
     * @return
     * @throws Exception
     */
    @PutMapping(path = "/{applicationId}/modules/{moduleId}/views/{viewId}")
    @ResponseStatus(HttpStatus.OK)
    @ApiOperation(value = "更新视图", notes = "更新视图")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "applicationId", value = "软件id", required = true, paramType = "path", dataType = "string"),
            @ApiImplicitParam(name = "moduleId", value = "模块id", required = true, paramType = "path", dataType = "string"),
            @ApiImplicitParam(name = "viewId", value = "视图id", required = true, paramType = "path", dataType = "string"),
            @ApiImplicitParam(name = "viewType", value = "视图类型", required = true, paramType = "query", dataType = "int"),
            @ApiImplicitParam(name = "content", value = "请求包体", required = true, paramType = "body", dataType = "string")})
    public Resource doUpdateView(@PathVariable String applicationId, @PathVariable String moduleId, @PathVariable String viewId,
                                 @RequestParam int viewType, @RequestBody String content) throws Exception {
        try {
            AbstractView view = null;
            JSONObject json = JSONObject.fromObject(content);
            json.remove("uri");
            switch(viewType){
                case ViewConstant.VIEW_TYPE_NORMAL:
                    view = (ListView)json2obj(json,ListView.class);
                    break;
                case ViewConstant.VIEW_TYPE_MAP:
                    view = (MapView)json2obj(json,MapView.class);
                    break;
                case ViewConstant.VIEW_TYPE_TREE:
                    view = (TreeView)json2obj(json,TreeView.class);
                    prepareTreeLinkJson(view);
                    break;
                case ViewConstant.VIEW_TYPE_CALENDAR:
                    view = (CalendarView)json2obj(json,CalendarView.class);
                    break;
                case ViewConstant.VIEW_TYPE_COLLAPSIBLE:
                    view = (CollapsibleView)json2obj(json,CollapsibleView.class);
                    break;
                case ViewConstant.VIEW_TYPE_GANTT:
                    view = (GanttView)json2obj(json,GanttView.class);
                    break;
            }
            boolean validate = validate(view);
            if(!validate){
                return error(500, "保存失败，列字段映射配置错误", null);
            }
            if(view != null) {
                view.setApplicationid(applicationId);
                view.setParentId(moduleId);
                view.setId(viewId);
                view.setUri(view.getUri());
                //重名校验
                doSaveValidate(view);
                viewService.saveOrUpdate(view);

                return success("ok", null);
            }

            return error(500, "保存失败，没找到相关视图类型",null);
        } catch (Exception e) {
            e.printStackTrace();
            return error(500, e.getMessage(), null);
        }
    }

    /**
     * 删除视图（可批量）
     *
     * @param content 请求包体
     * @return
     * @throws Exception
     */
    @DeleteMapping(path = "/modules/views")
    @ResponseStatus(HttpStatus.OK)
    @ApiOperation(value = "删除视图", notes = "删除视图（可批量）")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "content", value = "请求包体", required = true, paramType = "body", dataType = "string")})
    public Resource doDeleteView(@RequestBody String content) throws Exception {
        try {
            List<String> ids = parse(content).json();
            String[] idArray = {};
            idArray = ids.toArray(idArray);

            viewService.delete(idArray);

            return success("ok", null);
        } catch (Exception e) {
            e.printStackTrace();
            return error(500, e.getMessage(), null);
        }
    }

    /**
     * 一键生成视图
     *
     * @param formId 表单id
     * @return
     * @throws Exception
     */
    @PostMapping(path = "/modules/views")
    @ResponseStatus(HttpStatus.CREATED)
    @ApiOperation(value = "一键生成视图", notes = "一键生成视图")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "formId", value = "表单id", required = true, paramType = "query", dataType = "String")})
    public Resource doOneKeyCreateView(@RequestParam String formId) throws Exception {
        try {
            formService.oneKeyCreateView(formId);
            return success("ok", null);
        } catch (Exception e) {
            e.printStackTrace();
            return error(500, e.getMessage(), null);
        }
    }

    /**
     * 获取视图列列表
     *
     * @param viewId 模块id
     * @return
     * @throws Exception
     */
    @GetMapping(path = "/modules/views/{viewId}/columns")
    @ResponseStatus(HttpStatus.OK)
    @ApiOperation(value = "获取视图列列表", notes = "获取视图列列表")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "viewId", value = "视图id", required = true, paramType = "path", dataType = "string")})
    public Resource doGetViewColumnsList(@PathVariable String viewId) throws Exception {

        try {
            AbstractView view = viewService.findById(viewId);
            view.setColumns(null);
            List<Column> columns = view.getColumns();

            JSONArray columnArray = new JSONArray();
            for (Column column : columns) {
                String columnId = column.getId();
                String columnName = column.getName();
                int columnOrderno = column.getOrderno();
                JSONObject columnJson = new JSONObject();
                columnJson.put("id", columnId);
                columnJson.put("name", columnName);
                columnJson.put("orderno", columnOrderno);
                columnArray.add(columnJson);
            }
            return success("ok", columnArray);
        } catch (Exception e) {
            e.printStackTrace();
            return error(500, e.getMessage(), null);
        }
    }

    /**
     * 获取视图列详情
     *
     * @param columnId 视图列id
     * @return
     * @throws Exception
     */
    @GetMapping(path = "/modules/views/columns/{columnId}")
    @ResponseStatus(HttpStatus.OK)
    @ApiOperation(value = "获取视图列详情", notes = "获取视图列详情")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "columnId", value = "视图列id", required = true, paramType = "path", dataType = "string")
    })
    public Resource doGetViewColumnsDetailed(@PathVariable String columnId) throws Exception {
        try {
            Column column = columnsService.findById(columnId);
            return success("ok", column);
        } catch (Exception e) {
            e.printStackTrace();
            return error(500, e.getMessage(), null);
        }
    }

    /**
     * 新建视图列
     *
     * @param applicationId 软件id
     * @param viewId 视图id
     * @param content  请求包体
     * @return
     * @throws Exception
     */
    @PostMapping(path = "/{applicationId}/modules/views/{viewId}/columns")
    @ResponseStatus(HttpStatus.CREATED)
    @ApiOperation(value = "新建视图列", notes = "新建视图列")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "applicationId", value = "软件id", required = true, paramType = "path", dataType = "string"),
            @ApiImplicitParam(name = "viewId", value = "视图id", required = true, paramType = "path", dataType = "string"),
            @ApiImplicitParam(name = "content", value = "请求包体", required = true, paramType = "body", dataType = "string")})
    public Resource doCreateViewColumn(@PathVariable String applicationId, @PathVariable String viewId, @RequestBody String content) throws Exception {
        try {
            JSONObject json = JSONObject.fromObject(content);
            Column column = (Column)json2obj(json,Column.class);

            if(column != null) {
                List<Column> columns = columnsService.queryAccurateByName(viewId, column.getName());
                if(columns.size() > 0){
                    return error(500, "该列名称已存在", null);
                }
                column.setApplicationid(applicationId);
                column.setParentId(viewId);
                String columnId = Sequence.getDesignTimeSequence();
                column.setId(columnId);

                columnsService.save(column);

                JSONObject columnJson = new JSONObject();
                columnJson.put("id", columnId);

                return success("ok", columnJson);
            }
            return success("ok", null);
        } catch (Exception e) {
            e.printStackTrace();
            return error(500, e.getMessage(), null);
        }
    }

    /**
     * 批量新建视图列
     *
     * @param applicationId 软件id
     * @param viewId 视图id
     * @param content  请求包体
     * @return
     * @throws Exception
     */
    @PostMapping(path = "/{applicationId}/modules/views/{viewId}/columns/batch")
    @ResponseStatus(HttpStatus.CREATED)
    @ApiOperation(value = "批量新建视图列", notes = "批量新建视图列")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "applicationId", value = "软件id", required = true, paramType = "path", dataType = "string"),
            @ApiImplicitParam(name = "viewId", value = "视图id", required = true, paramType = "path", dataType = "string"),
            @ApiImplicitParam(name = "content", value = "请求包体", required = true, paramType = "body", dataType = "string")})
    public Resource doCreateViewColumnBatch(@PathVariable String applicationId, @PathVariable String viewId, @RequestBody String content) throws Exception {
        try {
            JSONArray columnArray = cn.myapps.util.json.JsonTmpUtil.fromObject(content);
            JSONArray columnArray2 = new JSONArray();
            for (Object jsonArray:columnArray) {
                String columnContent = jsonArray.toString();
                JSONObject json = JSONObject.fromObject(columnContent);
                Column column = (Column) json2obj(json, Column.class);

                if (column != null) {
                    column.setApplicationid(applicationId);
                    column.setParentId(viewId);
                    String columnId = Sequence.getDesignTimeSequence();
                    column.setId(columnId);

                    columnsService.save(column);

                    JSONObject columnJson = new JSONObject();
                    columnJson.put("id", columnId);
                    columnArray2.add(columnJson);
                }
            }

            return success("ok", columnArray2);
        } catch (Exception e) {
            e.printStackTrace();
            return error(500, e.getMessage(), null);
        }
    }

    /**
     * 更新视图列
     *
     * @param applicationId  视图id
     * @param viewId 视图id
     * @param columnId  视图列id
     * @param content 请求包体
     * @return
     * @throws Exception
     */
    @PutMapping(path = "/{applicationId}/modules/views/{viewId}/columns/{columnId}")
    @ResponseStatus(HttpStatus.OK)
    @ApiOperation(value = "更新视图列", notes = "更新视图列")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "applicationId", value = "软件id", required = true, paramType = "path", dataType = "string"),
            @ApiImplicitParam(name = "viewId", value = "视图id", required = true, paramType = "path", dataType = "string"),
            @ApiImplicitParam(name = "columnId", value = "视图列id", required = true, paramType = "path", dataType = "string"),
            @ApiImplicitParam(name = "content", value = "请求包体", required = true, paramType = "body", dataType = "string")})
    public Resource doUpdateViewColumn(@PathVariable String applicationId, @PathVariable String viewId, @PathVariable String columnId, @RequestBody String content) throws Exception {
        try {
            JSONObject json = JSONObject.fromObject(content);
            Column column = (Column)json2obj(json,Column.class);

            if(column != null) {
                List<Column> columns = columnsService.queryAccurateByName(viewId, column.getName());
                for(Iterator<Column> iterator = columns.iterator();iterator.hasNext();){
                    Column c = iterator.next();
                    if(!c.getId().equals(columnId)){
                        return error(500, "该列名称已存在", null);
                    }
                }

                column.setApplicationid(applicationId);
                column.setParentId(viewId);
                column.setId(columnId);
                column.setCustomIcon((Boolean) json.get("customIcon"));

                columnsService.update(column);
            }
            return success("ok", null);
        } catch (Exception e) {
            e.printStackTrace();
            return error(500, e.getMessage(), null);
        }
    }

    /**
     * 删除视图列（可批量）
     *
     * @param content 请求包体
     * @return
     * @throws Exception
     */
    @DeleteMapping(path = "/modules/views/columns")
    @ResponseStatus(HttpStatus.OK)
    @ApiOperation(value = "删除视图列", notes = "删除视图列（可批量）")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "content", value = "请求包体", required = true, paramType = "body", dataType = "string")})
    public Resource doDeleteViewColumn(@RequestBody String content) throws Exception {
        try {
            List<String> ids = parse(content).json();
            String[] idArray = {};
            idArray = ids.toArray(idArray);

            columnsService.delete(idArray);

            return success("ok", null);
        } catch (Exception e) {
            e.printStackTrace();
            return error(500, e.getMessage(), null);
        }
    }

    /**
     * 获取视图操作列表
     *
     * @param viewId 视图id
     * @return
     * @throws Exception
     */
    @GetMapping(path = "/modules/views/{viewId}/activitys")
    @ResponseStatus(HttpStatus.OK)
    @ApiOperation(value = "获取视图操作列表", notes = "获取视图操作列表")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "viewId", value = "视图id", required = true, paramType = "path", dataType = "string")
    })
    public Resource doGetViewActivityList(@PathVariable String viewId){

        try {
            AbstractView view = viewService.findById(viewId);
            view.setActivities(null);
            List<Activity> list = view.getActivities();
            return success("ok", list);
        } catch (Exception e) {
            e.printStackTrace();
            return error(500, e.getMessage(), null);
        }
    }

    /**
     * 获取视图操作详情
     *
     * @param activityId 视图操作id
     * @return
     * @throws Exception
     */
    @GetMapping(path = "/modules/views/activitys/{activityId}")
    @ResponseStatus(HttpStatus.OK)
    @ApiOperation(value = "获取视图操作详情", notes = "获取视图操作详情")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "activityId", value = "视图操作id", required = true, paramType = "path", dataType = "string")
    })
    public Resource doGetViewActivity(@PathVariable String activityId) throws Exception {
        try {
            Activity activity = activityService.findById(activityId);
            return success("ok", activity);
        } catch (Exception e) {
            e.printStackTrace();
            return error(500, e.getMessage(), null);
        }
    }

    /**
     * 新建视图操作
     *
     * @param applicationId 软件id
     * @param viewId 视图id
     * @param content  请求包体
     * @return
     * @throws Exception
     */
    @PostMapping(path = "/{applicationId}/modules/views/{viewId}/activitys")
    @ResponseStatus(HttpStatus.CREATED)
    @ApiOperation(value = "新建视图操作", notes = "新建视图操作")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "applicationId", value = "软件id", required = true, paramType = "path", dataType = "string"),
            @ApiImplicitParam(name = "viewId", value = "视图id", required = true, paramType = "path", dataType = "string"),
            @ApiImplicitParam(name = "content", value = "请求包体", required = true, paramType = "body", dataType = "string")})
    public Resource doCreateViewActivity(@PathVariable String applicationId, @PathVariable String viewId, @RequestBody String content) throws Exception {
        try {
            JSONObject json = JSONObject.fromObject(content);
            Activity activity = (Activity)json2obj(json,Activity.class);

            List<Activity> activities = new ArrayList<>();
            ActivityUtil activityUtil = new ActivityUtil();
            AbstractView view = viewService.findById(viewId);
            //获取该视图下的按钮
            if(!StringUtil.isBlank(view.getActivityXML())){
                activities = (List<Activity>) activityUtil.parseXML(view.getActivityXML());
            }
            //校验名称是否已存在
            for(Activity act: activities) {
                if(act != null &&  (StringUtil.isBlank(activity.getId()) || !act.getId().equals(activity.getId()))  && activity.getName().equals(act.getName())) {
                    throw new Exception("该按钮名称已存在");
                }
            }
            if(activity != null) {
                activity.setApplicationid(applicationId);
                activity.setParentView(viewId);
                activity.setParentId(viewId);

                String activityId = Sequence.getDesignTimeSequence();
                if(StringUtil.isBlank(activity.getId())){
                    activity.setId(activityId);
                }
                activityService.saveOrUpdate(activity);

                JSONObject columnJson = new JSONObject();
                columnJson.put("id", activityId);

                return success("ok", columnJson);
            }
            return success("ok", null);
        } catch (Exception e) {
            e.printStackTrace();
            return error(500, e.getMessage(), null);
        }
    }

    /**
     * 更新视图操作
     *
     * @param applicationId  视图id
     * @param viewId 视图id
     * @param activityId  视图列id
     * @param content 请求包体
     * @return
     * @throws Exception
     */
    @PutMapping(path = "/{applicationId}/modules/views/{viewId}/activitys/{activityId}")
    @ResponseStatus(HttpStatus.OK)
    @ApiOperation(value = "更新视图操作", notes = "更新视图操作")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "applicationId", value = "软件id", required = true, paramType = "path", dataType = "string"),
            @ApiImplicitParam(name = "viewId", value = "视图id", required = true, paramType = "path", dataType = "string"),
            @ApiImplicitParam(name = "activityId", value = "视图操作id", required = true, paramType = "path", dataType = "string"),
            @ApiImplicitParam(name = "content", value = "请求包体", required = true, paramType = "body", dataType = "string")})
    public Resource doUpdateViewActivity(@PathVariable String applicationId, @PathVariable String viewId, @PathVariable String activityId, @RequestBody String content) throws Exception {
        try {
            JSONObject json = JSONObject.fromObject(content);
            Activity activity = (Activity)json2obj(json,Activity.class);

            if(activity != null) {
                activity.setApplicationid(applicationId);
                activity.setParentId(viewId);
                activity.setParentView(viewId);
                activity.setId(activityId);

                activityService.update(activity);
            }
            return success("ok", null);
        } catch (Exception e) {
            e.printStackTrace();
            return error(500, e.getMessage(), null);
        }
    }

    /**
     * 删除视图操作（可批量）
     *
     * @param ids 请求包体
     * @return
     * @throws Exception
     */
    @DeleteMapping(path = "/modules/views/activitys")
    @ResponseStatus(HttpStatus.OK)
    @ApiOperation(value = "删除视图操作", notes = "删除视图操作（可批量）")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "content", value = "请求包体", required = true, paramType = "body", dataType = "string")})
    public Resource doDeleteViewActivity(@RequestBody String[] ids) {
        try {
            activityService.delete(ids);
            return success("ok", null);
        } catch (Exception e) {
            e.printStackTrace();
            return error(500, e.getMessage(), null);
        }
    }

    /**
     * 获取数据来源表单字段（包含系统变量）
     *
     * @param formId 表单id
     * @return
     * @throws Exception
     */
    @GetMapping(path = "/getFormValueStoreFields/{formId}")
    @ResponseStatus(HttpStatus.OK)
    @ApiOperation(value = "获取数据来源表单字段（包含系统变量）", notes = "获取数据来源表单字段（包含系统变量）")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "formId", value = "表单id", required = true, paramType = "path", dataType = "string")
    })
    public Resource getFormValueStoreFields(@PathVariable String formId) {
        try{
            FormHelper formHelper = new FormHelper();
            Map<String, String> map = formHelper.getValueStoreFields(formId);
            return success("ok", map);
        }catch (Exception e){
            e.printStackTrace();
            return error(500, e.getMessage(), null);
        }
    }

    /**
     * 视图获取系统筛选字段
     *
     * @param formId 数据来源表单id
     * @return
     * @throws Exception
     */
    @GetMapping(path = "/getSystemScreeningFields/{formId}")
    @ResponseStatus(HttpStatus.OK)
    @ApiOperation(value = "视图获取系统筛选字段", notes = "视图获取系统筛选字段")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "formId", value = "数据来源表单id", required = true, paramType = "path", dataType = "string")
    })
    public Resource getSystemScreeningFields(@PathVariable String formId) {
        Map<String, String> map = new LinkedHashMap<String, String>();
        try{
            ViewHelper viewHelper = new ViewHelper();
            map = viewHelper.createFieldOptions(formId);
        }catch (Exception e){
            e.printStackTrace();
        }
        return success("ok", map);
    }

    /**
     * 根据筛选字段获取作用域
     *
     * @param authField 选中的筛选字段
     * @return
     * @throws Exception
     */
    @GetMapping(path = "/getScopeByField/{authField}")
    @ResponseStatus(HttpStatus.OK)
    @ApiOperation(value = "根据筛选字段获取作用域", notes = "根据筛选字段获取作用域")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "authField", value = "选中的筛选字段", required = true, paramType = "path", dataType = "string")
    })
    public Resource getScopeByField(@PathVariable String authField) {

        Map<String, String> map = new LinkedHashMap<String, String>();

        try{
            ViewHelper viewHelper = new ViewHelper();
            map = viewHelper.createAuthFieldOptions("",authField,"");
        }catch (Exception e){
            e.printStackTrace();
        }
        return success("ok", map);
    }

    /**
     * 视图列排序修改
     *
     * @return
     * @throws Exception
     */
    @PutMapping(path = "/views/column/order")
    @ApiOperation(value = "视图列排序修改", notes = "视图列排序修改")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "oId", value = "选中的列id", required = true, paramType = "query", dataType = "string"),
            @ApiImplicitParam(name = "nId", value = "互换的列id", required = true, paramType = "query", dataType = "string")
    })
    public Resource updateColumnOrder(@RequestParam String oId,@RequestParam String nId) {

        try {
            Column o = columnsService.findById(oId);
            Column n = columnsService.findById(nId);
            int tempOrder = o.getOrderno();
            o.setOrderno(n.getOrderno());
            n.setOrderno(tempOrder);
            columnsService.update(o);
            columnsService.update(n);
            return success("ok", "修改成功");
        } catch (Exception e) {
            e.printStackTrace();
            return error(500, e.getMessage(), null);
        }
    }

    /**
     * 操作列表排序修改
     *
     * @return
     * @throws Exception
     */
    @PutMapping(path = "/activity/order")
    @ApiOperation(value = "操作列表排序修改", notes = "操作列表排序修改")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "oId", value = "选中操作按钮id", required = true, paramType = "query", dataType = "string"),
            @ApiImplicitParam(name = "nId", value = "互换操作按钮id", required = true, paramType = "query", dataType = "string")
    })
    public Resource updateActivityOrder(@RequestParam String oId,@RequestParam String nId) {

        try {
            Activity o = activityService.findById(oId);
            Activity n = activityService.findById(nId);
            int tempOrder = o.getOrderno();
            o.setOrderno(n.getOrderno());
            n.setOrderno(tempOrder);
            activityService.update(o);
            activityService.update(n);
            return success("ok", "修改成功");
        } catch (Exception e) {
            e.printStackTrace();
            return error(500, e.getMessage(), null);
        }
    }

    /**
     * 保存前的校验
     *
     * @param view
     * @throws Exception
     */
    private void doSaveValidate(AbstractView view) throws Exception {
        //检验重名
        Collection<AbstractView> tempViews = viewService.queryByNameAndModule(view.getName(), view.getParentId());
        if (tempViews != null && !tempViews.isEmpty()) {
            AbstractView tempView = tempViews.iterator().next();
            if (view.getId() == null || view.getId().trim().length() <= 0) {// 判断新建不能重名
                throw new OBPMValidateException("{*[viewExist]*}");
            } else if (!view.getId().trim().equalsIgnoreCase(tempView.getId())){// 修改不能重名
                throw new OBPMValidateException("{*[viewExist]*}");
            }
        }
    }

    /**
     * 准备树形视图链接模式的linkvojson数据
     * @param view
     */
    private void prepareTreeLinkJson(AbstractView view) throws Exception{
        String nodeLinkId = ((TreeView)view).getNodeLinkId();
        if(!StringUtil.isBlank(nodeLinkId)){
            LinkDesignTimeService linkDesignTimeService = DesignTimeServiceManager.linkDesignTimeService();
            LinkVO linkVO = linkDesignTimeService.findById(nodeLinkId);
            if(linkVO != null){
                JSONObject link = new JSONObject();
                link.put("id", linkVO.getId());
                link.put("name", linkVO.getName());
                link.put("type", linkVO.getType());
                link.put("actionContent", linkVO.getActionContent());
                link.put("applicationid", linkVO.getApplicationid());
                link.put("queryString", "\""+linkVO.getQueryString()+"\"");
                ((TreeView)view).setLinkVoJson(link.toString());
            }
        }
    }

    /**
     * 校验
     * @param type
     * @param viewType
     * @return
     */
    private boolean validate(AbstractView view){
        List<Column> columns = view.getColumns();

        int count = 0;
        for(Column column : columns){
            String mappingField = column.getMappingField();
            if(!StringUtil.isBlank(mappingField)){
                count++;
            }
        }
        switch(view.getIntValue()){
            case ViewConstant.VIEW_TYPE_NORMAL:
                break;
            case ViewConstant.VIEW_TYPE_MAP:
            case ViewConstant.VIEW_TYPE_TREE:
                if(count < 3){
                    return false;
                }
                break;
            case ViewConstant.VIEW_TYPE_CALENDAR:
                if(count < 1){
                    return false;
                }
                break;
            case ViewConstant.VIEW_TYPE_COLLAPSIBLE:
                break;
            case ViewConstant.VIEW_TYPE_GANTT:
                if(count < 4){
                    return false;
                }
                break;
        }
        return true;
    }

}
