package cn.myapps.designtime.style.controller;

import cn.myapps.common.exception.OBPMValidateException;
import org.apache.commons.beanutils.BeanUtils;
import org.springframework.http.HttpStatus;
import org.springframework.http.MediaType;
import org.springframework.stereotype.Component;
import org.springframework.web.bind.annotation.DeleteMapping;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.PutMapping;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.ResponseStatus;

import cn.myapps.common.data.DataPackage;
import cn.myapps.common.model.stylerepository.StyleRepositoryVO;
import cn.myapps.designtime.common.controller.AbstractDesignTimeController;
import cn.myapps.designtime.common.controller.Resource;
import cn.myapps.designtime.common.service.DesignTimeServiceManager;
import cn.myapps.designtime.style.repository.service.StyleRepositoryDesignTimeService;
import cn.myapps.util.StringUtil;
import cn.myapps.util.sequence.Sequence;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiImplicitParam;
import io.swagger.annotations.ApiImplicitParams;
import io.swagger.annotations.ApiOperation;
import net.sf.json.JSONObject;

@Api(tags = "样式库模块")
@Component
@RequestMapping(path = "/api/designtime/applications", produces = MediaType.APPLICATION_JSON_UTF8_VALUE)
public class StylesController extends AbstractDesignTimeController {

    private StyleRepositoryDesignTimeService styleRepositoryDesignTimeService;

    public StylesController() throws Exception {
        styleRepositoryDesignTimeService = DesignTimeServiceManager.styleRepositoryDesignTimeService();
    }

    /**
     * 获取样式库列表（可根据名字查询）
     *
     * @param applicationId 软件id
     * @return
     * @throws Exception
     */
    @GetMapping(path = "/{applicationId}/styles")
    @ResponseStatus(HttpStatus.OK)
    @ApiOperation(value = "获取样式库列表", notes = "获取样式库列表")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "applicationId", value = "软件id", required = true, paramType = "path", dataType = "string"),
            @ApiImplicitParam(name = "pageNo", value = "页码", required = true, paramType = "path", dataType = "string"),
            @ApiImplicitParam(name = "linesPerPage", value = "页条数", required = true, paramType = "path", dataType = "string"),
            @ApiImplicitParam(name = "name", value = "样式库名称", required = true, paramType = "path", dataType = "string")})
    public Resource getStylesList(@PathVariable String applicationId, String name,String pageNo,String linesPerPage){

        int page = (pageNo != null && pageNo.length() > 0) ? Integer.parseInt(pageNo) : 1;
        int line = (linesPerPage != null && linesPerPage.length() > 0) ? Integer.parseInt(linesPerPage) : 10;
        try {
            DataPackage<StyleRepositoryVO> data = styleRepositoryDesignTimeService.query(applicationId, name, page, line);
            return success("ok", data);
        } catch (Exception e) {
            e.printStackTrace();
            return error(500, e.getMessage(), null);
        }
    }

    /**
     * 获取样式库详情
     *
     * @param styleId 样式库id
     * @return
     * @throws Exception
     */
    @GetMapping(path = "/{applicationId}/styles/{styleId}")
    @ResponseStatus(HttpStatus.OK)
    @ApiOperation(value = "获取样式库详情", notes = "获取样式库详情")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "styleId", value = "样式库id", required = true, paramType = "path", dataType = "string")
    })
    public Resource doGetStyle(@PathVariable String applicationId,@PathVariable String styleId) throws Exception {
        try {
            StyleRepositoryVO style = styleRepositoryDesignTimeService.findById(styleId);
            return success("ok", style);
        } catch (Exception e) {
            e.printStackTrace();
            return error(500, e.getMessage(), null);
        }
    }

    /**
     * 新建样式库
     *
     * @param applicationId 软件id
     * @param content  请求包体
     * @return
     * @throws Exception
     */
    @PostMapping(path = "/{applicationId}/styles")
    @ResponseStatus(HttpStatus.CREATED)
    @ApiOperation(value = "新建样式库", notes = "新建样式库")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "applicationId", value = "软件id", required = true, paramType = "path", dataType = "string"),
            @ApiImplicitParam(name = "content", value = "请求包体", required = true, paramType = "body", dataType = "string")})
    public Resource doCreateStyles(@PathVariable String applicationId, @RequestBody String content) throws Exception {
        try {

            JSONObject json = JSONObject.fromObject(content);
            StyleRepositoryVO rv = (StyleRepositoryVO)json2obj(json,StyleRepositoryVO.class);
            rv.setApplicationid(applicationId);
            rv.setParentId(applicationId);

            if(StringUtil.isBlank(rv.getId())){
                rv.setId(Sequence.getDesignTimeSequence());
            }
            doSaveValidate(rv);
            styleRepositoryDesignTimeService.save(rv);

            JSONObject formJson = new JSONObject();
            formJson.put("id", rv.getId());

            return success("ok", formJson);
        } catch (OBPMValidateException ex){
            ex.printStackTrace();
            return error(40001, ex.getValidateMessage(), null);
        } catch (Exception e) {
            e.printStackTrace();
            return error(500, e.getMessage(), null);
        }
    }

    /**
     * 更新样式库
     *
     * @param applicationId 软件id
     * @param styleId 样式库id
     * @param content 请求包体
     * @return
     * @throws Exception
     */
    @PutMapping(path = "/{applicationId}/styles/{styleId}")
    @ResponseStatus(HttpStatus.OK)
    @ApiOperation(value = "更新样式库", notes = "更新样式库")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "applicationId", value = "软件id", required = true, paramType = "path", dataType = "string"),
            @ApiImplicitParam(name = "styleId", value = "样式库id", required = true, paramType = "path", dataType = "string"),
            @ApiImplicitParam(name = "content", value = "请求包体", required = true, paramType = "body", dataType = "string")
    })
    public Resource doUpdateStyles(@PathVariable String applicationId, @PathVariable String styleId, @RequestBody String content) throws Exception {
        try {
            JSONObject json = JSONObject.fromObject(content);
            StyleRepositoryVO rv = (StyleRepositoryVO)json2obj(json,StyleRepositoryVO.class);

            StyleRepositoryVO orgStyle = styleRepositoryDesignTimeService.findById(styleId);
            StyleRepositoryVO style = (StyleRepositoryVO) BeanUtils.cloneBean(orgStyle);
            style.setApplicationid(applicationId);
            style.setContent(rv.getContent());
            style.setName(rv.getName());
            style.setDescription(rv.getDescription());
            doSaveValidate(rv);
            styleRepositoryDesignTimeService.saveOrUpdate(style);

            return success("ok", null);
        } catch (OBPMValidateException ex){
            ex.printStackTrace();
            return error(40001, ex.getValidateMessage(), null);
        } catch (Exception e) {
            e.printStackTrace();
            return error(500, e.getMessage(), null);
        }
    }

    /**
     * 删除样式库（可批量）
     *
     * @param ids 请求包体
     * @return
     * @throws Exception
     */
    @DeleteMapping(path = "/{applicationId}/styles")
    @ResponseStatus(HttpStatus.OK)
    @ApiOperation(value = "删除样式库", notes = "删除样式库（可批量）")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "content", value = "请求包体", required = true, paramType = "body", dataType = "string")})
    public Resource doDeleteStyles(@RequestBody String[] ids) throws Exception {
        try {
            styleRepositoryDesignTimeService.delete(ids);
            return success("ok", "删除成功");
        } catch (Exception e) {
            e.printStackTrace();
            return error(500, e.getMessage(), null);
        }
    }

    /**
     * 保存前的校验
     *
     * @param repositoryVO
     * @throws Exception
     */
    private void doSaveValidate(StyleRepositoryVO repositoryVO) throws Exception {
        StyleRepositoryVO po =  styleRepositoryDesignTimeService.getRepositoryByName(repositoryVO.getName(), repositoryVO.getApplicationid());
        if (po != null && StringUtil.isBlank(repositoryVO.getId())) {
            //新建
            throw new OBPMValidateException(" 库名称重复 ,请重新命名再保存！ ");
        }
        else {
            //更新
            if (po != null && !StringUtil.isBlank(repositoryVO.getId()) && !repositoryVO.getId().equals(po.getId())) {
                throw new OBPMValidateException(" 库名称重复 ,请重新命名再保存！ ");
            }
        }
    }
}