package cn.myapps.designtime.report.controller;

import java.io.File;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Iterator;
import java.util.List;

import javax.servlet.http.Cookie;

import org.apache.commons.lang.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.http.HttpStatus;
import org.springframework.http.MediaType;
import org.springframework.stereotype.Component;
import org.springframework.web.bind.annotation.DeleteMapping;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.PutMapping;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.ResponseStatus;
import org.springframework.web.multipart.MultipartFile;

import cn.myapps.base.web.WebUser;
import cn.myapps.common.Environment;
import cn.myapps.common.data.DataPackage;
import cn.myapps.common.data.ParamsTable;
import cn.myapps.common.exception.OBPMValidateException;
import cn.myapps.common.model.report.QueryColumnInfo;
import cn.myapps.common.model.report.Report;
import cn.myapps.common.util.Security;
import cn.myapps.conf.FeignConfig;
import cn.myapps.designtime.common.controller.AbstractDesignTimeController;
import cn.myapps.designtime.common.controller.Resource;
import cn.myapps.designtime.common.util.fs.VirtualFileSystemUtils;
import cn.myapps.designtime.report.feignservice.RuntimeFeignService;
import cn.myapps.designtime.report.service.ReportDesignTimeService;
import cn.myapps.util.StringUtil;
import cn.myapps.util.file.FileOperate;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiImplicitParam;
import io.swagger.annotations.ApiImplicitParams;
import io.swagger.annotations.ApiOperation;
import net.sf.json.JSONArray;
import net.sf.json.JSONObject;

@Api(tags = "报表模块")
@Component
@RequestMapping(path = "/api/designtime/applications", produces = MediaType.APPLICATION_JSON_UTF8_VALUE)
public class ReportsController extends AbstractDesignTimeController {

    @Autowired
    private ReportDesignTimeService reportDesignTimeService;

    @Autowired
    private RuntimeFeignService runtimeFeignService;

    /**
     * 获取报表列表（可根据名字或者备注查询）
     *
     * @param moduleId 模块id
     * @return
     * @throws Exception
     */
    @GetMapping(path = "/modules/{moduleId}/reports")
    @ResponseStatus(HttpStatus.OK)
    @ApiOperation(value = "获取报表列表", notes = "获取报表列表")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "moduleId", value = "模块id", required = true, paramType = "path", dataType = "string"),
            @ApiImplicitParam(name = "pageNo", value = "页码", required = true, paramType = "query", dataType = "string"),
            @ApiImplicitParam(name = "linesPerPage", value = "页条数", required = true, paramType = "query", dataType = "string"),
            @ApiImplicitParam(name = "searchword", value = "名字", required = true, paramType = "query", dataType = "string"),
            @ApiImplicitParam(name = "isPrint", value = "获取打印模板列表", required = true, paramType = "query", dataType = "string")
    })
    public Resource getReportsList(@PathVariable String moduleId, String isPrint,String searchword,String pageNo,String linesPerPage){
        int page = (pageNo != null && pageNo.length() > 0) ? Integer.parseInt(pageNo) : 1;
        int line = (linesPerPage != null && linesPerPage.length() > 0) ? Integer.parseInt(linesPerPage) : 10;
        try {
            DataPackage<Report> data = reportDesignTimeService.queryByNameOrDescript(moduleId, searchword, page, line);
//            reportDesignTimeService.query
            if(!StringUtil.isBlank(isPrint) && isPrint.equals("1")){
                Collection<Report> datas = data.getDatas();
                for (Iterator<Report> iter = datas.iterator(); iter.hasNext();) {
                    Report report = iter.next();
                    //判断不是打印模板则移除
                    if(report.getIsPrint() != Integer.valueOf(isPrint)){
                        iter.remove();
                    }
                }
                data.setDatas(datas);
            }
            return success("ok", data);
        } catch (Exception e) {
            e.printStackTrace();
            return error(500, e.getMessage(), null);
        }
    }

    /**
     * 获取报表详情
     *
     * @param reportId 报表id
     * @return
     * @throws Exception
     */
    @GetMapping(path = "/reports/{reportId}")
    @ResponseStatus(HttpStatus.OK)
    @ApiOperation(value = "获取报表详情", notes = "获取报表详情")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "reportId", value = "报表id", required = true, paramType = "path", dataType = "string")
    })
    public Resource doGetReport(@PathVariable String reportId) throws Exception {
        try {
            Report report = reportDesignTimeService.findById(reportId);
            return success("ok", report);
        } catch (Exception e) {
            e.printStackTrace();
            return error(500, e.getMessage(), null);
        }
    }

    /**
     * 新建报表
     *
     * @param applicationId 软件id
     * @param content  请求包体
     * @return
     * @throws Exception
     */
    @PostMapping(path = "/{applicationId}/modules/{moduleId}/reports")
    @ApiOperation(value = "新建报表", notes = "新建报表")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "applicationId", value = "软件id", required = true, paramType = "path", dataType = "string"),
            @ApiImplicitParam(name = "moduleId", value = "模块id", required = true, paramType = "path", dataType = "string"),
            @ApiImplicitParam(name = "content", value = "请求包体", required = true, paramType = "body", dataType = "string")})
    public Resource doCreateReports(@PathVariable String applicationId, @PathVariable String moduleId, @RequestBody String content) throws Exception {

        JSONObject json = JSONObject.fromObject(content);
        Report report = (Report)json2obj(json,Report.class);
        try {
            report.setApplicationid(applicationId);
            report.setParentId(moduleId);
            //校验是否重复名
            validate(report);
            String reportUri = report.getUri();
            reportUri = reportUri.substring(0, reportUri.lastIndexOf("/"));
            String jrxmlFilePath = "/workspace" + reportUri + "/" + report.getName() + ".jrxml";
            // 保存jrxml路径到report里
            report.setJrxmlFilePath(jrxmlFilePath);
            reportDesignTimeService.save(report);

            JSONObject jsonObject = new JSONObject();
            jsonObject.put("id", report.getId());
            jsonObject.put("jrxmlFilePath", report.getJrxmlFilePath());
            return success("ok", jsonObject);
        } catch (Exception e) {
            e.printStackTrace();
            return error(500, e.getMessage(), null);
        }
    }


    /**
     * 更新报表
     *
     * @param content 请求包体
     * @return
     * @throws Exception
     */
    @PutMapping(path = "/reports")
    @ApiOperation(value = "更新报表", notes = "更新报表")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "content", value = "请求包体", required = true, paramType = "body", dataType = "string")
    })
    public Resource doUpdateReports(@RequestBody String content) throws Exception {
        String jrxmlFileRealPath="";
        String jasperFileRealPath = "";
        JSONObject json = JSONObject.fromObject(content);
        Report report = (Report)json2obj(json,Report.class);
        try {
            if(!StringUtil.isBlank((String)json.get("appId"))){
                report.setApplicationid((String)json.get("appId"));
            }
            Environment env = Environment.getInstance();
            String reportUri = report.getUri();
            reportUri = reportUri.substring(0, reportUri.lastIndexOf("/"));
            String jrxmlFilePath = "/workspace" + reportUri + "/" + report.getName() + ".jrxml";
            jrxmlFileRealPath = env.getRealPath(jrxmlFilePath);
            //删除修改前的jrxml文件
            if(!StringUtils.isBlank(report.getId())) {
                File file = new File(jrxmlFileRealPath);
                if(file.exists()){
                    VirtualFileSystemUtils.delete(env.getRealPath(jrxmlFileRealPath));
                }
            }
            // 保存jrxml路径到report里
            report.setJrxmlFilePath(jrxmlFilePath);



            //如果报表画布为空
            if(StringUtil.isBlank(report.getJrxmlString())){
                report.setIsDefaultStyle("1");
                reportDesignTimeService.saveOrUpdate(report);
            }else {
                // 将上传的jrxml字符串以文件形式保存在上传目录里
                FileOperate.writeFileUTF(jrxmlFileRealPath, report.getJrxmlString(), true );
                //将jrxml编译为jasper文件
                //JasperCompileManager.compileReportToFile(jrxmlFileRealPath,jasperFileRealPath);
            }
            reportDesignTimeService.saveOrUpdate(report);
            return success("ok", null);
        } catch (Exception e) {
            e.printStackTrace();
            return error(500, e.getMessage(), null);
        }
    }

    /**
     * 删除报表（可批量）
     *
     * @param ids 请求包体
     * @return
     * @throws Exception
     */
    @DeleteMapping(path = "/reports")
    @ApiOperation(value = "删除报表", notes = "删除报表（可批量）")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "content", value = "请求包体", required = true, paramType = "body", dataType = "string")})
    public Resource doDeleteReports(@RequestBody String[] ids) throws Exception {
        try {
            reportDesignTimeService.delete(ids);
            return success("ok", "删除成功");
        } catch (Exception e) {
            e.printStackTrace();
            return error(500, e.getMessage(), null);
        }
    }

    /**
     * 通过数据源获取返回结果集的列信息
     *
     * @param dataSourceName 数据源名称
     * @return
     * @throws Exception
     */
    @PostMapping(path = "/{applicationId}/reports/getQueryColumnInfos")
    @ApiOperation(value = "通过数据源获取返回结果集的列信息", notes = "通过数据源获取返回结果集的列信息")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "applicationId", value = "软件id", required = true, paramType = "path", dataType = "string"),
            @ApiImplicitParam(name = "dataSourceName", value = "数据源名称", required = false, paramType = "query", dataType = "string"),
            @ApiImplicitParam(name = "viewId", value = "视图id", required = false, paramType = "query", dataType = "string"),
            @ApiImplicitParam(name = "isForm", value = "是否为表单", required = true, paramType = "query", dataType = "string"),
            @ApiImplicitParam(name = "reportId", value = "报表id", required = false, paramType = "query", dataType = "string"),
            @ApiImplicitParam(name = "type", value = "数据源类型", required = true, paramType = "query", dataType = "string"),
            @ApiImplicitParam(name = "content", value = "请求包体", required = true, paramType = "body", dataType = "string")
    })
    public Resource getQueryColumnInfos(@PathVariable String applicationId,@RequestBody String content, String dataSourceName,String viewId, String isForm, String reportId,@RequestParam String type) {
        try {
            JSONObject jsonObject = JSONObject.fromObject(content);
            String scriptString = (String) jsonObject.get("scriptString");
            Report report = new Report();
            if(!StringUtil.isBlank(reportId)){
                report = reportDesignTimeService.findById(reportId);
            }
            WebUser user = getUser();
            ParamsTable params = getParams(); // 获取并设置参数
            List<QueryColumnInfo> queryColumnInfos = new ArrayList<>();
            // 获取查询结果列信息
            if(type.equals("DATASOURCE_TYPE_VIEW")){
                // 判断是否作为打印模板
                boolean flag = Boolean.parseBoolean(isForm);
                if (flag) {
                    queryColumnInfos = reportDesignTimeService.getFormColumnsInfos(viewId);
                } else {
                    queryColumnInfos = reportDesignTimeService.getViewColumnsInfos(viewId);
                }
            }else if(type.equals("DATASOURCE_TYPE_SQL")){
                queryColumnInfos = reportDesignTimeService.getQueryColumnInfos(scriptString, dataSourceName, applicationId, user, report, params );
            }else if(type.equals("DATASOURCE_TYPE_PROCEDURE")){
                queryColumnInfos = reportDesignTimeService.getProcedureColumnsInfos(scriptString, dataSourceName, applicationId, user, report, params );
            }else if(type.equals("DATASOURCE_TYPE_CUSTOM")){
                queryColumnInfos = runtimeFeignService.getCustomColumnsInfos(scriptString,applicationId);
            }
            //报表联查字段重复解决
            List<String> columnNmaes = new ArrayList<String>();
            for (QueryColumnInfo queryColumnInfo: queryColumnInfos) {
                String columnName = queryColumnInfo.getColumnName();
                for (int i = 0; i < columnNmaes.size(); i++) {
                    if (columnNmaes.contains(columnName)) {
                        columnName += '1';
                    }
                }
                queryColumnInfo.setColumnName(columnName);
                columnNmaes.add(columnName);
            }
            return success("ok", queryColumnInfos);
        } catch (Exception e) {
            e.printStackTrace();
            return error(500, e.getMessage(), null);
        }
    }

    /**
     * 获取jrxml字符串
     * @param id 报表id
     * @return
     */
    @GetMapping(path = "/reports/getJrxmlString")
    @ApiOperation(value = "获取jrxml字符串", notes = "获取jrxml字符串")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "id", value = "报表id", required = true, paramType = "query", dataType = "string")})
    public Resource getJrxmlString(String id){
        try{
            Environment env = Environment.getInstance();
            Report report = reportDesignTimeService.doView(id);
            String jrxmlPath = report.getJrxmlFilePath();
            String jrxmlRealPath = env.getRealPath(jrxmlPath);
            String jrxmlString = FileOperate.getFileContentAsStringUTF(jrxmlRealPath);
            return success("ok",jrxmlString);
        }catch (Exception e){
            e.printStackTrace();
            return error(500, e.getMessage(), null);
        }

    }

    /**
     * 获取ReportExamples的iscript脚本
     * @param exampleName 示例名称
     * @return
     */
    @GetMapping(path = "/reports/getExampleIscriptString")
    @ApiOperation(value = "获取ReportExamples的iscript脚本", notes = "获取ReportExamples的iscript脚本")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "exampleName", value = "示例名称", required = true, paramType = "query", dataType = "string")})
    public Resource getExampleIscriptString(@RequestParam String exampleName){
        try{
            String result = reportDesignTimeService.getExampleIscriptStringByName(exampleName);
            return success("ok",result);
        }catch (Exception e){
            e.printStackTrace();
            return error(500, e.getMessage(), null);
        }

    }

    /**
     * 接收上传图片接口
     * @param file 文件二进制流
     * @return
     */
    @PostMapping(path = "/reports/uploadImageFile")
    @ApiOperation(value = "接收上传图片接口", notes = "接收上传图片接口")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "file", value = "文件二进制流", required = true, paramType = "body", dataType = "string")
    })
    public Resource uploadImageFile(MultipartFile file){
        try{
            Environment env = Environment.getInstance();

            //获取上传的文件名
            String path = "/uploads/reports/images/" + file.getOriginalFilename();

            String realPath = env.getRealPath(path);
            //写入文件夹
            FileOperate.writeFile(realPath, file.getInputStream());
            return success("ok",realPath);
        }catch (Exception e){
            e.printStackTrace();
            return error(500, e.getMessage(), null);
        }

    }

    /**
     * 返回报表的jasper文件路径
     * @param reportId  报表id
     * @return
     */
    @GetMapping(path = "/reports/getReportJasperPath")
    @ApiOperation(value = "返回报表的jasper文件路径", notes = "返回报表的jasper文件路径")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "reportId", value = "报表id", required = true, paramType = "query", dataType = "string")
    })
    public Resource getReportJasperPath(@RequestParam String reportId){
        try{
            Environment env = Environment.getInstance();
            Report report = reportDesignTimeService.doView(reportId);
            //获取真实jasper文件存放的路径
            String realPath = env.getRealPath(report.getJrxmlFilePath()).replace(".jrxml",".jasper");

            return success("ok",realPath);
        }catch (Exception e){
            e.printStackTrace();
            return error(500, e.getMessage(), null);
        }

    }

    /**
     * 报表预览返回html路径
     * @param id 报表id
     * @return
     */
    @GetMapping(path = "/reports/showjrxml/{id}")
    @ApiOperation(value = "报表预览返回html路径", notes = "报表预览返回html路径")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "id", value = "报表id", required = true, paramType = "path", dataType = "string")
    })
    public Resource showJrxmlReport(@PathVariable String id,@RequestParam(required = false) String domainId) {
        try {
            if (!StringUtil.isBlank(id)) {
                WebUser user = getUser();
                String htmlPath = "";

                Report report = reportDesignTimeService.doView(id);
                JSONObject requestBody = new JSONObject();
                requestBody.put("appId",report.getApplicationid());
                requestBody.put("reportId",report.getId());
                requestBody.put("domainId",domainId);
                //通过feign调用runtime接口预览
                try{
                    htmlPath = runtimeFeignService.showJrxmlReport(id, requestBody.toString(), null);
                    if(!StringUtil.isBlank(htmlPath)&&htmlPath.contains("系统异常")){
                        return error(500,htmlPath,null);
                    }
                    return  success("ok",htmlPath);
                }catch (Exception e){
                    return error(500,"系统错误，请联系管理员！",null);
                }
            } else {
                return error(500,"请指定报表!",null);
            }

        } catch (Exception e) {
            e.printStackTrace();
            return error(500,e.getMessage(),null);
        }

    }

    /**
     * 返回企业域列表
     * @return
     */
    @GetMapping(path = "/reports/getDomainList")
    @ApiOperation(value = "返回企业域列表", notes = "返回企业域列表")
    public Resource getDomainList() {
        try{
            String accessToken = "";
            Cookie[] cookies = request.getCookies();
            for (Cookie cookie : cookies) {
                String name = cookie.getName();
                if (Security.DESIGNER_TOKEN.equals(name)) {
                    accessToken = cookie.getValue();
                }
            }
            FeignConfig.ADMIN_TOKEN = accessToken;
            JSONObject requestDomainList = new JSONObject();
            JSONObject result = new JSONObject();
            try{
                result = (JSONObject) JSONObject.fromObject(runtimeFeignService.getDomainList(requestDomainList.toString())).get("data");
            }catch (RuntimeException re){
                re.printStackTrace();
                return error(500,"系统错误，请联系管理员！",null);
            }
            JSONArray resultDomainList = (JSONArray)result.get("datas");
            return success("ok",resultDomainList);
        }catch (Exception e){
            e.printStackTrace();
            return error(500,e.getMessage(),null);
        }

    }

    /**
     * 校验是否有重复的名称
     * @throws Exception
     */
    public void validate(Report report) throws Exception {

        List<Report> list = reportDesignTimeService.list(report.getParentId(),report.getName());
        if(StringUtils.isBlank(report.getId())) {
            //新建
            for (Iterator<Report> iterator = list.iterator(); iterator.hasNext();) {
                Report r = iterator.next();
                if(r.getName().equals(report.getName())){
                    throw new OBPMValidateException("{*[report.name.exist]*}");
                }
            }
        }else {
            //更新
            for (Iterator<Report> iterator = list.iterator(); iterator.hasNext();) {
                Report r = iterator.next();
                if(r.getId().equals(report.getId())) {
                    continue;
                }
                if(r.getName().equals(report.getName())){
                    throw new OBPMValidateException("{*[report.name.exist]*}");
                }
            }
        }
        for (Iterator<Report> iterator = list.iterator(); iterator.hasNext();) {
            Report r = iterator.next();
            if(r.getName().equals(report.getName()) && !r.getId().equals(report.getId())){
                throw new OBPMValidateException( "{*[report.name.exist]*}");
            }
        }
    }
}
