package cn.myapps.designtime.menu.controller;

import cn.myapps.common.Environment;
import cn.myapps.common.FileSystemDesignTimeSerializable;
import cn.myapps.common.ModelSuffix;
import cn.myapps.common.data.ParamsTable;
import cn.myapps.common.exception.OBPMValidateException;
import cn.myapps.common.model.application.Application;
import cn.myapps.common.model.report.Report;
import cn.myapps.common.model.resource.ResourceType;
import cn.myapps.common.model.resource.ResourceVO;
import cn.myapps.common.model.view.AbstractView;
import cn.myapps.common.util.PropertyUtil;
import cn.myapps.common.util.StringUtil;
import cn.myapps.designtime.application.service.ApplicationDesignTimeService;
import cn.myapps.designtime.common.controller.AbstractDesignTimeController;
import cn.myapps.designtime.common.controller.Resource;
import cn.myapps.designtime.common.service.DesignTimeServiceManager;
import cn.myapps.designtime.form.service.FormDesignTimeService;
import cn.myapps.designtime.report.service.ReportDesignTimeService;
import cn.myapps.designtime.resource.service.ResourceDesignTimeService;
import cn.myapps.designtime.view.service.ViewDesignTimeService;
import cn.myapps.runtime.dynaform.form.ejb.Form;
import cn.myapps.util.sequence.Sequence;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiImplicitParam;
import io.swagger.annotations.ApiImplicitParams;
import io.swagger.annotations.ApiOperation;
import net.sf.json.JSONObject;
import org.springframework.http.MediaType;
import org.springframework.stereotype.Component;
import org.springframework.web.bind.annotation.*;

import javax.imageio.ImageIO;
import java.awt.image.BufferedImage;
import java.io.File;
import java.text.DecimalFormat;
import java.util.*;

import static com.jayway.jsonpath.JsonPath.parse;

@Api(tags = "菜单设计模块")
@Component
@RequestMapping(path = "/api/designtime/applications", produces = MediaType.APPLICATION_JSON_UTF8_VALUE)
public class MenuController extends AbstractDesignTimeController {

    private ResourceDesignTimeService resourceDesignTimeService;

    public MenuController() throws Exception {
        resourceDesignTimeService = DesignTimeServiceManager.resourceDesignTimeService();
    }

    /**
     * 获取菜单列表
     *
     * @param applicationId 软件id
     * @param isMobile      是否为手机菜单
     * @return
     * @throws Exception
     */
    @GetMapping(path = "/{applicationId}/menus")
    @ApiOperation(value = "获取菜单列表", notes = "获取菜单列表")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "applicationId", value = "软件id", required = true, paramType = "path", dataType = "string"),
            @ApiImplicitParam(name = "isMobile", value = "是否移动端", required = true, paramType = "query", dataType = "boolean"),
            @ApiImplicitParam(name = "parentId", value = "父级id", required = false, paramType = "query", dataType = "string")
    })
    public Resource doGetMenusList(@PathVariable String applicationId, @RequestParam boolean isMobile, String parentId) throws Exception {
        try {
            List list = new ArrayList<>();
            List<ResourceVO> menusList = new ArrayList<>();
            if(StringUtil.isBlank(parentId)){
                parentId = applicationId;
            }
            if(isMobile){
                menusList = resourceDesignTimeService.getChildsByParentId(parentId, ModelSuffix.MOBILE_MENU_PATH_SUFFIX, ModelSuffix.MOBILE_MENU_FILE_SUFFIX, "");
            } else {
                menusList = resourceDesignTimeService.getChildsByParentId(parentId, "");
            }
            for(ResourceVO resourceVO : menusList){
                Map map = new HashMap();
                map.put("name",resourceVO.getName());
                map.put("id",resourceVO.getId());
                map.put("superior",resourceVO.getSuperior());
                map.put("uri",resourceVO.getUri());
                map.put("applicationId",resourceVO.getApplicationid());
                map.put("permissionType",resourceVO.getPermissionType());
                List<ResourceVO> subMenus ;
                if(isMobile){
                    subMenus = resourceDesignTimeService.getChildsByParentId(resourceVO.getId(), ModelSuffix.MOBILE_MENU_PATH_SUFFIX, ModelSuffix.MOBILE_MENU_FILE_SUFFIX, "");
                } else {
                    subMenus = resourceDesignTimeService.getChildsByParentId(resourceVO.getId(), "");
                }
                //hasChild 有子集的为false，没有子集为true (配合前端渲染数据)
                if(subMenus != null && !subMenus.isEmpty()){
                    map.put("hasChild", false);
                } else {
                    map.put("hasChild", true);
                }
                list.add(map);
            }

            return success("ok", list);
        } catch(Exception e){
            e.printStackTrace();
            return error(500, e.getMessage(), null);
        }
    }

    /**
     * 获取菜单详情
     *
     * @param menuId 菜单id
     * @return
     * @throws Exception
     */
    @GetMapping(path = "/menus/{menuId}")
    @ApiOperation(value = "获取菜单详情", notes = "获取菜单详情")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "menuId", value = "菜单id", required = true, paramType = "path", dataType = "string")
    })
    public Resource doGetMenuDetailed(@PathVariable String menuId) throws Exception {
        try {
            ResourceVO menu = resourceDesignTimeService.findById(menuId);
            //兼容类型为报表菜单的旧数据
            if(!StringUtil.isBlank(menu.getLinkType()) && menu.getLinkType().equals("09") && StringUtil.isBlank(menu.getModuleid())){
                ReportDesignTimeService reportService = DesignTimeServiceManager.reportDesignTimeService();
                String reportId = menu.getActionContent().substring(menu.getActionContent().indexOf("=")+1);
                Report report = reportService.findById(reportId);
                menu.setActionContent(reportId);
                menu.setModuleid(report.getParentId());
            }
            return success("ok", menu);
        } catch (Exception e) {
            e.printStackTrace();
            return error(500, e.getMessage(), null);
        }
    }

    /**
     * 复制菜单
     *
     * @param ids  请求包体
     * @return
     * @throws Exception
     */
    @PostMapping(path = "/{applicationId}/menus/copy")
    @ApiOperation(value = "复制菜单", notes = "复制菜单")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "applicationId", value = "软件id", required = true, paramType = "path", dataType = "string"),
            @ApiImplicitParam(name = "destid", value = "目标上级菜单", required = true, paramType = "query", dataType = "string"),
            @ApiImplicitParam(name = "isMobile", value = "是否移动端", required = true, paramType = "query", dataType = "string"),
            @ApiImplicitParam(name = "ids", value = "请求包体", required = true, paramType = "body", dataType = "string")})
    public Resource doCopyMenus(@PathVariable String applicationId, @RequestBody String[] ids,@RequestParam String destid,@RequestParam String isMobile){
        try {
            ApplicationDesignTimeService appService = DesignTimeServiceManager.applicationDesignTimeService();
            ResourceDesignTimeService resourceService = DesignTimeServiceManager.resourceDesignTimeService();
            ResourceVO parent = resourceService.doView(destid);
            String parentId = "";
            if(parent != null){
                parentId = parent.getId();
            }
            if(isMobile.equals("true")){
                for(int i = 0; i < ids.length; i++){
                    ResourceVO copyMenu = resourceService.doView(ids[i]);
                    ResourceVO menu = ResourceVO.clone(copyMenu);
                    if(parent != null){
                        menu.setParent(parent);
                        menu.setParentId(parentId);
                    } else {
                        Application application = appService.findById(applicationId);
                        menu.setParent(application);
                        menu.setParentId(applicationId);
                    }
                    menu.setId("");
                    menu.setSuperior(parentId);
                    menu.setUri("");
                    menu.setIsMobile("false");
                    resourceService.save(menu);
                }
            } else {
                for(int i = 0; i < ids.length; i++){
                    ResourceVO copyMenu = resourceService.doView(ids[i]);
                    ResourceVO menu = ResourceVO.clone(copyMenu);
                    String linkType = menu.getLinkType();
                    if(!StringUtil.isBlank(linkType) && !"00".equals(linkType) && !"01".equals(linkType)
                            && !"02".equals(linkType) && !"09".equals(linkType) && !"12".equals(linkType)){
                        menu.setLinkType("");
                        menu.setActionContent("");
                    }
                    if(parent != null){
                        menu.setParent(parent);
                        menu.setParentId(parentId);
                    } else {
                        Application application = appService.findById(applicationId);
                        menu.setParent(application);
                        menu.setParentId(applicationId);
                        menu.setActionContent("");
                    }
                    menu.setId("");
                    menu.setSuperior(parentId);
                    menu.setUri("");
                    menu.setIsMobile("true");
                    resourceService.save(menu);
                }
            }
            return success("ok", "成功复制菜单");
        } catch (Exception e) {
            e.printStackTrace();
            return error(500, e.getMessage(), null);
        }
    }

    /**
     * 新建菜单
     *
     * @param applicationId 软件id
     * @param content  请求包体
     * @return
     * @throws Exception
     */
    @PostMapping(path = "/{applicationId}/menus")
    @ApiOperation(value = "新建菜单", notes = "新建菜单")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "applicationId", value = "软件id", required = true, paramType = "path", dataType = "string"),
            @ApiImplicitParam(name = "isMobile", value = "是否移动端", required = true, paramType = "path", dataType = "string"),
            @ApiImplicitParam(name = "content", value = "请求包体", required = true, paramType = "body", dataType = "string")})
    public Resource doCreateMenu(@PathVariable String applicationId, @RequestParam String isMobile, @RequestBody String content) throws Exception {
        try {

            JSONObject json = JSONObject.fromObject(content);
            ResourceVO menu = (ResourceVO)json2obj(json,ResourceVO.class);
            menu.setApplicationid(applicationId);
            String superiorId = menu.getSuperior();
            if(superiorId != null && !"".equals(superiorId)) {
                menu.setParentId(superiorId);
            } else {
                menu.setParentId(applicationId);
            }

            String menuId = Sequence.getDesignTimeSequence();
            menu.setId(menuId);
            menu.setIsMobile(isMobile);
            menu.setDescription(menu.getName());
            //校验
            String validateMessage = validate(menu);
            if(!StringUtil.isBlank(validateMessage)){
                return error(40001, validateMessage, null);
            }
            resourceDesignTimeService.saveOrUpdate(menu);

            JSONObject menuJson = new JSONObject();
            menuJson.put("id", menuId);

            return success("ok", menuJson);
        } catch (Exception e) {
            e.printStackTrace();
            return error(500, e.getMessage(), null);
        }
    }

    /**
     * 更新菜单
     *
     * @param applicationId 软件id
     * @param superiorId 上级菜单id
     * @param menuId  菜单id
     * @param content 请求包体
     * @return
     * @throws Exception
     */
    @PutMapping(path = "/{applicationId}/menus/{menuId}")
    @ApiOperation(value = "更新表单", notes = "更新表单")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "applicationId", value = "软件id", required = true, paramType = "path", dataType = "string"),
            @ApiImplicitParam(name = "superiorId", value = "上级菜单id", required = false, paramType = "path", dataType = "string"),
            @ApiImplicitParam(name = "menuId", value = "菜单id", required = true, paramType = "path", dataType = "string"),
            @ApiImplicitParam(name = "content", value = "请求包体", required = true, paramType = "body", dataType = "string")})
    public Resource doUpdateMenu(@PathVariable String applicationId,  String superiorId, @PathVariable String menuId, @RequestBody String content) throws Exception {

        try {
            JSONObject json = JSONObject.fromObject(content);
            ResourceVO menu = (ResourceVO)json2obj(json,ResourceVO.class);
            menu.setApplicationid(applicationId);

            if(menu.getSuperior() != null && !"".equals(menu.getSuperior())) {
                menu.setParentId(menu.getSuperior());
            } else {
                menu.setParentId(applicationId);
            }
            menu.setDescription(menu.getName());
            menu.setId(menuId);
            //校验
            String validateMessage = validate(menu);
            if(!StringUtil.isBlank(validateMessage)){
                return error(40001, validateMessage, null);
            }
            resourceDesignTimeService.saveOrUpdate(menu);

            return success("ok", null);
        } catch (Exception e) {
            e.printStackTrace();
            return error(500, e.getMessage(), null);
        }
    }

    /**
     * 删除菜单（可批量）
     *
     * @param content 请求包体
     * @return
     * @throws Exception
     */
    @DeleteMapping(path = "/menus")
    @ApiOperation(value = "删除菜单", notes = "删除菜单（可批量）")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "content", value = "请求包体", required = true, paramType = "body", dataType = "string")})
    public Resource doDeleteMenu(@RequestBody String content) throws Exception {
        try {
            List<String> ids = parse(content).json();
            String[] idArray = {};
            idArray = ids.toArray(idArray);

            resourceDesignTimeService.delete(idArray);

            return success("ok", null);
        } catch (Exception e) {
            e.printStackTrace();
            return error(500, e.getMessage(), null);
        }
    }

    /**
     * 根据表单创建菜单
     *
     * @param applicationId 软件id
     * @return
     * @throws Exception
     */
    @PostMapping(path = "/{applicationId}/form/{formId}/menus")
    @ApiOperation(value = "根据表单创建菜单", notes = "根据表单创建菜单")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "applicationId", value = "软件Id", required = true, paramType = "path", dataType = "string"),
            @ApiImplicitParam(name = "formId", value = "表单Id", required = true, paramType = "path", dataType = "string"),
            @ApiImplicitParam(name = "content", value = "请求包体", required = true, paramType = "body", dataType = "string")
    })
    public Resource doCreateMenuByForm(@PathVariable String applicationId, @PathVariable String formId, @RequestBody String content) throws Exception {
        try {
            FormDesignTimeService fp = DesignTimeServiceManager.formDesignTimeService();
            Form f = fp.findById(formId);
            if (f == null){
                return error(500,"{*[页面已失效，请重新打开！]*}",null);
            }

            JSONObject json = JSONObject.fromObject(content);
            String name = (String) json.get("name");
            String superior = (String) json.get("superior");
            String showType = (String) json.get("showType");

            // 设置Menu相关属性
            ResourceVO from_rVO = new ResourceVO();

            FileSystemDesignTimeSerializable parent ;
            if(StringUtil.isBlank(superior)) {
                from_rVO.setParentId(applicationId);
                ApplicationDesignTimeService applicationService = DesignTimeServiceManager.applicationDesignTimeService();
                parent = applicationService.findById(applicationId);
                from_rVO.setParent(parent);
            }else{
                from_rVO.setParentId(superior);
                ResourceDesignTimeService resourceService = DesignTimeServiceManager.resourceDesignTimeService();
                parent = resourceService.doView(superior);
                from_rVO.setParent(parent);
            }

            if ("mobile".equals(showType)) {
                from_rVO.setType(ResourceType.RESOURCE_TYPE_MOBILE);
                from_rVO.setIsMobile("true");
            } else {
                from_rVO.setType(ResourceType.RESOURCE_TYPE_MENU);// 否则为默认菜单类型
                from_rVO.setIsMobile("false");
            }

            from_rVO.setLinkName(name);
            from_rVO.setSuperior(superior);
            from_rVO.setDescription(name);
            from_rVO.setActionContent(formId);
            from_rVO.setLinkName(name);
            from_rVO.setApplicationid(f.getApplicationid());// 设置链接(link)应用id
            from_rVO.setLinkType(ResourceVO.LinkType.FORM.getCode());// 连接(link)表单类型
            from_rVO.setQueryString("[]");
            from_rVO.setOpentarget("detail");
            from_rVO.setModuleid(f.getParentId());
            from_rVO.setDescription(from_rVO.getName());// 菜单名
            from_rVO.setApplicationid(f.getApplicationid());
            from_rVO.setName(name);

            ResourceDesignTimeService rp = DesignTimeServiceManager.resourceDesignTimeService();
            // 创建菜单
            if(rp.validateDoCreateMenu(from_rVO)) {
                rp.saveOrUpdate(from_rVO);
            }else {
                throw new OBPMValidateException("{*[page.name.notexist]*}");
            }
            return success("ok", null);
        } catch (Exception e) {
            e.printStackTrace();
            return error(500,e.getMessage(),null);
        }
    }

    /**
     * 根据视图创建菜单
     *
     * @param viewId 视图id
     * @param applicationId 软件id
     * @return
     * @throws Exception
     */
    @PostMapping(path = "/{applicationId}/view/{viewId}/menus")
    @ApiOperation(value = "根据视图创建菜单", notes = "根据视图创建菜单")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "applicationId", value = "软件Id", required = true, paramType = "path", dataType = "string"),
            @ApiImplicitParam(name = "viewId", value = "视图Id", required = true, paramType = "path", dataType = "string"),
            @ApiImplicitParam(name = "content", value = "请求包体", required = true, paramType = "body", dataType = "string")
    })
    public Resource doCreateMenuByView(@PathVariable String applicationId, @PathVariable String viewId, @RequestBody String content) throws Exception {
        try {
            ViewDesignTimeService vp = DesignTimeServiceManager.viewDesignTimeService();
            AbstractView v =  vp.findById(viewId);
            if (v == null) {
                return error(500,"{*[页面已失效，请重新打开！]*}",null);
            }

            JSONObject json = JSONObject.fromObject(content);
            String name = (String) json.get("name");
            String superior = (String) json.get("superior");
            String showType = (String) json.get("showType");

            // 设置Menu相关属性
            ResourceVO view_rVO = new ResourceVO();
            if ("mobile".equals(showType)) {
                view_rVO.setType(ResourceType.RESOURCE_TYPE_MOBILE);
                view_rVO.setIsMobile("true");
            } else {
                view_rVO.setType(ResourceType.RESOURCE_TYPE_MENU);// 否则为默认菜单类型
                view_rVO.setIsMobile("false");
            }

            FileSystemDesignTimeSerializable parent ;

            if(StringUtil.isBlank(superior)) {
                view_rVO.setParentId(applicationId);
                ApplicationDesignTimeService applicationService = DesignTimeServiceManager.applicationDesignTimeService();
                parent = applicationService.findById(applicationId);
                view_rVO.setParent(parent);
            }else{
                view_rVO.setParentId(superior);
                ResourceDesignTimeService resourceService = DesignTimeServiceManager.resourceDesignTimeService();
                parent = resourceService.doView(superior);
                view_rVO.setParent(parent);
            }

            view_rVO.setSuperior(superior);
            view_rVO.setLinkName(name);
            view_rVO.setDescription(name);// 菜单名
            view_rVO.setActionContent(viewId);
            view_rVO.setApplicationid(v.getApplicationid());// 设置链接(link)应用id
            view_rVO.setLinkName(name);
            view_rVO.setLinkType(ResourceVO.LinkType.VIEW.getCode());// 连接(link)视图类型
            view_rVO.setQueryString("[]");
            view_rVO.setOpentarget("detail");
            view_rVO.setModuleid(v.getParentId());
            view_rVO.setApplicationid(v.getApplicationid());
            view_rVO.setName(name);

            // 创建menu
            ResourceDesignTimeService rp = DesignTimeServiceManager.resourceDesignTimeService();
            if(rp.validateDoCreateMenu(view_rVO)) {
                rp.saveOrUpdate(view_rVO);
            }else {
                throw new OBPMValidateException("{*[page.name.notexist]*}");
            }

            return success("ok", null);
        }  catch (Exception e) {
            e.printStackTrace();
            return error(500,e.getMessage(),null);
        }
    }

    /**
     * 获取所有菜单
     * @return
     */
    @GetMapping(path = "/{applicationId}/menu/getAllMenus")
    @ApiOperation(value = "获取所有菜单", notes = "获取所有菜单")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "applicationId", value = "软件Id", required = true, paramType = "path", dataType = "string"),
            @ApiImplicitParam(name = "showType", value = "类型", required = true, paramType = "query", dataType = "string"),
            @ApiImplicitParam(name = "type", value = "菜单类型", required = false, paramType = "query", dataType = "string"),
            @ApiImplicitParam(name = "currentMenuId", value = "当前菜单id", required = false, paramType = "query", dataType = "string"),
            @ApiImplicitParam(name = "name", value = "菜单名称", required = false, paramType = "query", dataType = "string")
    })
    public Resource getAllMenus(@RequestParam(required = false,defaultValue = "") String currentMenuId,@RequestParam(required = false,defaultValue = "")String type,@PathVariable String applicationId,@RequestParam String showType,String name){
        ResourceUtil resourceUtil = new ResourceUtil();
        try{
            Map<String, String> dm = new HashMap<String, String>();
            if (type != null) {
                ParamsTable params = new ParamsTable();

                ResourceDesignTimeService proxy = DesignTimeServiceManager.resourceDesignTimeService();
                params.setParameter("_orderby", "orderno");
                Collection<ResourceVO> dc = new ArrayList<ResourceVO>();

                if(showType.equals("pc")){
                    List<ResourceVO> resources = proxy.list(applicationId,  ModelSuffix.MENU_PATH_SUFFIX, ModelSuffix.MENU_FILE_SUFFIX, null);
                    for (ResourceVO resourceVO : resources) {
                        if(resourceVO.getUri().endsWith(".menu")){
                            dc.add(resourceVO);
                        }
                    }
                } else if(showType.equals("mobile")){
                    List<ResourceVO> resources = proxy.list(applicationId,  ModelSuffix.MOBILE_MENU_PATH_SUFFIX, ModelSuffix.MOBILE_MENU_FILE_SUFFIX, null);
                    for (ResourceVO resourceVO : resources) {
                        if(applicationId.equals(resourceVO.getParentId())){
                            dc.add(resourceVO);
                        }
                    }
                } else {
                    //pc菜单
                    dc = proxy.list(applicationId,  ModelSuffix.MENU_PATH_SUFFIX, ModelSuffix.MENU_FILE_SUFFIX, null);
                    //移动端菜单
                    List<ResourceVO> resources = proxy.list(applicationId, ModelSuffix.MOBILE_MENU_PATH_SUFFIX, ModelSuffix.MOBILE_MENU_FILE_SUFFIX, null);
                    dc.addAll(resources);
                }

                dm = resourceUtil.deepSearchMenuTree(dc, null,
                        currentMenuId, 0);
            } else {
                dm.put("", "无");
            }
            List list = new ArrayList();
            for(String key : dm.keySet()){
                Map data = new HashMap();
                data.put("id",key);
                data.put("value",dm.get(key));
                list.add(data);
            }
            return success("ok",list);
        }catch (Exception e){
            e.printStackTrace();
            return error(500,e.getMessage(),null);
        }
    }

    /**
     * 获取图标集合
     */
    @GetMapping(value = "/icons")
    @ApiOperation(value = "获取图标集合", notes = "获取图标集合")
    public Resource getIcons() {
        try {
            Collection<IconLibFile> rtn = new ArrayList<IconLibFile>();
            Environment ev = Environment.getInstance();
            ParamsTable params = getParams();
            String dirPath = params.getParameterAsString("path");
            String path = PropertyUtil.getPath() + "/uploads/lib/icon";
            String uploadsPath = "/uploads/lib/icon";
            if (!StringUtil.isBlank(dirPath)) {
                path = PropertyUtil.getPath() + dirPath;
                uploadsPath = dirPath;
            }

//            String path = PropertyUtil.getPath()+"/uploads/lib/icon";
            if (ev != null) {
                File dir = new File(path);
                File[] files = dir.listFiles();
                if (files != null && files.length > 0) {
                    DecimalFormat format = new DecimalFormat("#.##");
                    for (int i = 0; i < files.length; i++) {
                        if (files[i].isDirectory()) {
                            IconLibFile icon = new IconLibFile();
                            icon.setName(files[i].getName());
                            icon.setPath(uploadsPath + "/" + files[i].getName());
                            icon.setFileType(ResourceHelper.IconLibFile.DIR);
                            rtn.add(icon);
                        } else if (isImageFile(files[i].getName())) {
                            IconLibFile icon = new IconLibFile();
                            BufferedImage bufferedImage = ImageIO.read(files[i]);
                            icon.setSize(bufferedImage.getWidth() + " x " + bufferedImage.getHeight());
                            icon.setLength(format.format(files[i].length() / 1024.0) + " KB");
                            icon.setName(files[i].getName());
                            icon.setWidth(bufferedImage.getWidth());
                            icon.setPath(uploadsPath + "/" + files[i].getName());
                            icon.setFileType(ResourceHelper.IconLibFile.IMAGE);
                            rtn.add(icon);
                        }
                    }
                }
            }
            return success("ok", rtn);

        } catch (Exception e) {
            e.printStackTrace();
        }
        return null;
    }

    /**
     * 删除自定义图标
     * @param path 图标路径
     * @return
     * @throws Exception
     */
    @DeleteMapping(path = "/icon")
    @ApiOperation(value = "删除自定义图标", notes = "删除自定义图标")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "path", value = "图标路径", required = true, paramType = "path", dataType = "string") })
    public Resource doDeleteIcon(@RequestBody String[] path) throws Exception {
        for (int i = 0; i < path.length; i++) {
            String realPath = PropertyUtil.getPath() + path[i];
            File icon = new File(realPath);
            if (icon.exists()) {
                icon.delete();
            } else {
                return error(4001, "图标不存在", null);
            }
        }
        return success("ok", "删除成功");

    }

    /**
     * 判断文件名是否为合法的图片文件格式
     *
     * @param name
     * @return
     */
    private boolean isImageFile(String name) {
        name = name.toLowerCase();
        return name.indexOf("png") >= 0 || name.indexOf("ico") >= 0 || name.indexOf(".gif") >= 0
                || name.indexOf(".jpg") >= 0 || name.indexOf(".jpeg") >= 0 || name.indexOf(".bmp") >= 0;
    }

    /**
     * 校验方法
     * @param resourceVO
     * @return
     */
    private String validate(ResourceVO resourceVO) throws Exception {
        if(StringUtil.isBlank(resourceVO.getName())){
            return "菜单名称不能为空!";
        }

        List<ResourceVO> list = resourceDesignTimeService.list(resourceVO.getParentId(),"");
        for(ResourceVO r : list) {
            if(StringUtil.isBlank(resourceVO.getId())){//新建
                //判断同级模块不能重名
                if(r.getName().equals(resourceVO.getName()) && r.getParentId().equals(resourceVO.getParentId()) && r.getFileSuffix().equals(resourceVO.getFileSuffix())) {
                    return "同级菜单名称已存在!";
                }
                if(resourceVO.getParent() instanceof ResourceVO && r.getName().equals(resourceVO.getName()) && resourceVO.getParent().getName().equals(r.getName())){
                    return "名称不可以跟上级相同!";
                }
            } else {
                //判断同级模块不能重名
                if(r.getName().equals(resourceVO.getName()) && r.getParentId().equals(resourceVO.getParentId()) && !resourceVO.getId().equals(r.getId()) && r.getFileSuffix().equals(resourceVO.getFileSuffix())) {
                    return "同级菜单名称已存在!";
                }
                if(resourceVO.getParent() instanceof ResourceVO && r.getName().equals(resourceVO.getName()) && resourceVO.getParent().getName().equals(r.getName()) && !resourceVO.getId().equals(r.getId())){
                    return "名称不可以跟上级相同!";
                }
            }
        }
        return null;
    }

    /**
     * 图标库文件对象
     *
     * @author Happy
     */
    public class IconLibFile {
        public static final int IMAGE = 1;
        public static final int DIR = 2;
        private String name;
        private String size;
        private String length;
        private int width;
        private String path;
        private int fileType;

        public IconLibFile() {
            super();
        }

        public int getWidth() {
            return width;
        }

        public void setWidth(int width) {
            this.width = width;
        }

        public String getName() {
            return name;
        }

        public void setName(String name) {
            this.name = name;
        }

        public String getSize() {
            return size;
        }

        public void setSize(String size) {
            this.size = size;
        }

        public String getLength() {
            return length;
        }

        public void setLength(String length) {
            this.length = length;
        }

        public String getPath() {
            return path;
        }

        public void setPath(String path) {
            this.path = path;
        }

        public int getFileType() {
            return fileType;
        }

        public void setFileType(int fileType) {
            this.fileType = fileType;
        }
    }
}
