package cn.myapps.designtime.widget.controller;

import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import cn.myapps.designtime.common.cache.DesignTimeSerializableCache;
import org.springframework.http.HttpStatus;
import org.springframework.http.MediaType;
import org.springframework.stereotype.Component;
import org.springframework.web.bind.annotation.DeleteMapping;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.PutMapping;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.ResponseStatus;

import cn.myapps.common.data.DataPackage;
import cn.myapps.common.data.ParamsTable;
import cn.myapps.common.exception.OBPMValidateException;
import cn.myapps.common.model.summary.SummaryCfgVO;
import cn.myapps.common.model.widget.PageWidget;
import cn.myapps.designtime.common.controller.AbstractDesignTimeController;
import cn.myapps.designtime.common.controller.Resource;
import cn.myapps.designtime.common.service.DesignTimeServiceManager;
import cn.myapps.designtime.summary.service.SummaryCfgDesignTimeService;
import cn.myapps.designtime.widget.service.PageWidgetDesignTimeService;
import cn.myapps.util.StringUtil;
import cn.myapps.util.sequence.Sequence;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiImplicitParam;
import io.swagger.annotations.ApiImplicitParams;
import io.swagger.annotations.ApiOperation;
import net.sf.json.JSONArray;
import net.sf.json.JSONObject;

@Api(tags = "小工具模块")
@Component
@RequestMapping(path = "/api/designtime/applications", produces = MediaType.APPLICATION_JSON_UTF8_VALUE)
public class PageWidgetController extends AbstractDesignTimeController {

    private PageWidgetDesignTimeService pageWidgetDesignTimeService;

    public PageWidgetController() throws Exception {
        pageWidgetDesignTimeService = DesignTimeServiceManager.pageWidgetDesignTimeService();
    }

    /**
     * 获取小工具列表（可根据名字查询）
     *
     * @param applicationId 软件id
     * @return
     * @throws Exception
     */
    @GetMapping(path = "/{applicationId}/widgets")
    @ResponseStatus(HttpStatus.OK)
    @ApiOperation(value = "获取小工具列表", notes = "获取小工具列表")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "applicationId", value = "软件id", required = true, paramType = "path", dataType = "string"),
            @ApiImplicitParam(name = "pageNo", value = "页码", required = true, paramType = "query", dataType = "string"),
            @ApiImplicitParam(name = "linesPerPage", value = "页条数", required = true, paramType = "query", dataType = "string"),
            @ApiImplicitParam(name = "name", value = "名字", required = true, paramType = "query", dataType = "string")
    })
    public Resource getWidgetsList(@PathVariable String applicationId, String name){
        ParamsTable params = getParams();
        String pageNo =  params.getParameterAsString("pageNo");
        String linesPerPage =  params.getParameterAsString("linesPerPage");
        int page = (pageNo != null && pageNo.length() > 0) ? Integer.parseInt(pageNo) : 1;
        int line = (linesPerPage != null && linesPerPage.length() > 0) ? Integer.parseInt(linesPerPage) : 10;
        try {
            DataPackage<PageWidget> data = pageWidgetDesignTimeService.query(applicationId, name, page, line);
            return success("ok", data);
        } catch (Exception e) {
            e.printStackTrace();
            return error(500, e.getMessage(), null);
        }
    }

    /**
     * 获取小工具详情
     *
     * @param widgetId 小工具id
     * @return
     * @throws Exception
     */
    @GetMapping(path = "/widgets/{widgetId}")
    @ResponseStatus(HttpStatus.OK)
    @ApiOperation(value = "获取小工具详情", notes = "获取小工具详情")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "widgetId", value = "小工具id", required = true, paramType = "path", dataType = "string")
    })
    public Resource doGetWidget(@PathVariable String widgetId) {
        try {
            PageWidget pageWidget = pageWidgetDesignTimeService.findById(widgetId);
            String summaryName = "";
            if(pageWidget != null && PageWidget.TYPE_SUMMARY.equals(pageWidget.getType())){
                SummaryCfgDesignTimeService summaryProcess = DesignTimeServiceManager.summaryCfgDesignTimeService();
                SummaryCfgVO summary = summaryProcess.findById(pageWidget.getActionContent());
                if(summary != null)
                   summaryName = summary.getTitle();
            }
            Map map = new HashMap();
            map.put("data",pageWidget);
            map.put("summaryName",summaryName);
            return success("ok", map);
        } catch (Exception e) {
            e.printStackTrace();
            return error(500, e.getMessage(), null);
        }
    }

    /**
     * 新建小工具
     *
     * @param applicationId 软件id
     * @param content  请求包体
     * @return
     * @throws Exception
     */
    @PostMapping(path = "/{applicationId}/widgets")
    @ResponseStatus(HttpStatus.CREATED)
    @ApiOperation(value = "新建小工具", notes = "新建小工具")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "applicationId", value = "软件id", required = true, paramType = "path", dataType = "string"),
            @ApiImplicitParam(name = "content", value = "请求包体", required = true, paramType = "body", dataType = "string")})
    public Resource doCreateWidgets(@PathVariable String applicationId, @RequestBody String content){
        try {

            JSONObject json = JSONObject.fromObject(content);
            PageWidget pageWidget = (PageWidget)json2obj(json,PageWidget.class);

            pageWidget.setApplicationid(applicationId);
            pageWidget.setParentId(applicationId);

            if(StringUtil.isBlank(pageWidget.getId())){
                pageWidget.setId(Sequence.getDesignTimeSequence());
            }
            if (pageWidget.getName().equals("")) {
                throw new OBPMValidateException(
                        "{*[page.name.notexist]*}");
            }
            //判断名称是否含有特殊字符
            String regEx = "[ \\~\\!\\/\\@\\#\\$\\%\\^\\&\\*\\(\\)\\-\\=\\+\\\\\\|\\[\\{\\}\\]\\;\\:\\\'\\\"\\,\\<\\.\\>\\/\\?]";
            Pattern p = Pattern.compile(regEx);
            Matcher m = p.matcher(pageWidget.getName());

            if (m.find()){
                throw new OBPMValidateException("{*[Name]*}{*[can.not.exist.invalidchar]*}");
            }
            doSaveValidate(pageWidget);
            pageWidgetDesignTimeService.save(pageWidget);

            JSONObject jsonObject = new JSONObject();
            jsonObject.put("id", pageWidget.getId());
            return success("ok", jsonObject);
        } catch (OBPMValidateException ex){
            ex.printStackTrace();
            return error(40001, ex.getValidateMessage(), null);
        } catch (Exception e) {
            e.printStackTrace();
            return error(500, e.getMessage(), null);
        }
    }

    /**
     * 更新小工具
     *
     * @param content 请求包体
     * @return
     * @throws Exception
     */
    @PutMapping(path = "/widgets")
    @ResponseStatus(HttpStatus.OK)
    @ApiOperation(value = "更新小工具", notes = "更新小工具")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "content", value = "请求包体", required = true, paramType = "body", dataType = "string")
    })
    public Resource doUpdateWidgets(@RequestBody String content) {
        try {
            JSONObject json = JSONObject.fromObject(content);
            //更新前先查询
            PageWidget oldpageWidget = pageWidgetDesignTimeService.findById(json.getString("id"));
            PageWidget  pageWidget = (PageWidget)json2obj(json,PageWidget.class);
            pageWidget.setApplicationid(oldpageWidget.getApplicationid());
            doSaveValidate(pageWidget);

            pageWidgetDesignTimeService.saveOrUpdate(pageWidget);
            return success("ok", null);
        } catch (OBPMValidateException ex){
            ex.printStackTrace();
            return error(40001, ex.getValidateMessage(), null);
        } catch (Exception e) {
            e.printStackTrace();
            return error(500, e.getMessage(), null);
        }
    }

    /**
     * 删除小工具（可批量）
     *
     * @param ids 请求包体
     * @return
     * @throws Exception
     */
    @DeleteMapping(path = "/widgets")
    @ResponseStatus(HttpStatus.OK)
    @ApiOperation(value = "删除小工具", notes = "删除小工具（可批量）")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "content", value = "请求包体", required = true, paramType = "body", dataType = "string")})
    public Resource doDeleteWidgets(@RequestBody String[] ids) {
        try {
            pageWidgetDesignTimeService.delete(ids);
            for (String id : ids){
                DesignTimeSerializableCache.remove(id);
            }
            return success("ok", "删除成功");
        } catch (Exception e) {
            e.printStackTrace();
            return error(500, e.getMessage(), null);
        }
    }

    /**
     * 保存前的校验
     *
     * @param widget
     * @throws Exception
     */
    private void doSaveValidate(PageWidget widget) throws Exception {
        List<PageWidget> list = pageWidgetDesignTimeService.list(widget.getApplicationid(), widget.getName());
        for(PageWidget pageWidget:list){
            if(widget.getId().equals(pageWidget.getId()) && widget.getName().equals(pageWidget.getName())){
                break;
            }
            if(pageWidget.getName().equals(widget.getName())){
                throw new OBPMValidateException(" 该名称已存在,请重新命名再保存！ ");
            }
        }
    }
}
