package cn.myapps.designtime.validate.controller;

import cn.myapps.common.data.DataPackage;
import cn.myapps.common.exception.OBPMValidateException;
import cn.myapps.common.model.validaterepository.ValidateRepositoryVO;
import cn.myapps.designtime.common.controller.AbstractDesignTimeController;
import cn.myapps.designtime.common.controller.Resource;
import cn.myapps.designtime.common.service.DesignTimeServiceManager;
import cn.myapps.designtime.validate.repository.service.ValidateRepositoryDesignTimeService;
import cn.myapps.util.StringUtil;
import cn.myapps.util.sequence.Sequence;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiImplicitParam;
import io.swagger.annotations.ApiImplicitParams;
import io.swagger.annotations.ApiOperation;
import net.sf.json.JSONObject;
import org.apache.commons.beanutils.BeanUtils;
import org.springframework.http.HttpStatus;
import org.springframework.http.MediaType;
import org.springframework.stereotype.Component;
import org.springframework.web.bind.annotation.*;

@Api(tags = "检验库模块")
@Component
@RequestMapping(path = "/api/designtime/applications", produces = MediaType.APPLICATION_JSON_UTF8_VALUE)
public class ValidatesController extends AbstractDesignTimeController {

    private ValidateRepositoryDesignTimeService validateRepositoryDesignTimeService;

    public ValidatesController() throws Exception {
        validateRepositoryDesignTimeService = DesignTimeServiceManager.validateRepositoryDesignTimeService();
    }

    /**
     * 获取检验库列表（可根据名字查询）
     *
     * @param applicationId 软件id
     * @return
     * @throws Exception
     */
    @GetMapping(path = "/{applicationId}/validates")
    @ResponseStatus(HttpStatus.OK)
    @ApiOperation(value = "获取检验库列表", notes = "获取检验库列表")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "applicationId", value = "软件id", required = true, paramType = "path", dataType = "string"),
            @ApiImplicitParam(name = "pageNo", value = "页码", required = true, paramType = "path", dataType = "string"),
            @ApiImplicitParam(name = "linesPerPage", value = "页条数", required = true, paramType = "path", dataType = "string"),
            @ApiImplicitParam(name = "name", value = "检验库名称", required = true, paramType = "path", dataType = "string")})
    public Resource getValidatesList(@PathVariable String applicationId, String name,String pageNo,String linesPerPage){
        int page = (pageNo != null && pageNo.length() > 0) ? Integer.parseInt(pageNo) : 1;
        int line = (linesPerPage != null && linesPerPage.length() > 0) ? Integer.parseInt(linesPerPage) : 10;
        try {
            DataPackage<ValidateRepositoryVO> data = validateRepositoryDesignTimeService.query(applicationId, name, page, line);

            return success("ok", data);
        } catch (Exception e) {
            e.printStackTrace();
            return error(500, e.getMessage(), null);
        }
    }

    /**
     * 获取检验库详情
     *
     * @param validId 检验库id
     * @return
     * @throws Exception
     */
    @GetMapping(path = "/{applicationId}/validates/{validId}")
    @ResponseStatus(HttpStatus.OK)
    @ApiOperation(value = "获取样式库详情", notes = "获取样式库详情")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "validId", value = "检验库id", required = true, paramType = "path", dataType = "string")
    })
    public Resource doGetValidate(@PathVariable String applicationId,@PathVariable String validId) throws Exception {
        try {
            ValidateRepositoryVO validate = validateRepositoryDesignTimeService.findById(validId);
            return success("ok", validate);
        } catch (Exception e) {
            e.printStackTrace();
            return error(500, e.getMessage(), null);
        }
    }

    /**
     * 新建检验库
     *
     * @param applicationId 软件id
     * @param content  请求包体
     * @return
     * @throws Exception
     */
    @PostMapping(path = "/{applicationId}/validates")
    @ResponseStatus(HttpStatus.CREATED)
    @ApiOperation(value = "新建检验库", notes = "新建检验库")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "applicationId", value = "软件id", required = true, paramType = "path", dataType = "string"),
            @ApiImplicitParam(name = "content", value = "请求包体", required = true, paramType = "body", dataType = "string")})
    public Resource doCreateValidates(@PathVariable String applicationId, @RequestBody String content) throws Exception {
        try {

            JSONObject json = JSONObject.fromObject(content);
            ValidateRepositoryVO rv = (ValidateRepositoryVO)json2obj(json,ValidateRepositoryVO.class);
            rv.setApplicationid(applicationId);
            rv.setParentId(applicationId);

            if(StringUtil.isBlank(rv.getId())){
                rv.setId(Sequence.getDesignTimeSequence());
            }
            doSaveValidate(rv);
            validateRepositoryDesignTimeService.save(rv);

            JSONObject jsonObject = new JSONObject();
            jsonObject.put("id", rv.getId());

            return success("ok", jsonObject);
        } catch (OBPMValidateException ex){
            ex.printStackTrace();
            return error(40001, ex.getValidateMessage(), null);
        } catch (Exception e) {
            e.printStackTrace();
            return error(500, e.getMessage(), null);
        }
    }

    /**
     * 更新检验库
     *
     * @param applicationId 软件id
     * @param validId 检验库id
     * @param content 请求包体
     * @return
     * @throws Exception
     */
    @PutMapping(path = "/{applicationId}/validates/{validId}")
    @ResponseStatus(HttpStatus.OK)
    @ApiOperation(value = "更新检验库", notes = "更新检验库")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "applicationId", value = "软件id", required = true, paramType = "path", dataType = "string"),
            @ApiImplicitParam(name = "validId", value = "检验库id", required = true, paramType = "path", dataType = "string"),
            @ApiImplicitParam(name = "content", value = "请求包体", required = true, paramType = "body", dataType = "string")
    })
    public Resource doUpdateValidates(@PathVariable String applicationId, @PathVariable String validId, @RequestBody String content) throws Exception {
        try {
            JSONObject json = JSONObject.fromObject(content);
            ValidateRepositoryVO rv = (ValidateRepositoryVO)json2obj(json,ValidateRepositoryVO.class);

            ValidateRepositoryVO oldValidate = validateRepositoryDesignTimeService.findById(validId);
            ValidateRepositoryVO validate = (ValidateRepositoryVO) BeanUtils.cloneBean(oldValidate);

            validate.setApplicationid(applicationId);
            validate.setContent(rv.getContent());
            validate.setName(rv.getName());
            validate.setDescription(rv.getDescription());
            doSaveValidate(validate);
            validateRepositoryDesignTimeService.saveOrUpdate(validate);

            return success("ok", null);
        } catch (OBPMValidateException ex){
            ex.printStackTrace();
            return error(40001, ex.getValidateMessage(), null);
        } catch (Exception e) {
            e.printStackTrace();
            return error(500, e.getMessage(), null);
        }
    }

    /**
     * 删除检验库（可批量）
     *
     * @param ids 请求包体
     * @return
     * @throws Exception
     */
    @DeleteMapping(path = "/{applicationId}/validates")
    @ResponseStatus(HttpStatus.OK)
    @ApiOperation(value = "删除检验库", notes = "删除检验库（可批量）")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "content", value = "请求包体", required = true, paramType = "body", dataType = "string")})
    public Resource doDeleteValidates(@RequestBody String[] ids) throws Exception {
        try {
            validateRepositoryDesignTimeService.delete(ids);
            return success("ok", "删除成功");
        } catch (Exception e) {
            e.printStackTrace();
            return error(500, e.getMessage(), null);
        }
    }

    /**
     * 保存前的校验
     *
     * @param repositoryVO
     * @throws Exception
     */
    private void doSaveValidate(ValidateRepositoryVO repositoryVO) throws Exception {
        boolean exist = validateRepositoryDesignTimeService.isValidateNameExist(repositoryVO.getId(), repositoryVO.getName(), repositoryVO.getApplicationid());
        if (exist) {
            throw new OBPMValidateException(" 库名称重复 ,请重新命名再保存！ ");
        }
    }
}
