package cn.myapps.designtime.summary.controller;

import java.util.Iterator;
import java.util.List;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import cn.myapps.designtime.form.service.FormDesignTimeService;
import cn.myapps.runtime.dynaform.form.ejb.Form;
import org.apache.commons.lang.StringUtils;
import org.springframework.http.HttpStatus;
import org.springframework.http.MediaType;
import org.springframework.stereotype.Component;
import org.springframework.web.bind.annotation.DeleteMapping;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.PutMapping;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.ResponseStatus;

import cn.myapps.common.data.DataPackage;
import cn.myapps.common.exception.OBPMValidateException;
import cn.myapps.common.model.summary.SummaryCfgVO;
import cn.myapps.designtime.common.controller.AbstractDesignTimeController;
import cn.myapps.designtime.common.controller.Resource;
import cn.myapps.designtime.common.service.DesignTimeServiceManager;
import cn.myapps.designtime.summary.service.SummaryCfgDesignTimeService;
import cn.myapps.util.StringUtil;
import cn.myapps.util.sequence.Sequence;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiImplicitParam;
import io.swagger.annotations.ApiImplicitParams;
import io.swagger.annotations.ApiOperation;
import net.sf.json.JSONArray;
import net.sf.json.JSONObject;

@Api(tags = "表单摘要模块")
@Component
@RequestMapping(path = "/api/designtime/applications", produces = MediaType.APPLICATION_JSON_UTF8_VALUE)
public class SummaryController extends AbstractDesignTimeController {

    private SummaryCfgDesignTimeService summaryCfgVODesignTimeService;

    private FormDesignTimeService formService;

    public SummaryController() throws Exception {
        summaryCfgVODesignTimeService = DesignTimeServiceManager.summaryCfgDesignTimeService();
        formService = DesignTimeServiceManager.formDesignTimeService();
    }

    /**
     * 获取表单摘要列表
     *
     * @param applicationId 软件id
     * @return
     * @throws Exception
     */
    @GetMapping(path = "/{applicationId}/summarys")
    @ResponseStatus(HttpStatus.OK)
    @ApiOperation(value = "获取表单摘要列表", notes = "获取表单摘要列表")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "applicationId", value = "软件id", required = true, paramType = "path", dataType = "string"),
            @ApiImplicitParam(name = "scope", value = "摘要类型", required = false, paramType = "query", dataType = "string"),
            @ApiImplicitParam(name = "pageNo", value = "页码", required = true, paramType = "query", dataType = "string"),
            @ApiImplicitParam(name = "linesPerPage", value = "页条数", required = true, paramType = "query", dataType = "string"),
            @ApiImplicitParam(name = "formId", value = "表单id", required = true, paramType = "query", dataType = "string")
    })
    public Resource getSummarysList(@PathVariable String applicationId,String scope, String formId, String pageNo, String linesPerPage){

        int page = (pageNo != null && pageNo.length() > 0) ? Integer.parseInt(pageNo) : 1;
        int line = (linesPerPage != null && linesPerPage.length() > 0) ? Integer.parseInt(linesPerPage) : 10;
        try {
            DataPackage<SummaryCfgVO> data = null;
            if(!StringUtil.isBlank(formId)){
                data = summaryCfgVODesignTimeService.query(formId, null, page, line);
            } else {
                data = new DataPackage<SummaryCfgVO>();
                data.setDatas(summaryCfgVODesignTimeService.doQueryByApplication(applicationId));
                data.setRowCount(data.getDatas().size());
                data.setPageNo(page);
                data.setLinesPerPage(line);
            }
            JSONObject result = new JSONObject();
            JSONArray array = new JSONArray();
            for(SummaryCfgVO summaryCfgVO:data.getDatas()){
                //筛选摘要类型数据
                if(!StringUtil.isBlank(scope) && summaryCfgVO.getScope() != Integer.valueOf(scope)){
                    continue;
                }
                JSONObject jsonObject = new JSONObject();
                jsonObject.put("id", summaryCfgVO.getId());
                jsonObject.put("name", summaryCfgVO.getTitle());
                jsonObject.put("type", summaryCfgVO.getType());
                jsonObject.put("title", summaryCfgVO.getTitle());
                jsonObject.put("uri", summaryCfgVO.getUri());
                jsonObject.put("fieldNames",summaryCfgVO.getFieldNames());
                jsonObject.put("summaryScript",summaryCfgVO.getSummaryScript());
                jsonObject.put("scope",summaryCfgVO.getScope());
                jsonObject.put("formId",summaryCfgVO.getFormId());
                jsonObject.put("isShowTags",summaryCfgVO.getIsShowTags());
                if(summaryCfgVO.getFormId() != null){
                    Form form = formService.doView(summaryCfgVO.getFormId());
                    if(form != null){
                        jsonObject.put("formName",form.getName());
                    }
                }
                array.add(jsonObject);
            }
            result.put("linesPerPage",data.getLinesPerPage());
            result.put("pageCount",data.getPageCount());
            result.put("pageNo",data.getPageNo());
            result.put("rowCount",data.getRowCount());
            result.put("data",array);
            return success("ok", result);
        } catch (Exception e) {
            e.printStackTrace();
            return error(500, e.getMessage(), null);
        }
    }

    /**
     * 获取表单摘要详情
     *
     * @param summaryId 表单摘要id
     * @return
     * @throws Exception
     */
    @GetMapping(path = "/summarys/{summaryId}")
    @ResponseStatus(HttpStatus.OK)
    @ApiOperation(value = "获取表单摘要详情", notes = "获取表单摘要详情")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "summaryId", value = "表单摘要id", required = true, paramType = "path", dataType = "string")
    })
    public Resource doGetSummary(@PathVariable String summaryId) {
        try {
            SummaryCfgVO summaryCfgVO = summaryCfgVODesignTimeService.findById(summaryId);
            return success("ok", summaryCfgVO);
        } catch (Exception e) {
            e.printStackTrace();
            return error(500, e.getMessage(), null);
        }
    }

    /**
     * 新建表单摘要
     *
     * @param applicationId 软件id
     * @param content  请求包体
     * @return
     * @throws Exception
     */
    @PostMapping(path = "/{applicationId}/summarys")
    @ResponseStatus(HttpStatus.CREATED)
    @ApiOperation(value = "新建表单摘要", notes = "新建表单摘要")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "applicationId", value = "软件id", required = true, paramType = "path", dataType = "string"),
            @ApiImplicitParam(name = "content", value = "请求包体", required = true, paramType = "body", dataType = "string")})
    public Resource doCreateSummarys(@PathVariable String applicationId, @RequestBody String content){
        try {

            JSONObject json = JSONObject.fromObject(content);
            SummaryCfgVO summaryCfgVO = (SummaryCfgVO)json2obj(json,SummaryCfgVO.class);

            summaryCfgVO.setApplicationid(applicationId);
            summaryCfgVO.setName(summaryCfgVO.getTitle());

            if(StringUtil.isBlank(summaryCfgVO.getId())){
                summaryCfgVO.setId(Sequence.getDesignTimeSequence());
            }
            if (summaryCfgVO.getName()==null || summaryCfgVO.getName().equals("")) {
                throw new OBPMValidateException(
                        "{*[page.name.notexist]*}");
            }
            //判断名称是否含有特殊字符
            String regEx = "[ \\~\\!\\/\\@\\#\\$\\%\\^\\&\\*\\(\\)\\-\\=\\+\\\\\\|\\[\\{\\}\\]\\;\\:\\\'\\\"\\,\\<\\.\\>\\/\\?]";
            Pattern p = Pattern.compile(regEx);
            Matcher m = p.matcher(summaryCfgVO.getName());

            if (m.find()){
                throw new OBPMValidateException("{*[Name]*}{*[can.not.exist.invalidchar]*}");
            }
            doSaveValidate(summaryCfgVO);
            summaryCfgVODesignTimeService.save(summaryCfgVO);

            JSONObject jsonObject = new JSONObject();
            jsonObject.put("id", summaryCfgVO.getId());
            return success("ok", jsonObject);
        } catch (Exception e) {
            e.printStackTrace();
            return error(500, e.getMessage(), null);
        }
    }

    /**
     * 更新表单摘要
     *
     * @param content 请求包体
     * @return
     * @throws Exception
     */
    @PutMapping(path = "/summarys")
    @ResponseStatus(HttpStatus.OK)
    @ApiOperation(value = "更新表单摘要", notes = "更新表单摘要")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "content", value = "请求包体", required = true, paramType = "body", dataType = "string")
    })
    public Resource doUpdateSummarys(@RequestBody String content) {
        try {
            JSONObject json = JSONObject.fromObject(content);
            SummaryCfgVO summaryCfgVO = (SummaryCfgVO)json2obj(json,SummaryCfgVO.class);
            doSaveValidate(summaryCfgVO);
            summaryCfgVODesignTimeService.saveOrUpdate(summaryCfgVO);
            return success("ok", null);
        } catch (Exception e) {
            e.printStackTrace();
            return error(500, e.getMessage(), null);
        }
    }

    /**
     * 删除表单摘要（可批量）
     *
     * @param ids 请求包体
     * @return
     * @throws Exception
     */
    @DeleteMapping(path = "/summarys")
    @ResponseStatus(HttpStatus.OK)
    @ApiOperation(value = "删除表单摘要", notes = "删除表单摘要（可批量）")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "content", value = "请求包体", required = true, paramType = "body", dataType = "string")})
    public Resource doDeleteSummarys(@RequestBody String[] ids) {
        try {
            summaryCfgVODesignTimeService.delete(ids);
            return success("ok", "删除成功");
        } catch (Exception e) {
            e.printStackTrace();
            return error(500, e.getMessage(), null);
        }
    }
    /**
     * 保存前的校验
     *
     * @param summaryCfgVO
     * @throws Exception
     */
    private void doSaveValidate(SummaryCfgVO summaryCfgVO) throws Exception {
        List<SummaryCfgVO> list = summaryCfgVODesignTimeService.list(summaryCfgVO.getParentId(),summaryCfgVO.getName());
        if(StringUtils.isBlank(summaryCfgVO.getId())) {
            //新建
            for (Iterator<SummaryCfgVO> iterator = list.iterator(); iterator.hasNext();) {
                SummaryCfgVO r = iterator.next();
                if(r.getName().equals(summaryCfgVO.getName())){
                    throw new OBPMValidateException("名称已经存在！");
                }
            }
        }else {
            //更新
            for (Iterator<SummaryCfgVO> iterator = list.iterator(); iterator.hasNext();) {
                SummaryCfgVO r = iterator.next();
                if(r.getId().equals(summaryCfgVO.getId())) {
                    continue;
                }
                if(r.getName().equals(summaryCfgVO.getName())){
                    throw new OBPMValidateException("名称已经存在！");
                }
            }
        }

        List<SummaryCfgVO> summaryCfgVOList = summaryCfgVODesignTimeService.list(summaryCfgVO.getParentId(),null);
        //判断是否以及存在待办摘要
        if(summaryCfgVO.getScope() == 0) {
            for(SummaryCfgVO summary : summaryCfgVOList) {
                if(summary.getScope() == 0 && !summary.getId().equals(summaryCfgVO.getId())) {
                    throw new OBPMValidateException( "待办摘要已经存在！");
                }
            }
        }
    }
}
