package cn.myapps.designtime.role.controller;

import cn.myapps.authtime.common.service.AuthTimeServiceManager;
import cn.myapps.common.data.DataPackage;
import cn.myapps.common.model.role.Role;
import cn.myapps.common.util.StringUtil;
import cn.myapps.designtime.common.controller.AbstractDesignTimeController;
import cn.myapps.designtime.common.controller.Resource;
import cn.myapps.designtime.role.service.RoleDesignTimeService;
import cn.myapps.util.sequence.Sequence;
import io.swagger.annotations.ApiImplicitParam;
import io.swagger.annotations.ApiImplicitParams;
import io.swagger.annotations.ApiOperation;
import net.sf.json.JSONObject;
import org.springframework.context.annotation.Scope;
import org.springframework.stereotype.Component;
import org.springframework.web.bind.annotation.*;

import java.util.Collection;
import java.util.Iterator;
import java.util.List;

import static com.jayway.jsonpath.JsonPath.parse;

/**
 * 角色模块
 */
@Component
@RequestMapping(value = "/api/designtime")
@Scope("prototype")
public class RoleDesignTimeController extends AbstractDesignTimeController {

    private RoleDesignTimeService roleService;

    public RoleDesignTimeController() throws Exception {
        roleService = AuthTimeServiceManager.roleRuntimeService();
    }

    /**
     * 获取角色列表（可根据名字查询）
     * @return
     */
    @GetMapping("/applications/{applicationId}/roles")
    @ApiOperation(value = "获取角色列表（可根据名字查询）", notes = "获取角色列表（可根据名字查询）")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "applicationId", value = "软件id", required = true, paramType = "path", dataType = "string"),
            @ApiImplicitParam(name = "name", value = "查询名称", required = false, paramType = "query", dataType = "string"),
            @ApiImplicitParam(name = "pageNo", value = "第几页", required = false, paramType = "query", dataType = "Integer"),
            @ApiImplicitParam(name = "linesPerPage", value = "每页显示条数", required = false, paramType = "query", dataType = "Integer")
    })
    public Resource getRoles(@PathVariable String applicationId, @RequestParam(required = false) String name, @RequestParam(required = false, defaultValue = "1") Integer pageNo, @RequestParam(required = false, defaultValue = "10") Integer linesPerPage){
        try{
            DataPackage<Role> roles = roleService.query(applicationId, name, pageNo, linesPerPage);

            return success("ok", roles);
        } catch(Exception e){
            e.printStackTrace();
            return error(500, e.getMessage(), null);
        }
    }

    /**
     * 根据名称精确查询查询角色
     * @param applicationId
     * @param name
     * @return
     */
    @GetMapping("/getRoleByName")
    public Resource getRoleByName(String applicationId,String name){
        try{
            Role roleData = roleService.getRoleByName(name,applicationId);
            return success("ok",roleData);
        }catch (Exception e){
            e.printStackTrace();
            return error(500,e.getMessage(),null);

        }
    }

    /**
     * 获取角色详情
     * @return
     */
    @GetMapping("/applications/roles/{roleId}")
    @ApiOperation(value = "获取角色详情", notes = "获取角色详情")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "roleId", value = "角色id", required = true, paramType = "path", dataType = "string")
    })
    public Resource getRole(@PathVariable String roleId){
        try{
            Role role = roleService.findById(roleId);

            return success("ok", role);
        } catch(Exception e){
            e.printStackTrace();
            return error(500, e.getMessage(), null);
        }
    }

    /**
     * 新建角色
     * @return
     */
    @PostMapping("/applications/{applicationId}/roles")
    @ApiOperation(value = "新建角色", notes = "新建角色")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "applicationId", value = "软件Id", required = true, paramType = "path", dataType = "string"),
            @ApiImplicitParam(name = "content", value = "角色数据包体", required = true, paramType = "body", dataType = "string")
    })
    public Resource createRole(@PathVariable String applicationId, @RequestBody String content){
        try{
            JSONObject json = JSONObject.fromObject(content);
            Role role = (Role)json2obj(json, Role.class);
            role.setParentId(applicationId);
            role.setApplicationid(applicationId);
            role.setSortId(Sequence.getTimeSequence());

            String validateMessage = validate(role);
            if(!StringUtil.isBlank(validateMessage)){
                return error(40001, validateMessage, null);
            }

            role.setId(Sequence.getDesignTimeSequence());
            roleService.save(role);

            JSONObject result = new JSONObject();
            result.put("id", role.getId());

            return success("ok", result);
        } catch(Exception e){
            e.printStackTrace();
            return error(500, e.getMessage(), null);
        }
    }

    /**
     * 更新角色
     * @return
     */
    @PutMapping("/applications/roles/{roleId}")
    @ApiOperation(value = "更新角色", notes = "更新角色")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "roleId", value = "角色Id", required = true, paramType = "path", dataType = "string"),
            @ApiImplicitParam(name = "content", value = "角色数据包体", required = true, paramType = "body", dataType = "string")
    })
    public Resource updateRole(@PathVariable String roleId, @RequestBody String content){
        try{
            Role r = roleService.findById(roleId);
            Role role = Role.clone(r);

            JSONObject json = JSONObject.fromObject(content);
            String name = json.getString("name");
            String roleNo = json.getString("roleNo");
            int status = json.getInt("status");
            boolean defaultRole = json.getBoolean("defaultRole");

            role.setName(name);
            role.setRoleNo(roleNo);
            role.setStatus(status);
            role.setDefaultRole(defaultRole);

            String validateMessage = validate(role);
            if(!StringUtil.isBlank(validateMessage)){
                return error(40001, validateMessage, null);
            }

            roleService.update(role);

            return success("ok", null);
        } catch(Exception e){
            e.printStackTrace();
            return error(500, e.getMessage(), null);
        }
    }

    /**
     * 删除角色
     * @return
     */
    @DeleteMapping("/applications/roles")
    @ApiOperation(value = "删除角色", notes = "删除角色")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "content", value = "角色Id集合", required = true, paramType = "body", dataType = "string")
    })
    public Resource deleteModule(@RequestBody String content){
        try{
            List<String> ids = parse(content).json();
            String[] idArray = {};
            idArray = ids.toArray(idArray);
            roleService.delete(idArray);

            return success("ok", null);
        } catch(Exception e){
            e.printStackTrace();
            return error(500, e.getMessage(), null);
        }
    }

    /**
     * 校验方法
     * @param role
     * @return
     * @throws Exception
     */
    private String validate(Role role) throws Exception{
        if(StringUtil.isBlank(role.getName())){
            return "角色名称不能为空!";
        }
        if(StringUtil.isBlank(role.getRoleNo())){
            return "角色编号不能为空!";
        }

        Collection<Role> roleList =  roleService.getRolesByApplication(role.getApplicationid());
        if (!roleList.isEmpty()) {

            for( Iterator<Role> iterator = roleList.iterator();iterator.hasNext();){
                Role tempRole = iterator.next();
                if (StringUtil.isBlank(role.getId())) {// 判断新建不能重名
                    if(role.getName().equals(tempRole.getName())){
                        return "该角色名称已存在";
                    }
                    if (tempRole.getRoleNo().equals(role.getRoleNo())) {
                        return "该角色编号已存在!";
                    }
                } else {
                    if(!tempRole.getId().equals(role.getId())  && role.getName().equals(tempRole.getName())){
                        return "该角色名称已存在";
                    }
                    if (!tempRole.getId().equals(role.getId()) && tempRole.getRoleNo().equals(role.getRoleNo())) {
                        return "该角色编号已存在!";
                    }
                }
            }
        }
        return null;
    }
}
