package cn.myapps.designtime.page.controller;

import cn.myapps.common.data.DataPackage;
import cn.myapps.common.exception.OBPMValidateException;
import cn.myapps.common.model.page.Page;
import cn.myapps.designtime.common.controller.AbstractDesignTimeController;
import cn.myapps.designtime.common.controller.Resource;
import cn.myapps.designtime.common.service.DesignTimeServiceManager;
import cn.myapps.designtime.page.service.PageDesignTimeService;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiImplicitParam;
import io.swagger.annotations.ApiImplicitParams;
import io.swagger.annotations.ApiOperation;
import net.sf.json.JSONObject;
import org.apache.commons.beanutils.BeanUtils;
import org.apache.commons.lang.StringUtils;
import org.springframework.http.HttpStatus;
import org.springframework.http.MediaType;
import org.springframework.stereotype.Component;
import org.springframework.web.bind.annotation.*;

import java.util.Iterator;
import java.util.List;

import static com.jayway.jsonpath.JsonPath.parse;

@Api(tags = "页面操作模块")
@Component
@RequestMapping(path = "/api/designtime/applications", produces = MediaType.APPLICATION_JSON_UTF8_VALUE)
public class PageDesignTimeController extends AbstractDesignTimeController {

    private PageDesignTimeService pageService;

    public PageDesignTimeController() {
        pageService = DesignTimeServiceManager.pageDesignTimeService();
    }

    /**
     * 获取页面列表
     * @param applicationId
     *      软件id
     * @param name
     *      页面名称
     * @param pageNo
     *      页数
     * @param linesPerPage
     *      每页显示条数
     * @return
     */
    @GetMapping(path = "/{applicationId}/pages")
    @ResponseStatus(HttpStatus.OK)
    @ApiOperation(value = "获取页面列表", notes = "获取页面列表")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "applicationId", value = "软件id", required = true, paramType = "path", dataType = "string"),
            @ApiImplicitParam(name = "name", value = "名称", required = false, paramType = "query", dataType = "string"),
            @ApiImplicitParam(name = "pageNo", value = "页码", required = false, paramType = "query", dataType = "int"),
            @ApiImplicitParam(name = "linesPerPage", value = "页条数", required = false, paramType = "query", dataType = "int")
    })
    public Resource getPages(@PathVariable String applicationId, @RequestParam(required = false) String name, @RequestParam(required = false) String requestType, @RequestParam(required = false) String status, @RequestParam(required = false, defaultValue = "1") int pageNo, @RequestParam(required = false, defaultValue = "10") int linesPerPage) {
        try {
            DataPackage<Page> result = pageService.query(applicationId, name, pageNo, linesPerPage);
            return success("ok", result);
        } catch (Exception e) {
            e.printStackTrace();
            return error(500, e.getMessage(), null);
        }
    }

    /**
     * 获取页面详情
     * @param pageId pageId
     * @return
     * @throws Exception
     */
    @GetMapping(path = "/pages/{pageId}")
    @ResponseStatus(HttpStatus.OK)
    @ApiOperation(value = "获取页面详情", notes = "获取页面详情")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "pageId", value = "主键", required = true, paramType = "path", dataType = "string")
    })
    public Resource getPage(@PathVariable String pageId) throws Exception {
        try {
            Page page = pageService.findById(pageId);
            return success("ok", page);
        } catch (Exception e) {
            e.printStackTrace();
            return error(500, e.getMessage(), null);
        }
    }

    /**
     * 新建页面
     * @param applicationId 软件id
     * @param content  请求包体
     * @return
     * @throws Exception
     */
    @PostMapping(path = "/{applicationId}/pages")
    @ResponseStatus(HttpStatus.CREATED)
    @ApiOperation(value = "新建页面", notes = "新建页面")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "applicationId", value = "软件id", required = true, paramType = "path", dataType = "string"),
            @ApiImplicitParam(name = "content", value = "请求包体", required = true, paramType = "body", dataType = "string")
    })
    public Resource createPage(@PathVariable String applicationId, @RequestBody String content) throws Exception {
        try {

            JSONObject json = JSONObject.fromObject(content);
            Page page = (Page)json2obj(json,Page.class);
            page.setApplicationid(applicationId);
            page.setParentId(applicationId);

            doSaveValidate(page);
            pageService.saveOrUpdate(page);

            JSONObject pageJson = new JSONObject();
            pageJson.put("id", page.getId());
            return success("ok", pageJson);
        } catch (OBPMValidateException ex){
            ex.printStackTrace();
            return error(40001, ex.getValidateMessage(), null);
        } catch (Exception e) {
            e.printStackTrace();
            return error(500, e.getMessage(), null);
        }
    }

    /**
     * 更新页面
     * @param applicationId 软件id
     * @param content  请求包体
     * @return
     * @throws Exception
     */
    @PutMapping(path = "/{applicationId}/pages/{pageId}")
    @ResponseStatus(HttpStatus.CREATED)
    @ApiOperation(value = "更新页面", notes = "更新页面")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "applicationId", value = "软件id", required = true, paramType = "path", dataType = "string"),
            @ApiImplicitParam(name = "content", value = "请求包体", required = true, paramType = "body", dataType = "string")
    })
    public Resource updatePage(@PathVariable String applicationId, @PathVariable String pageId, @RequestBody String content) throws Exception {
        try {

            JSONObject json = JSONObject.fromObject(content);
            Page pg = (Page)json2obj(json,Page.class);

            Page oldPage = pageService.findById(pageId);

            Page page = (Page) BeanUtils.cloneBean(oldPage);
            page.setApplicationid(applicationId);
            page.setName(pg.getName());
            page.setTemplateContext(pg.getTemplateContext());

            doSaveValidate(page);
            pageService.saveOrUpdate(page);
            return success("ok", null);
        } catch (OBPMValidateException ex){
            ex.printStackTrace();
            return error(40001, ex.getValidateMessage(), null);
        } catch (Exception e) {
            e.printStackTrace();
            return error(500, e.getMessage(), null);
        }
    }

    /**
     * 删除页面（可批量）
     * @param ids 请求包体
     * @return
     * @throws Exception
     */
    @DeleteMapping(path = "/pages")
    @ResponseStatus(HttpStatus.OK)
    @ApiOperation(value = "删除页面（可批量）", notes = "删除页面（可批量）")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "content", value = "请求包体", required = true, paramType = "body", dataType = "string")
    })
    public Resource doDeleteEips(@RequestBody String content) throws Exception {
        try {
            List<String> ids = parse(content).json();
            String[] idArray = {};
            idArray = ids.toArray(idArray);
            pageService.delete(idArray);
            return success("ok", "删除成功");
        } catch (Exception e) {
            e.printStackTrace();
            return error(500, e.getMessage(), null);
        }
    }

    /**
     * 保存前的校验
     * @throws Exception
     */
    private void doSaveValidate(Page pg) throws Exception {
        List<Page> list = pageService.list(pg.getParentId(),pg.getName());
        if(StringUtils.isBlank(pg.getId())) {
            //新建
            for (Iterator<Page> iterator = list.iterator(); iterator.hasNext();) {
                Page page = iterator.next();
                if(page.getName().equals(pg.getName())){
                    throw new OBPMValidateException("名称已经存在！");
                }
            }
        }else {
            //更新
            for (Iterator<Page> iterator = list.iterator(); iterator.hasNext();) {
                Page page = iterator.next();
                if(page.getId().equals(pg.getId())) {
                    continue;
                }
                if(page.getName().equals(pg.getName())){
                    throw new OBPMValidateException("名称已经存在！");
                }
            }
        }
    }
}
