package cn.myapps.designtime.module.controller;

import cn.myapps.common.model.module.Module;
import cn.myapps.designtime.common.controller.AbstractDesignTimeController;
import cn.myapps.designtime.common.controller.Resource;
import cn.myapps.designtime.common.service.DesignTimeServiceManager;
import cn.myapps.designtime.module.service.ModuleDesignTimeService;
import cn.myapps.designtime.module.service.ModuleDwrService;
import cn.myapps.util.StringUtil;
import cn.myapps.util.sequence.Sequence;
import io.swagger.annotations.ApiImplicitParam;
import io.swagger.annotations.ApiImplicitParams;
import io.swagger.annotations.ApiOperation;
import net.sf.json.JSONObject;
import org.springframework.context.annotation.Scope;
import org.springframework.stereotype.Component;
import org.springframework.web.bind.annotation.*;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

/**
 * 模块资源
 */
@Component
@RequestMapping(value = "/api/designtime")
@Scope("prototype")
public class ModuleDesignTimeController extends AbstractDesignTimeController {

    private ModuleDesignTimeService moduleService;

    public ModuleDesignTimeController() throws Exception{
        moduleService = DesignTimeServiceManager.moduleDesignTimeService();
    }

    /**
     * 根据上级获取模块列表
     * @return
     */
    @GetMapping("/applications/{applicationId}/modules")
    @ApiOperation(value = "根据上级获取模块列表", notes = "根据上级获取模块列表")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "parentId", value = "上级Id", required = true, paramType = "query", dataType = "string"),
            @ApiImplicitParam(name = "applicationId", value = "软件Id", required = true, paramType = "path", dataType = "string")
    })
    public Resource getModulesByParentId(@PathVariable String applicationId,String parentId){
        try{
            if(StringUtil.isBlank(parentId)){
                parentId = applicationId;
            }
            List list = new ArrayList<>();
            List<Module> modules = moduleService.getChildsByParentId(parentId, "");
            for(Module module:modules){
                Map map = new HashMap();
                map.put("name",module.getName());
                map.put("id",module.getId());
                map.put("superior",module.getSuperior());
                map.put("uri",module.getUri());
                map.put("applicationId",module.getApplicationid());
                List<Module> subModules = moduleService.getChildsByParentId(module.getId(), null);
                //hasChild 有子集的为false，没有子集为true (配合前端渲染数据)
                if(subModules != null && !subModules.isEmpty()){
                    map.put("hasChild", false);
                } else {
                    map.put("hasChild", true);
                }
                list.add(map);
            }

            return success("ok", list);
        } catch(Exception e){
            e.printStackTrace();
            return error(500, e.getMessage(), null);
        }
    }

    /**
     * 获取模块详情
     * @return
     */
    @GetMapping("/applications/modules/{moduleId}")
    @ApiOperation(value = "获取模块详情", notes = "获取模块详情")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "moduleId", value = "模块Id", required = true, paramType = "path", dataType = "string")
    })
    public Resource getModule(@PathVariable String moduleId){
        try{
            Module module = moduleService.findById(moduleId);
            Map map = new HashMap();
            map.put("id",module.getId());
            map.put("name",module.getName());
            map.put("orderNo",module.getOrderNo());
            map.put("description",module.getDescription());
            map.put("uri",module.getUri());
            map.put("superior",module.getSuperior());
            if(!StringUtil.isBlank(module.getSuperior())){
                Module parentModule = moduleService.findById(module.getSuperior());
                map.put("superiorName",parentModule.getName());
            }else{
                map.put("superiorName","");
            }
            return success("ok", map);
        } catch(Exception e){
            e.printStackTrace();
            return error(500, e.getMessage(), null);
        }
    }

    /**
     * 新建模块
     * @return
     */
    @PostMapping("/applications/{applicationId}/modules")
    @ApiOperation(value = "新建模块", notes = "新建模块")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "applicationId", value = "软件Id", required = true, paramType = "path", dataType = "string"),
            @ApiImplicitParam(name = "content", value = "模块数据包体", required = true, paramType = "body", dataType = "string")
    })
    public Resource createModule(@PathVariable String applicationId, @RequestBody String content){
        try{
            JSONObject json = JSONObject.fromObject(content);
            Module module = (Module)json2obj(json, Module.class);
            if(StringUtil.isBlank(module.getSuperior())){
                module.setParentId(applicationId);
            } else {
                module.setParentId(module.getSuperior());
            }
            module.setApplicationid(applicationId);

            String validateMessage = validate(module);
            if(!StringUtil.isBlank(validateMessage)){
                return error(40001, validateMessage, null);
            }

            module.setId(Sequence.getDesignTimeSequence());
            moduleService.save(module);

            JSONObject result = new JSONObject();
            result.put("id", module.getId());

            return success("ok", result);
        } catch(Exception e){
            e.printStackTrace();
            return error(500, e.getMessage(), null);
        }
    }

    /**
     * 更新模块
     * @return
     */
    @PutMapping("/applications/{applicationId}/modules/{moduleId}")
    @ApiOperation(value = "更新模块", notes = "更新模块")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "applicationId", value = "软件Id", required = true, paramType = "path", dataType = "string"),
            @ApiImplicitParam(name = "moduleId", value = "模块Id", required = true, paramType = "path", dataType = "string"),
            @ApiImplicitParam(name = "content", value = "模块数据包体", required = true, paramType = "body", dataType = "string")
    })
    public Resource updateModule(@PathVariable String applicationId, @PathVariable String moduleId, @RequestBody String content){
        try{
            JSONObject json = JSONObject.fromObject(content);
            String name = json.getString("name");
            String description = json.getString("description");
            int orderNo = json.getInt("orderNo");
            String superior = json.getString("superior");

            Module oldModule = moduleService.findById(moduleId);
            Module module = Module.clone(oldModule);
            module.setDescription(description);
            module.setOrderNo(orderNo);
            module.setSuperior(superior);
            if(StringUtil.isBlank(module.getSuperior())){
                module.setParentId(applicationId);
            } else {
                module.setParentId(module.getSuperior());
            }
            module.setName(name);

            String validateMessage = validate(module);
            if(!StringUtil.isBlank(validateMessage)){
                return error(40001, validateMessage, null);
            }

            moduleService.update(module);

            return success("ok", null);
        } catch(Exception e){
            e.printStackTrace();
            return error(500, e.getMessage(), null);
        }
    }

    /**
     * 删除模块
     * @return
     */
    @DeleteMapping("/applications/modules")
    @ApiOperation(value = "删除模块", notes = "删除模块")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "ids", value = "请求包体", required = true, paramType = "body", dataType = "string")
    })
    public Resource deleteModule(@RequestBody String[] ids){
        try{
            moduleService.delete(ids);

            return success("ok", null);
        } catch(Exception e){
            e.printStackTrace();
            return error(500, e.getMessage(), null);
        }
    }

    /**
     * 获取树形结构所有模块列表
     */
    @GetMapping("/applications/{applicationId}/allmodules")
    @ApiOperation(value = "获取树形结构所有模块列表", notes = "获取树形结构所有模块列表")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "applicationId", value = "软件Id", required = true, paramType = "path", dataType = "string"),
            @ApiImplicitParam(name = "currentModuleId", value = "当前模块id", required = false, paramType = "query", dataType = "string")
    })
    public Resource getTreeModule(@PathVariable String applicationId,String currentModuleId){
        try{
            ModuleDwrService moduleService = new ModuleDwrService();
            Map<String,String> map = moduleService.getModuleSelected(applicationId,currentModuleId);
            List list = new ArrayList();
            for(String key : map.keySet()){
                Map data = new HashMap();
                data.put("id",key);
                data.put("value",map.get(key));
                list.add(data);
            }
            return success("ok", list);
        } catch(Exception e){
            e.printStackTrace();
            return error(500, e.getMessage(), null);
        }
    }
    /**
     * 校验方法
     * @param module
     * @return
     */
    private String validate(Module module) throws Exception {
        if(StringUtil.isBlank(module.getName())){
            return "模块名称不能为空!";
        }

        List<Module> list = moduleService.list(module.getParentId(),"");
        for(Module m : list) {
            if(StringUtil.isBlank(module.getId())){//新建
                //判断同级模块不能重名
                if(m.getName().equals(module.getName()) && m.getParentId().equals(module.getParentId())) {
                    return "同级模块名称已存在!";
                }else if(module.getParent() instanceof Module && m.getName().equals(module.getName()) && module.getParent().getName().equals(m.getName())){
                    return "名称不可以跟上级相同!";
                }
            } else {
                //判断同级模块不能重名
                if(m.getName().equals(module.getName()) && m.getParentId().equals(module.getParentId()) && !module.getId().equals(m.getId())) {
                    return "同级模块名称已存在!";
                } else if(module.getParent() instanceof Module && m.getName().equals(module.getName()) && module.getParent().getName().equals(m.getName()) && !module.getId().equals(m.getId())){
                    return "名称不可以跟上级相同!";
                } else if(module.getName().equals(m.getName()) && m.getParentId().equals(module.getId())) {
                    return "下级存在相同名称!";
                }
            }
        }
        return null;
    }

}
