package cn.myapps.designtime.dts.excelimport.config.controller;

import java.util.Collection;
import java.util.Iterator;
import java.util.List;

import cn.myapps.common.exception.OBPMValidateException;
import org.apache.commons.lang.StringUtils;
import org.springframework.http.HttpStatus;
import org.springframework.http.MediaType;
import org.springframework.stereotype.Component;
import org.springframework.web.bind.annotation.DeleteMapping;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.PutMapping;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.ResponseStatus;

import cn.myapps.common.data.DataPackage;
import cn.myapps.common.model.excelimport.IMPMappingConfigVO;
import cn.myapps.designtime.common.controller.AbstractDesignTimeController;
import cn.myapps.designtime.common.controller.Resource;
import cn.myapps.designtime.common.service.DesignTimeServiceManager;
import cn.myapps.designtime.dts.excelimport.config.service.IMPMappingConfigDesignTimeService;
import cn.myapps.util.sequence.Sequence;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiImplicitParam;
import io.swagger.annotations.ApiImplicitParams;
import io.swagger.annotations.ApiOperation;
import net.sf.json.JSONArray;
import net.sf.json.JSONObject;

@Api(tags = "excel配置设计模块")
@Component
@RequestMapping(path = "/api/designtime/applications", produces = MediaType.APPLICATION_JSON_UTF8_VALUE)
public class ExcelConfigsController extends AbstractDesignTimeController {

    private IMPMappingConfigDesignTimeService impMappingConfigDesignTimeService;

    public ExcelConfigsController() throws Exception {
        impMappingConfigDesignTimeService = DesignTimeServiceManager.impMappingConfigDesignTimeService();
    }

    /**
     * 获取excel配置列表
     *
     * @param applicationId 软件id
     * @return
     * @throws Exception
     */
    @GetMapping(path = "/{applicationId}/excelconfigs")
    @ResponseStatus(HttpStatus.OK)
    @ApiOperation(value = "获取excel配置列表", notes = "获取excel配置列表（可根据名字查询）")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "applicationId", value = "软件id", required = true, paramType = "path", dataType = "string"),
            @ApiImplicitParam(name = "name", value = "名称", required = false, paramType = "query", dataType = "string"),
            @ApiImplicitParam(name = "pageNo", value = "当前页数", required = false, paramType = "query", dataType = "string"),
            @ApiImplicitParam(name = "linesPerPage", value = "每页行数", required = false, paramType = "query", dataType = "string"),
    })
    public Resource doGetExcelList(@PathVariable String applicationId, @RequestParam(required = false) String name,
                                  String pageNo, String linesPerPage) throws Exception {
        int page = (pageNo != null && pageNo.length() > 0) ? Integer.parseInt(pageNo) : 1;
        int line = (linesPerPage != null && linesPerPage.length() > 0) ? Integer.parseInt(linesPerPage) : 10;
        try {
            DataPackage<IMPMappingConfigVO> impMappingConfigVOs = impMappingConfigDesignTimeService.query(applicationId, name, page, line);

            return success("ok", impMappingConfigVOs);
        } catch (Exception e) {
            e.printStackTrace();
            return error(500, e.getMessage(), null);
        }
    }

    /**
     * 获取excel配置详情
     *
     * @param applicationId 软件id
     * @param excelConfigId excel配置id
     * @return
     * @throws Exception
     */
    @GetMapping(path = "/{applicationId}/excelconfigs/{excelConfigId}")
    @ResponseStatus(HttpStatus.OK)
    @ApiOperation(value = "获取excel配置详情", notes = "获取excel配置详情")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "applicationId", value = "软件id", required = true, paramType = "path", dataType = "string"),
            @ApiImplicitParam(name = "excelConfigId", value = "excel配置id", required = true, paramType = "path", dataType = "string")
    })
    public Resource doGetExcelDetailed(@PathVariable String applicationId, @PathVariable String excelConfigId) throws Exception {
        try {
            IMPMappingConfigVO impMappingConfigVO = impMappingConfigDesignTimeService.findById(excelConfigId);

            return success("ok", impMappingConfigVO);
        } catch (Exception e) {
            e.printStackTrace();
            return error(500, e.getMessage(), null);
        }
    }

    /**
     * 新建excel配置
     *
     * @param applicationId 软件id
     * @param content  请求包体
     * @return
     * @throws Exception
     */
    @PostMapping(path = "/{applicationId}/excelconfigs")
    @ResponseStatus(HttpStatus.CREATED)
    @ApiOperation(value = "新建excel配置", notes = "新建excel配置")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "applicationId", value = "软件id", required = true, paramType = "path", dataType = "string"),
            @ApiImplicitParam(name = "content", value = "请求包体", required = true, paramType = "body", dataType = "string")})
    public Resource doCreateForm(@PathVariable String applicationId, @RequestBody String content) throws Exception {
        try {

            JSONObject json = JSONObject.fromObject(content);
            IMPMappingConfigVO impMappingConfigVO = (IMPMappingConfigVO)json2obj(json,IMPMappingConfigVO.class);
            impMappingConfigVO.setApplicationid(applicationId);
            impMappingConfigVO.setParentId(applicationId);

            String impMappingConfigId = Sequence.getDesignTimeSequence();
            impMappingConfigVO.setId(impMappingConfigId);

            doSaveValidate(impMappingConfigVO);
            impMappingConfigDesignTimeService.save(impMappingConfigVO);

            JSONObject impMappingConfigIdJson = new JSONObject();
            impMappingConfigIdJson.put("id", impMappingConfigId);

            return success("ok", impMappingConfigIdJson);
        } catch (Exception e) {
            e.printStackTrace();
            return error(500, e.getMessage(), null);
        }
    }

    /**
     * 更新excel配置
     *
     * @param applicationId 软件id
     * @param content 请求包体
     * @return
     * @throws Exception
     */
    @PutMapping(path = "/{applicationId}/excelconfigs")
    @ResponseStatus(HttpStatus.OK)
    @ApiOperation(value = "更新excel配置", notes = "更新excel配置")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "applicationId", value = "软件id", required = true, paramType = "path", dataType = "string"),
            @ApiImplicitParam(name = "impId", value = "excel配置id", required = true, paramType = "path", dataType = "string"),
            @ApiImplicitParam(name = "content", value = "请求包体", required = true, paramType = "body", dataType = "string")})
    public Resource doUpdateForm(@PathVariable String applicationId, @RequestBody String content) throws Exception {

        try {
            JSONObject json = JSONObject.fromObject(content);
            IMPMappingConfigVO impMappingConfigVO = (IMPMappingConfigVO)json2obj(json,IMPMappingConfigVO.class);
            impMappingConfigVO.setApplicationid(applicationId);
            impMappingConfigVO.setParentId(applicationId);

            doSaveValidate(impMappingConfigVO);
            impMappingConfigDesignTimeService.doUpdate(impMappingConfigVO);

            return success("ok", null);
        } catch (Exception e) {
            e.printStackTrace();
            return error(500, e.getMessage(), null);
        }
    }

    /**
     * 删除excel配置（可批量）
     *
     * @param ids 请求包体
     * @return
     * @throws Exception
     */
    @DeleteMapping(path = "/excelconfigs")
    @ResponseStatus(HttpStatus.OK)
    @ApiOperation(value = "删除excel配置", notes = "删除excel配置（可批量）")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "content", value = "请求包体", required = true, paramType = "body", dataType = "string")})
    public Resource doDeleteForm(@RequestBody String[] ids) throws Exception {
        try {

            impMappingConfigDesignTimeService.delete(ids);
            return success("ok", null);
        } catch (Exception e) {
            e.printStackTrace();
            return error(500, e.getMessage(), null);
        }
    }

    /**
     * 保存前的校验
     *
     * @param impMappingConfigVO
     * @throws Exception
     */
    private void doSaveValidate(IMPMappingConfigVO impMappingConfigVO) throws Exception {
        List<IMPMappingConfigVO> list = impMappingConfigDesignTimeService.list(impMappingConfigVO.getParentId(),impMappingConfigVO.getName());
        if(StringUtils.isBlank(impMappingConfigVO.getId())) {
            //新建
            for (Iterator<IMPMappingConfigVO> iterator = list.iterator(); iterator.hasNext();) {
                IMPMappingConfigVO r = iterator.next();
                if(r.getName().equals(impMappingConfigVO.getName())){
                    throw new OBPMValidateException("名称已经存在！");
                }
            }
        }else {
            //更新
            for (Iterator<IMPMappingConfigVO> iterator = list.iterator(); iterator.hasNext();) {
                IMPMappingConfigVO r = iterator.next();
                if(r.getId().equals(impMappingConfigVO.getId())) {
                    continue;
                }
                if(r.getName().equals(impMappingConfigVO.getName())){
                    throw new OBPMValidateException("名称已经存在！");
                }
            }
        }
        for (Iterator<IMPMappingConfigVO> iterator = list.iterator(); iterator.hasNext();) {
            IMPMappingConfigVO r = iterator.next();
            if(r.getName().equals(impMappingConfigVO.getName()) && !r.getId().equals(impMappingConfigVO.getId())){
                throw new OBPMValidateException( "名称已经存在！");
            }
        }
    }
}
