package cn.myapps.designtime.chart.controller;

import org.springframework.http.HttpStatus;
import org.springframework.http.MediaType;
import org.springframework.stereotype.Component;
import org.springframework.web.bind.annotation.DeleteMapping;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.PutMapping;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.ResponseStatus;

import cn.myapps.common.data.DataPackage;
import cn.myapps.common.data.ParamsTable;
import cn.myapps.common.model.chart.Chart;
import cn.myapps.designtime.chart.service.ChartDesignTimeService;
import cn.myapps.designtime.common.controller.AbstractDesignTimeController;
import cn.myapps.designtime.common.controller.Resource;
import cn.myapps.designtime.common.service.DesignTimeServiceManager;
import cn.myapps.util.StringUtil;
import cn.myapps.util.sequence.Sequence;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiImplicitParam;
import io.swagger.annotations.ApiImplicitParams;
import io.swagger.annotations.ApiOperation;
import net.sf.json.JSONArray;
import net.sf.json.JSONObject;

@Api(tags = "统计图模块")
@Component
@RequestMapping(path = "/api/designtime/applications", produces = MediaType.APPLICATION_JSON_UTF8_VALUE)
public class ChartController extends AbstractDesignTimeController {

    private ChartDesignTimeService chartDesignTimeService;

    public ChartController() throws Exception {
        chartDesignTimeService = DesignTimeServiceManager.chartDesignTimeService();
    }

    /**
     * 获取统计图列表（可根据名字或者备注查询）
     *
     * @param moduleId 模块id
     * @return
     * @throws Exception
     */
    @GetMapping(path = "/modules/{moduleId}/charts")
    @ResponseStatus(HttpStatus.OK)
    @ApiOperation(value = "获取统计图列表", notes = "获取统计图列表")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "moduleId", value = "模块id", required = true, paramType = "path", dataType = "string"),
            @ApiImplicitParam(name = "currpage", value = "页码", required = true, paramType = "query", dataType = "string"),
            @ApiImplicitParam(name = "pagelines", value = "页条数", required = true, paramType = "query", dataType = "string"),
            @ApiImplicitParam(name = "searchword", value = "名字", required = true, paramType = "query", dataType = "string")
    })
    public Resource getChartsList(@PathVariable String moduleId, String searchword){
        ParamsTable params = getParams();
        String currpage =  params.getParameterAsString("currpage");
        String pagelines =  params.getParameterAsString("pagelines");
        int page = (currpage != null && currpage.length() > 0) ? Integer.parseInt(currpage) : 1;
        int line = (pagelines != null && pagelines.length() > 0) ? Integer.parseInt(pagelines) : 10;
        try {
            DataPackage<Chart> data = chartDesignTimeService.queryByNameOrDescript(moduleId, searchword, page, line);
            JSONObject result = new JSONObject();
            JSONArray array = new JSONArray();
            for(Chart chart:data.getDatas()){
                JSONObject rvJson = new JSONObject();
                rvJson.put("id", chart.getId());
                rvJson.put("name", chart.getName());
                rvJson.put("note", chart.getDescription());
                array.add(rvJson);
            }
            result.put("linesPerPage",data.getLinesPerPage());
            result.put("pageCount",data.getPageCount());
            result.put("pageNo",data.getPageNo());
            result.put("rowCount",data.getRowCount());
            result.put("data",array);
            return success("ok", result);
        } catch (Exception e) {
            e.printStackTrace();
            return error(500, e.getMessage(), null);
        }
    }

    /**
     * 获取统计图详情
     *
     * @param chartId 统计图id
     * @return
     * @throws Exception
     */
    @GetMapping(path = "/charts/{chartId}")
    @ResponseStatus(HttpStatus.OK)
    @ApiOperation(value = "获取统计图详情", notes = "获取统计图详情")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "chartId", value = "统计图id", required = true, paramType = "path", dataType = "string")
    })
    public Resource doGetChart(@PathVariable String chartId) throws Exception {
        try {
            Chart chart = chartDesignTimeService.findById(chartId);
            return success("ok", chart);
        } catch (Exception e) {
            e.printStackTrace();
            return error(500, e.getMessage(), null);
        }
    }

    /**
     * 新建统计图
     *
     * @param applicationId 软件id
     * @param content  请求包体
     * @return
     * @throws Exception
     */
    @PostMapping(path = "/{applicationId}/modules/{moduleId}/charts")
    @ResponseStatus(HttpStatus.CREATED)
    @ApiOperation(value = "新建统计图", notes = "新建统计图")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "applicationId", value = "软件id", required = true, paramType = "path", dataType = "string"),
            @ApiImplicitParam(name = "moduleId", value = "模块id", required = true, paramType = "path", dataType = "string"),
            @ApiImplicitParam(name = "content", value = "请求包体", required = true, paramType = "body", dataType = "string")})
    public Resource doCreateCharts(@PathVariable String applicationId, @PathVariable String moduleId, @RequestBody String content) throws Exception {
        try {

            JSONObject json = JSONObject.fromObject(content);
            Chart chart = (Chart)json2obj(json,Chart.class);
            chart.setApplicationid(applicationId);
            chart.setParentId(moduleId);

            if(StringUtil.isBlank(chart.getId())){
                chart.setId(Sequence.getDesignTimeSequence());
            }

            chartDesignTimeService.save(chart);

            JSONObject jsonObject = new JSONObject();
            jsonObject.put("id", chart.getId());

            return success("ok", jsonObject);
        } catch (Exception e) {
            e.printStackTrace();
            return error(500, e.getMessage(), null);
        }
    }

    /**
     * 更新统计图
     *
     * @param content 请求包体
     * @return
     * @throws Exception
     */
    @PutMapping(path = "/charts")
    @ResponseStatus(HttpStatus.OK)
    @ApiOperation(value = "更新统计图", notes = "更新统计图")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "applicationId", value = "软件id", required = true, paramType = "path", dataType = "string"),
            @ApiImplicitParam(name = "chartId", value = "统计图id", required = true, paramType = "path", dataType = "string"),
            @ApiImplicitParam(name = "content", value = "请求包体", required = true, paramType = "body", dataType = "string")
    })
    public Resource doUpdateCharts(@RequestBody String content) throws Exception {
        try {
            JSONObject json = JSONObject.fromObject(content);
            Chart chart = (Chart)json2obj(json,Chart.class);

            chartDesignTimeService.saveOrUpdate(chart);
            return success("ok", null);
        } catch (Exception e) {
            e.printStackTrace();
            return error(500, e.getMessage(), null);
        }
    }

    /**
     * 删除统计图（可批量）
     *
     * @param ids 请求包体
     * @return
     * @throws Exception
     */
    @DeleteMapping(path = "/charts")
    @ResponseStatus(HttpStatus.OK)
    @ApiOperation(value = "删除统计图", notes = "删除统计图（可批量）")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "content", value = "请求包体", required = true, paramType = "body", dataType = "string")})
    public Resource doDeleteCharts(@RequestBody String[] ids) throws Exception {
        try {
            chartDesignTimeService.delete(ids);
            return success("ok", "删除成功");
        } catch (Exception e) {
            e.printStackTrace();
            return error(500, e.getMessage(), null);
        }
    }
}
