package cn.myapps.designtime.api.group.controller;

import cn.myapps.common.data.DataPackage;
import cn.myapps.common.exception.OBPMValidateException;
import cn.myapps.common.model.api.group.ApiGroup;
import cn.myapps.designtime.api.group.service.ApiGroupDesignTimeService;
import cn.myapps.designtime.common.controller.AbstractDesignTimeController;
import cn.myapps.designtime.common.controller.Resource;
import cn.myapps.designtime.common.service.DesignTimeServiceManager;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiImplicitParam;
import io.swagger.annotations.ApiImplicitParams;
import io.swagger.annotations.ApiOperation;
import net.sf.json.JSONObject;
import org.apache.commons.beanutils.BeanUtils;
import org.apache.commons.lang.StringUtils;
import org.springframework.http.HttpStatus;
import org.springframework.http.MediaType;
import org.springframework.stereotype.Component;
import org.springframework.web.bind.annotation.*;

import java.util.Iterator;
import java.util.List;

import static com.jayway.jsonpath.JsonPath.parse;

@Api(tags = "API分组操作模块")
@Component
@RequestMapping(path = "/api/designtime/applications", produces = MediaType.APPLICATION_JSON_UTF8_VALUE)
public class ApiGroupDesignTimeController extends AbstractDesignTimeController {

    private ApiGroupDesignTimeService apiGroupService;

    public ApiGroupDesignTimeController() {
        apiGroupService = DesignTimeServiceManager.apiGroupDesignTimeService();
    }

    /**
     * 获取api分组列表
     * @param applicationId
     *      软件id
     * @param name
     *      api名称
     * @param pageNo
     *      页数
     * @param linesPerPage
     *      每页显示条数
     * @return
     */
    @GetMapping(path = "/{applicationId}/apigroups")
    @ResponseStatus(HttpStatus.OK)
    @ApiOperation(value = "获取api分组列表", notes = "获取api分组列表")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "applicationId", value = "软件id", required = true, paramType = "path", dataType = "string"),
            @ApiImplicitParam(name = "name", value = "名称", required = false, paramType = "query", dataType = "string"),
            @ApiImplicitParam(name = "pageNo", value = "页码", required = false, paramType = "query", dataType = "int"),
            @ApiImplicitParam(name = "linesPerPage", value = "页条数", required = false, paramType = "query", dataType = "int")
    })
    public Resource getApiGroups(@PathVariable String applicationId, @RequestParam(required = false) String name, @RequestParam(required = false, defaultValue = "1") int pageNo, @RequestParam(required = false, defaultValue = "10") int linesPerPage) {
        try {
            DataPackage<ApiGroup> result = apiGroupService.query(applicationId, name, pageNo, linesPerPage);
            return success("ok", result);
        } catch (Exception e) {
            e.printStackTrace();
            return error(500, e.getMessage(), null);
        }
    }

    /**
     * 获取api分组详情
     * @param apiGroupId api分组id
     * @return
     * @throws Exception
     */
    @GetMapping(path = "/apigroups/{apiGroupId}")
    @ResponseStatus(HttpStatus.OK)
    @ApiOperation(value = "获取api分组详情", notes = "获取api分组详情")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "apiGroupId", value = "主键", required = true, paramType = "path", dataType = "string")
    })
    public Resource getApiGroup(@PathVariable String apiGroupId) {
        try {
            ApiGroup apiGroup = apiGroupService.findById(apiGroupId);
            return success("ok", apiGroup);
        } catch (Exception e) {
            e.printStackTrace();
            return error(500, e.getMessage(), null);
        }
    }

    /**
     * 新建api分组
     * @param applicationId 软件id
     * @param content  请求包体
     * @return
     * @throws Exception
     */
    @PostMapping(path = "/{applicationId}/apigroups")
    @ResponseStatus(HttpStatus.CREATED)
    @ApiOperation(value = "新建api分组", notes = "新建api分组")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "applicationId", value = "软件id", required = true, paramType = "path", dataType = "string"),
            @ApiImplicitParam(name = "content", value = "请求包体", required = true, paramType = "body", dataType = "string")
    })
    public Resource createApiGroup(@PathVariable String applicationId, @RequestBody String content) throws Exception {
        try {

            JSONObject json = JSONObject.fromObject(content);
            ApiGroup apiGroup = (ApiGroup)json2obj(json,ApiGroup.class);
            apiGroup.setApplicationid(applicationId);
            apiGroup.setParentId(applicationId);

            doSaveValidate(apiGroup);
            apiGroupService.saveOrUpdate(apiGroup);

            JSONObject eipJson = new JSONObject();
            eipJson.put("id", apiGroup.getId());
            return success("ok", eipJson);
        } catch (OBPMValidateException ex){
            ex.printStackTrace();
            return error(40001, ex.getValidateMessage(), null);
        } catch (Exception e) {
            e.printStackTrace();
            return error(500, e.getMessage(), null);
        }
    }

    /**
     * 更新api分组
     * @param applicationId 软件id
     * @param content  请求包体
     * @return
     * @throws Exception
     */
    @PutMapping(path = "/{applicationId}/apigroups/{apiGroupId}")
    @ResponseStatus(HttpStatus.CREATED)
    @ApiOperation(value = "更新api分组", notes = "更新api分组")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "applicationId", value = "软件id", required = true, paramType = "path", dataType = "string"),
            @ApiImplicitParam(name = "content", value = "请求包体", required = true, paramType = "body", dataType = "string")
    })
    public Resource updateApiGroup(@PathVariable String applicationId, @PathVariable String apiGroupId, @RequestBody String content) {
        try {

            JSONObject json = JSONObject.fromObject(content);
            ApiGroup ag = (ApiGroup)json2obj(json,ApiGroup.class);

            ApiGroup oldApiGroup = apiGroupService.findById(apiGroupId);

            ApiGroup apiGroup = (ApiGroup) BeanUtils.cloneBean(oldApiGroup);
            apiGroup.setApplicationid(applicationId);
            apiGroup.setName(ag.getName());

            doSaveValidate(apiGroup);
            apiGroupService.saveOrUpdate(apiGroup);
            return success("ok", null);
        } catch (OBPMValidateException ex){
            ex.printStackTrace();
            return error(40001, ex.getValidateMessage(), null);
        } catch (Exception e) {
            e.printStackTrace();
            return error(500, e.getMessage(), null);
        }
    }

    /**
     * 删除apif分组（可批量）
     * @param ids 请求包体
     * @return
     * @throws Exception
     */
    @DeleteMapping(path = "/apigroups")
    @ResponseStatus(HttpStatus.OK)
    @ApiOperation(value = "删除api", notes = "删除api（可批量）")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "content", value = "请求包体", required = true, paramType = "body", dataType = "string")
    })
    public Resource deleteAipGroups(@RequestBody String content) {
        try {
            List<String> ids = parse(content).json();
            String[] idArray = {};
            idArray = ids.toArray(idArray);
            apiGroupService.delete(idArray);
            return success("ok", "删除成功");
        } catch (Exception e) {
            e.printStackTrace();
            return error(500, e.getMessage(), null);
        }
    }

    /**
     * 保存前的校验
     * @throws Exception
     */
    private void doSaveValidate(ApiGroup ac) throws Exception {
        List<ApiGroup> list = apiGroupService.list(ac.getParentId(),ac.getName());
        if(StringUtils.isBlank(ac.getId())) {
            //新建
            for (Iterator<ApiGroup> iterator = list.iterator(); iterator.hasNext();) {
                ApiGroup apiGroup = iterator.next();
                if(apiGroup.getName().equals(ac.getName())){
                    throw new OBPMValidateException("名称已经存在！");
                }
            }
        }else {
            //更新
            for (Iterator<ApiGroup> iterator = list.iterator(); iterator.hasNext();) {
                ApiGroup apiGroup = iterator.next();
                if(apiGroup.getId().equals(ac.getId())) {
                    continue;
                }
                if(apiGroup.getName().equals(ac.getName())){
                    throw new OBPMValidateException("名称已经存在！");
                }
            }
        }
    }
}
