package cn.myapps.common.data;

import java.io.Serializable;
import java.util.*;
import java.util.Map.Entry;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import javax.portlet.PortletRequest;
import javax.servlet.http.HttpServletRequest;

import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONObject;
import com.alibaba.fastjson.JSONPObject;
import com.bcxin.web.commons.ArgumentResolverAdvice;
import com.jayway.jsonpath.Configuration;
import com.jayway.jsonpath.DocumentContext;
import com.jayway.jsonpath.Option;
import org.apache.commons.beanutils.DynaProperty;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import cn.myapps.common.util.SimpleDateUtil;
import cn.myapps.common.util.StringUtil;
import org.springframework.web.servlet.HandlerMapping;

import static com.jayway.jsonpath.JsonPath.parse;

/**
 * 本类是对http协议对象数进行封装, 并可以对参数进行基本的增,删和格式化参数等功能 并通过实现 java.io.Serializable
 * 接口以启用其序列化功能,
 */
public class ParamsTable implements Serializable {
	private static final Logger LOG = LoggerFactory.getLogger(ParamsTable.class);

	/**
	 * 版本的UID.
	 */
	private static final long serialVersionUID = -3970066605460641346L;
	/**
	 * DebugCookie名
	 */
	public static final String DEBUG_COOKIE_NAME = "DEBUG";
	/**
	 * 初始化参数集合(Map)
	 */
	private HashMap<String, Object> params = new HashMap<String, Object>();

	/**
	 * The site http
	 */
	private String siteHttp;

	/**
	 * The context path
	 */
	private String contextPath;

	/**
	 * The session id
	 */
	private String sessionid;

	/**
	 * 业务docId
	 */
	private String docId;

	private transient HttpServletRequest httpRequest;

	private transient PortletRequest portletRequest = null;

	public ParamsTable() {

	}

	public ParamsTable(ParamsTable params) {
		this.setHttpRequest(params.getHttpRequest());
		this.setSessionid(params.getSessionid());
		this.setContextPath(params.getContextPath());
		this.params.putAll(params.getParams());
	}

	public String getDocId()
	{
		return docId;
	}

	public void setDocId(String docId) {
		this.docId = docId;
	}

	public Object getRequest() {
		if (httpRequest != null) {
			return httpRequest;
		} else {
			return portletRequest;
		}
	}

	/**
	 * 添加参数到URL.以参数的形式传递
	 *
	 */
	public void addAllParameter(Map<String, Object> valueMap) {
		params.putAll(valueMap);
	}

	/**
	 * Set参数到URL.以参数的形式传递
	 *
	 * @param name
	 *            URL参数名
	 * @param value
	 *            URL参数值.
	 */
	public void setParameter(String name, Object value) {
		if (value instanceof String) {
			params.put(name, (String) value);
		} else {
			params.put(name, value);
		}

		if (name != null && name.equalsIgnoreCase("docid")) {
			this.setDocId(String.valueOf(value));
		}
	}

	/**
	 * 取出参数
	 *
	 * @param name
	 *            URL参数名
	 * @return URL参数值
	 */
	public Object getParameter(String name) {
		Optional<String> paramKeyOptional =
				params.keySet().stream().filter(ii -> ii.equalsIgnoreCase(name)).findFirst();
		if (!paramKeyOptional.isPresent()) {
			return null;
		}

		return params.get(paramKeyOptional.get());
	}

	/**
	 * 取出URL参数值并格式化成字符串(java.lang.String)
	 *
	 * @param name
	 *            URL参数名
	 * @return URL参数值
	 */
	public String getParameterAsString(String name) {


		Object obj = this.getParameter(name);

		if (obj instanceof String)
			return StringUtil.dencodeHTML((String) obj);
		else if (obj instanceof String[])
			return StringUtil.unite((String[]) obj);
		else if (obj instanceof Object[])
			return StringUtil.unite((Object[]) obj);
		else
			return (obj != null) ? StringUtil.dencodeHTML(obj.toString()) : null;
	}

	/**
	 * 取出URL参数值并格式化成数组
	 *
	 * @param name
	 *            URL参数名
	 * @return URL参数值
	 */
	public String[] getParameterAsArray(String name) {
		Object obj = this.getParameter(name);
		if (obj != null) {
			if (obj instanceof String[]) {
				return (String[]) obj;
			} else {
				return new String[] { (String) obj };
			}
		} else {
			return new String[] {};
		}
	}

	/**
	 * 取出URL参数值并格式化成文本型
	 *
	 * @param name
	 *            URL参数名
	 * @return URL参数值
	 */
	public String getParameterAsText(String name) {
		return getParameterAsText(name, ";");
	}

	/**
	 * 取出URL参数值并格式化成文本型,以参数可进行切割
	 *
	 * @param name
	 *            URL参数名
	 * @param split
	 *            切割字符
	 * @return URL参数值
	 */
	public String getParameterAsText(String name, String split) {
		Object obj = this.getParameter(name);

		if (obj instanceof String)
			return (String) obj;
		else if (obj instanceof String[])
			return StringUtil.unite((String[]) obj, split);
		else
			return (obj != null) ? obj.toString() : null;
	}

	/**
	 * 取出URL参数值并格式化成Double型
	 *
	 * @param name
	 *            URL参数名
	 * @return URL参数值
	 */
	public Double getParameterAsDouble(String name) {
		Object obj = this.getParameter(name);
		if (obj instanceof String) {
			String value = (String) obj;

			try {
				return new Double(value);
			} catch (Exception e) {
				return new Double(0.0);
			}
		}

		return null;
	}

	public Integer getParameterAsInteger(String name) {
		Object obj = this.getParameter(name);
		if (obj instanceof String) {
			String value = (String) obj;

			try {
				return new Integer(value);
			} catch (Exception e) {
				return new Integer(0);
			}
		} else if (obj instanceof Integer) {
			return (Integer) obj;
		}

		return null;
	}

	public boolean getParameterAsBoolean(String name) {
		Object obj = this.getParameter(name);
		if (obj instanceof String) {
			String value = (String) obj;

			if (!StringUtil.isBlank(value) && value.equals("true")) {
				return true;
			}
		} else
			// 增加了对obj是否Boolean类型的判断
			if (obj instanceof Boolean) {
				return Boolean.parseBoolean(obj.toString());
			}

		return false;
	}

	/**
	 * 取出URL参数值并格式化成Long型
	 *
	 * @param name
	 *            URL参数名
	 * @return URL参数值
	 */
	public Long getParameterAsLong(String name) {
		Object obj = this.getParameter(name);

		if (obj instanceof String) {
			String value = (String) obj;
			try {
				return Long.valueOf(value);
			} catch (Exception e) {
				return Long.valueOf(0);
			}
		}

		return null;
	}

	/**
	 * 取出URL参数值并格式化成Date型
	 *
	 * @param name
	 *            URL参数名
	 * @return URL参数值
	 */
	public Date getParameterAsDate(String name) {
		Object obj = this.getParameter(name);
		if (obj instanceof String) {
			try {
				String value = (String) obj;
				if (value != null && SimpleDateUtil.isDate(value))
					return new Date(SimpleDateUtil.parseDate(value).getTime());
			} catch (Exception e) {
				return null;
			}
		}
		return null;
	}

	/**
	 * 返回参数名迭代器(Iterator)
	 *
	 * @return The parameter names.
	 */
	public Iterator<String> getParameterNames() {
		return params.keySet().iterator();
	}

	/**
	 * 可以用包装或适配器方法解决
	 *
	 *            request
	 * @return
	 */
	@SuppressWarnings("unchecked")
	public static ParamsTable convertHTTP(PortletRequest request) {
		ParamsTable params = new ParamsTable();

		params.siteHttp = request.getServerName();
		params.contextPath = request.getContextPath();
		// params.setParameter("realPath",
		// request.getServletContext().getRealPath("/"));
		Map m = request.getParameterMap();
		Iterator<Entry<String, Object>> iter = m.entrySet().iterator();
		while (iter.hasNext()) {
			Entry<String, Object> entry = iter.next();
			String name = entry.getKey();
			Object value = entry.getValue();
			try {
				// If there is only one string in the string array, the put the
				// string only, not array.
				if (value instanceof String[])
					if (((String[]) value).length > 1)
						params.setParameter(name, value);
					else
						params.setParameter(name, ((String[]) value)[0]);
				else
					params.setParameter(name, value);
			} catch (Exception e) {
				LOG.warn("{}", "Set parameter: " + name + " failed, the value is: " + value);
			}
		}

		params.sessionid = request.getPortletSession().getId();
		params.setPortletRequest(request);

		return params;
	}

	/**
	 * 检索来自HTTP协议的参数
	 *
	 * @SuppressWarnings HttpServlet API不支持泛型
	 * @param request
	 *            The http request.
	 * @return The paramters tables.
	 */
	@SuppressWarnings("unchecked")
	public static ParamsTable convertHTTP(HttpServletRequest request) {
		ParamsTable params = new ParamsTable();

		params.siteHttp = request.getServerName();
		params.contextPath = request.getContextPath();
		params.sessionid = request.getSession().getId();
		if(request.getAttribute(ArgumentResolverAdvice.BODY_KEY)!=null &&
				request.getAttribute(ArgumentResolverAdvice.BODY_KEY) instanceof String) {
			try {
				String body = (String) request.getAttribute(ArgumentResolverAdvice.BODY_KEY);
				JSONObject jsonpObject = JSON.parseObject(body);
				if (jsonpObject != null) {
					for (String key : jsonpObject.keySet()) {
						params.setParameter(key, jsonpObject.get(key));
					}
				}
			} catch (Exception ex) {
				LOG.error("convertHTTP-Body解析异常:{};异常:{}", request.getAttribute(ArgumentResolverAdvice.BODY_KEY), ex.toString());
			}
		}

		// 设置真实路径
		params.setParameter("realPath", request.getSession().getServletContext().getRealPath("/"));
		params.setHttpRequest(request);

		Object urlTemplateVariables = request.getAttribute(HandlerMapping.URI_TEMPLATE_VARIABLES_ATTRIBUTE);
		if(urlTemplateVariables instanceof Map) {
			Map<String, String> urlTemplateVariableMaps = (Map) urlTemplateVariables;
			for (String key : urlTemplateVariableMaps.keySet()) {
				if ("id".equalsIgnoreCase(key) && (request.getRequestURI() != null && request.getRequestURI().contains("/documents/"))) {
					String id = urlTemplateVariableMaps.get(key);
					params.setParameter("docid", id);
					params.setDocId(id);
				}

				params.setParameter(key, urlTemplateVariableMaps.get(key));
			}
		}

		// 设置request参数
		Map m = request.getParameterMap();
		Iterator<Entry<String, Object>> iter = m.entrySet().iterator();
		while (iter.hasNext()) {
			Entry<String, Object> entry = iter.next();
			String name = entry.getKey();
			Object value = entry.getValue();
			try {
				// If there is only one string in the string array, the put the
				// string only, not array.
				if (value instanceof String[])
					if (((String[]) value).length > 1)
						params.setParameter(name, value);
					else
						params.setParameter(name, ((String[]) value)[0]);
				else
					params.setParameter(name, value);
			} catch (Exception e) {
				LOG.warn("{}", "Set parameter: " + name + " failed, the value is: " + value);
			}
		}

		return params;
	}

	public HttpServletRequest getHttpRequest() {
		return httpRequest;
	}

	public void setHttpRequest(HttpServletRequest httpRequest) {
		this.httpRequest = httpRequest;
	}

	/**
	 * 取出URL参数值并格式化成Arrary型
	 *
	 * @param name
	 *            URL参数名
	 * @param index
	 *            arrary[index]
	 * @return URL参数值
	 */
	public Object getParameter(String name, int index) {
		Object obj = this.getParameter(name);
		if (obj != null && obj instanceof String[]) {
			String[] col = (String[]) obj;
			return col[index];
		}
		return null;
	}

	/**
	 * SetURL参数对Array
	 *
	 * @param name
	 *            URL参数名
	 * @param index
	 *            arrary[index]
	 * @param value
	 *            URL参数值
	 */
	public void setParameter(String name, int index, Object value) {
		Object obj = this.getParameter(name);
		if (obj != null && obj instanceof String[]) {
			String[] col = (String[]) obj;
			col[index] = (String) value;
		}
	}

	/**
	 * 取出参数的动态属性
	 *
	 * @return 以数组(Array)形式的动态属性.
	 */
	public DynaProperty[] getDynaProperties() {
		DynaProperty[] dynaProps = new DynaProperty[params.size()];
		Iterator<String> iter = params.keySet().iterator();
		int count = 0;

		while (iter.hasNext()) {
			String paramName = iter.next();
			DynaProperty prop = new DynaProperty(paramName, this.getParameter(paramName).getClass());
			dynaProps[count] = prop;
			count++;
		}
		return dynaProps;
	}

	/**
	 * 根据动态属性名取出动态属性
	 *
	 * @param name
	 *            动态属性名
	 * @return 动态属性对象
	 */
	public DynaProperty getDynaProperty(String name) {
		DynaProperty prop = null;

		if (this.getParameter(name) != null)
			prop = new DynaProperty(name, this.getParameter(name).getClass());
		else
			prop = new DynaProperty(name, String.class);

		return prop;
	}

	/**
	 * 移除参数
	 *
	 * @param name
	 *            参数名
	 */
	public void removeParameter(String name) {
		params.remove(name);
	}

	/**
	 * Get the the http site name.
	 *
	 * @return The http site name.
	 */
	public String getSiteHttp() {
		return siteHttp;
	}

	/**
	 * 返回的部分请求的URI，指示请求的范围内。上下文路径总是先在一个请求的URI。路径以一个“/”字符，但并没有结束的"/"字符。在默认（根）
	 * Servlet的情况下，此方法返回""。该容器不解码此字符串。
	 *
	 * @return 一个String指定请求的URI部分，指示请求的上下文
	 */
	public String getContextPath() {
		if (this.contextPath != null) {
			return this.contextPath;
		}
		return "/";

	}

	/**
	 * 设置上下文路径
	 *
	 * @param contextPath
	 *            以http协议形式路径
	 */
	public void setContextPath(String contextPath) {
		this.contextPath = contextPath;
	}

	/**
	 * Put all the map
	 *
	 * @param map
	 *            The map
	 */
	public void putAll(Map<String, ?> map) {
		this.params.putAll(map);
	}

	public Map<String, ?> getParams() {
		return this.params;
	}

	public boolean containsKey(Object key) {
		return this.params.containsKey(key);
	}

	/*
	 * (non-Javadoc)
	 *
	 * @see java.lang.Object#equals(java.lang.Object)
	 */
	public boolean equals(Object obj) {
		if (obj != null && obj instanceof ParamsTable) {
			ParamsTable pt = (ParamsTable) obj;
			return this.params.equals(pt.params);
		}
		return false;
	}

	public int hashCode() {
		return this.params.hashCode();
	}

	/**
	 * 取出应用的session标识
	 *
	 * @return the sessionid
	 */
	public String getSessionid() {
		return sessionid;
	}

	/**
	 * 设置应用的session标识
	 *
	 * @param sessionid
	 *            the sessionid to set
	 */
	public void setSessionid(String sessionid) {
		this.sessionid = sessionid;
	}

	public PortletRequest getPortletRequest() {
		return portletRequest;
	}

	public void setPortletRequest(PortletRequest portletRequest) {
		this.portletRequest = portletRequest;
	}

	public String toString() {
		return params.toString();
	}

	/**
	 * 仅执行查询的时候才有执行值脚本的意义
	 * 否则其他以用户提交的数据为主
	 * @return
	 */
	public boolean isAllowedExecuteFieldValueScript() {
		if (this.getHttpRequest() == null) {
			return true;
		}

		final Collection<String> allowedAction = Stream.of("/refresh", "/importExcel").collect(Collectors.toList());
		if ("GET".equalsIgnoreCase(this.getHttpRequest().getMethod()) ||
				(this.getHttpRequest().getRequestURI() != null &&
						allowedAction.stream().anyMatch(ii -> this.getHttpRequest().getRequestURI().contains(ii)))) {
			return true;
		}

		return false;
	}

	public ParamsTable copyContextParams() {
		ParamsTable newParams = new ParamsTable();
		/**
		 * 子文档也读取父文档的上下文信息
		 */
		newParams.setHttpRequest(this.getHttpRequest());
		newParams.setSessionid(this.getSessionid());
		newParams.setContextPath(this.getContextPath());

		return newParams;
	}

	public Map<String,?> getContextParams() {
		Map<String, Object> pms = new HashMap<>();
		if(this.getParams()!=null) {
			for (String key : this.getParams().keySet()) {
				pms.put(key, this.getParams().get(key));
			}
		}
		pms.put("contextPath", this.getContextPath());
		pms.put("docId", this.getDocId());
		pms.put("sessionId", this.getSessionid());
		pms.put("siteHttp", this.getSiteHttp());
		if (this.getHttpRequest() != null) {
			pms.put("request.requestURI", this.getHttpRequest().getRequestURI());
			pms.put("request.method", this.getHttpRequest().getMethod());
			Enumeration<String> headerNames = this.getHttpRequest().getHeaderNames();
			if (headerNames!=null) {
				while ( headerNames.hasMoreElements()) {
					String h = headerNames.nextElement();
					pms.put(String.format("request.head.%s", h), this.getHttpRequest().getHeader(h));
				}
			}
		}

		return pms;
	}
}
