package com.bcxin.tenant.domain.services.commands;

import com.bcxin.Infrastructures.commands.CommandAbstract;
import com.bcxin.Infrastructures.exceptions.ArgumentTenantException;
import com.bcxin.Infrastructures.exceptions.TenantExceptionAbstract;
import com.bcxin.Infrastructures.utils.DateUtil;
import com.bcxin.tenant.domain.snapshots.EmployeeLeaveExcelItemSnapshot;
import lombok.Getter;
import org.springframework.util.CollectionUtils;
import org.springframework.util.StringUtils;
import java.util.*;

@Getter
public class BatchLeaveEmployeeExcelCommand extends CommandAbstract{
    private final String organizationId;
    private final String path;
    private final Collection<BatchLeaveEmployeeExcelCommand.EmployeeCommandItem> commandItems;
    private final Set<String> treeCodes;

    public BatchLeaveEmployeeExcelCommand(String organizationId, String path, Collection<BatchLeaveEmployeeExcelCommand.EmployeeCommandItem> commandItems, Set<String> treeCodes) {
        this.organizationId = organizationId;
        this.path = path;
        this.commandItems = commandItems;
        this.treeCodes = treeCodes;
    }

    public static BatchLeaveEmployeeExcelCommand create(String organizationId,String path,Collection<BatchLeaveEmployeeExcelCommand.EmployeeCommandItem> commandItems, Set<String> treeCodes) {
        return new BatchLeaveEmployeeExcelCommand(organizationId, path, commandItems, treeCodes);
    }

    public void validate() {
        if (CollectionUtils.isEmpty(commandItems)) {
            throw new ArgumentTenantException("待离职的列表不能为空");
        }

        commandItems.forEach(ix -> {
            try {
                ix.validate();
            } catch (TenantExceptionAbstract ex) {
                ix.addError(ex.getMessage());
            }
        });
    }

    @Getter
    public static class EmployeeCommandItem extends CommandAbstract {
        private final EmployeeLeaveExcelItemSnapshot dataItem;

        public EmployeeCommandItem(EmployeeLeaveExcelItemSnapshot dataItem) {
            this.dataItem = dataItem;
        }

        public static BatchLeaveEmployeeExcelCommand.EmployeeCommandItem create(int paramIndex, String name, String idNum, String leaveDate, String remark) {
            return new BatchLeaveEmployeeExcelCommand.EmployeeCommandItem(new EmployeeLeaveExcelItemSnapshot(paramIndex, name, idNum, leaveDate,
                    remark));
        }

        public static BatchLeaveEmployeeExcelCommand.EmployeeCommandItem create(EmployeeLeaveExcelItemSnapshot snapshot) {
            return new BatchLeaveEmployeeExcelCommand.EmployeeCommandItem(snapshot);
        }

        @Override
        public void validate() {
            super.validate();

            //错误消息
            checkParameter(() -> {
                if (StringUtils.hasLength(this.getDataItem().getErrMsg())) {
                    throw new ArgumentTenantException(this.getDataItem().getErrMsg());
                }
            });
            checkParameter(() -> {
                if (!StringUtils.hasLength(this.getDataItem().getName())) {
                    throw new ArgumentTenantException("姓名不能为空");
                }
            });

            checkParameter(() -> {
                if (!StringUtils.hasLength(this.getDataItem().getIdNum())) {
                    throw new ArgumentTenantException("证件号不能为空");
                }
            });

            checkParameter(() -> {
                if (Objects.isNull(this.getDataItem().getLeaveDateText())) {
                    throw new ArgumentTenantException("离职日期不能为空");
                }
            });

            checkParameter(() -> {
                if (DateUtil.fromString(this.getDataItem().getLeaveDateText()) == null) {
                    throw new ArgumentTenantException("离职日期不符合格式要求");
                } else {
                    this.getDataItem().setLeaveDate(DateUtil.fromString(this.getDataItem().getLeaveDateText()));
                }
            });
            checkParameter(() -> {
                if (!DateUtil.format2ShortDate().equals(DateUtil.format2ShortDate(this.getDataItem().getLeaveDate()))) {
                    throw new ArgumentTenantException("离职日期必须是当天");
                }
            });
        }


        @Override
        public String toString() {
            return String.format("姓名=%s;证件号=%s;离职日期=%s;结果=%s;",
                    dataItem.getName(), dataItem.getIdNum(), dataItem.getLeaveDate(), this.getErrorResult());
        }
    }
}
